package ru.yandex.partner.libs.i18n;

import java.util.Locale;

import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.BeforeAll;
import org.junit.jupiter.api.Test;
import org.springframework.context.i18n.LocaleContextHolder;
import org.springframework.context.support.MessageSourceAccessor;

import ru.yandex.partner.libs.i18n.testmsg.TestMsg;

import static org.junit.jupiter.api.Assertions.assertEquals;

class GettextMessageSourceTest {
    private static MessageSourceAccessor messages;

    @BeforeAll
    static void init() {
        messages = new MessageSourceAccessor(new GettextMessageSource("translations"));
    }

    @Test
    void getTextTest() {

        LocaleContextHolder.setDefaultLocale(new Locale("ru"));

        assertEquals("CPC in Premium placement blocks",
                messages.getMessage(TestMsg.CPC_IN_PREMIUM_BLOCKS, new Locale("en"))
        );

        assertEquals("CPC в блоках Спецразмещения",
                messages.getMessage(TestMsg.CPC_IN_PREMIUM_BLOCKS, new Locale("ru"))
        );

        assertEquals("Süper Pozisyon bloklarında tıklama başına maliyet",
                messages.getMessage(TestMsg.CPC_IN_PREMIUM_BLOCKS, new Locale("tr"))
        );

        //empty or unknown locale must fallback to the default one
        assertEquals("CPC в блоках Спецразмещения",
                messages.getMessage(TestMsg.CPC_IN_PREMIUM_BLOCKS)
        );

        assertEquals("CPC в блоках Спецразмещения",
                messages.getMessage(TestMsg.CPC_IN_PREMIUM_BLOCKS, new Locale("de"))
        );


    }

    @Test
    void nGetTextTest() {

        assertEquals("Всего: -1 текстовых шаблонов",
                messages.getMessage(MsgWithArgs.of(TestMsg.TOTAL_TEXT_TEMPLATES, -1), new Locale("ru"))
        );
        assertEquals("Всего: 0 текстовых шаблонов",
                messages.getMessage(MsgWithArgs.of(TestMsg.TOTAL_TEXT_TEMPLATES, 0), new Locale("ru"))
        );

        assertEquals("Всего: 1 текстовый шаблон",
                messages.getMessage(MsgWithArgs.of(TestMsg.TOTAL_TEXT_TEMPLATES, 1), new Locale("ru"))
        );

        assertEquals("Всего: 2 текстовых шаблона",
                messages.getMessage(MsgWithArgs.of(TestMsg.TOTAL_TEXT_TEMPLATES, 2), new Locale("ru"))
        );

        assertEquals("Всего: 5 текстовых шаблонов",
                messages.getMessage(MsgWithArgs.of(TestMsg.TOTAL_TEXT_TEMPLATES, 5), new Locale("ru"))
        );


        assertEquals("Total: -1 text templates",
                messages.getMessage(MsgWithArgs.of(TestMsg.TOTAL_TEXT_TEMPLATES, -1), new Locale("en"))
        );
        assertEquals("Total: 0 text templates",
                messages.getMessage(MsgWithArgs.of(TestMsg.TOTAL_TEXT_TEMPLATES, 0), new Locale("en"))
        );

        assertEquals("Total: 1 text template",
                messages.getMessage(MsgWithArgs.of(TestMsg.TOTAL_TEXT_TEMPLATES, 1), new Locale("en"))
        );

        assertEquals("Total: 2 text templates",
                messages.getMessage(MsgWithArgs.of(TestMsg.TOTAL_TEXT_TEMPLATES, 2), new Locale("en"))
        );

        assertEquals("Total: 5 text templates",
                messages.getMessage(MsgWithArgs.of(TestMsg.TOTAL_TEXT_TEMPLATES, 5), new Locale("en"))
        );


    }

    @Test
    void pGetTextTest() {
        assertEquals("Find",
                messages.getMessage(TestMsg.ACTION_FIND, new Locale("en"))
        );
        assertEquals("Найти",
                messages.getMessage(TestMsg.ACTION_FIND, new Locale("ru"))
        );
        assertEquals("Bul",
                messages.getMessage(TestMsg.ACTION_FIND, new Locale("tr"))
        );

        assertEquals("Архив",
                messages.getMessage(TestMsg.ARCHIVE, new Locale("ru"))
        );
        assertEquals("Заархивировать",
                messages.getMessage(TestMsg.INDOOR_ACTION_ARCHIVE, new Locale("ru"))
        );
        assertEquals("Удалить",
                messages.getMessage(TestMsg.DOOH_ACTION_ARCHIVE, new Locale("ru"))
        );

    }

    @Test
    void npGetTextTest() {

        assertEquals("У Васи -1 яблок",
                messages.getMessage(MsgWithArgs.of(TestMsg.HAS_APPLES, "Васи", -1), new Locale("ru"))
        );
        assertEquals("У Васи 0 яблок",
                messages.getMessage(MsgWithArgs.of(TestMsg.HAS_APPLES, "Васи", 0), new Locale("ru"))
        );
        assertEquals("У Васи 1 яблоко",
                messages.getMessage(MsgWithArgs.of(TestMsg.HAS_APPLES, "Васи", 1), new Locale("ru"))
        );
        assertEquals("У Васи 2 яблока",
                messages.getMessage(MsgWithArgs.of(TestMsg.HAS_APPLES, "Васи", 2), new Locale("ru"))
        );
        assertEquals("У Васи 5 яблок",
                messages.getMessage(MsgWithArgs.of(TestMsg.HAS_APPLES, "Васи", 5), new Locale("ru"))
        );


        assertEquals("Basil has -1 apples",
                messages.getMessage(MsgWithArgs.of(TestMsg.HAS_APPLES, "Basil", -1), new Locale("en"))
        );
        assertEquals("Basil has 0 apples",
                messages.getMessage(MsgWithArgs.of(TestMsg.HAS_APPLES, "Basil", 0), new Locale("en"))
        );
        assertEquals("Basil has 1 apple",
                messages.getMessage(MsgWithArgs.of(TestMsg.HAS_APPLES, "Basil", 1), new Locale("en"))
        );
        assertEquals("Basil has 2 apples",
                messages.getMessage(MsgWithArgs.of(TestMsg.HAS_APPLES, "Basil", 2), new Locale("en"))
        );
        assertEquals("Basil has 5 apples",
                messages.getMessage(MsgWithArgs.of(TestMsg.HAS_APPLES, "Basil", 5), new Locale("en"))
        );

    }

    @Test
    void unescapeQuotesTest() {
        assertEquals("Status \"some_status\" does not exist",
                messages.getMessage(MsgWithArgs.of(TestMsg.STATUS_DOES_NOT_EXIST, "some_status"), new Locale("en"))
        );
        assertEquals("Статус \"some_status\" не существует",
                messages.getMessage(MsgWithArgs.of(TestMsg.STATUS_DOES_NOT_EXIST, "some_status"), new Locale("ru"))
        );
        assertEquals("Durum \"some_status\" bulunmamakta",
                messages.getMessage(MsgWithArgs.of(TestMsg.STATUS_DOES_NOT_EXIST, "some_status"), new Locale("tr"))
        );
    }

    @Test
    void bigNumberFormatting() {
        Assertions.assertEquals("В большом числе 9999 разряды не разделяются запятой",
                messages.getMessage(MsgWithArgs.of(TestMsg.NUMBER_FORMAT, 9999), new Locale("ru"))
        );

        // assuming same behaviour for stub implementation
        Assertions.assertEquals("Big number 9999 should not be comma-separated",
                messages.getMessage(MsgWithArgs.of(TestMsg.NUMBER_FORMAT, 9999), new Locale("en"))
        );
    }

}
