package ru.yandex.partner.libs.i18n.sync;

import java.io.File;
import java.io.IOException;
import java.util.List;

import com.fasterxml.jackson.databind.ObjectMapper;
import org.json.JSONException;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.BeforeAll;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.skyscreamer.jsonassert.JSONAssert;
import org.springframework.util.FileSystemUtils;

import ru.yandex.partner.libs.i18n.resolver.bundle.DashResourceBundleNameResolver;
import ru.yandex.partner.libs.i18n.resolver.bundle.ResourceBundleNameResolver;
import ru.yandex.partner.libs.i18n.tanker.client.TankerClient;
import ru.yandex.partner.libs.i18n.tanker.client.TankerClientException;
import ru.yandex.partner.libs.i18n.tanker.dto.json.TankerJsonKeysetDoc;
import ru.yandex.partner.libs.i18n.tanker.sync.TranslationDownloader;
import ru.yandex.partner.libs.i18n.testing.MessagesCollector;
import ru.yandex.partner.libs.i18n.testing.MsgNotEnumException;

import static org.mockito.Mockito.doReturn;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.reset;

public class TranslationDownloaderTest {

    private static TmpFileHelper tmpFileHelper;

    private static MessagesCollector messagesCollector;
    private static TankerClient tankerClient;
    private static File resourceDirectory;
    private static ResourceBundleNameResolver resourceBundleNameResolver;

    private static TranslationDownloader translationDownloader;

    @BeforeAll
    public static void init() throws IOException {
        tmpFileHelper = new TmpFileHelper();

        messagesCollector = new MessagesCollector("ru.yandex.partner.libs.i18n.testmsg");
        tankerClient = mock(TankerClient.class);
        resourceDirectory = tmpFileHelper.getTmpDir();
        resourceBundleNameResolver = new DashResourceBundleNameResolver("translations");

        translationDownloader = new TranslationDownloader(
                messagesCollector, tankerClient, resourceDirectory, resourceBundleNameResolver
        );
    }

    @BeforeEach
    @AfterEach
    public void cleanup() {
        FileSystemUtils.deleteRecursively(resourceDirectory);
        reset(tankerClient);
    }

    @Test
    public void downloadSuccessTest() throws TankerClientException, IOException, MsgNotEnumException, JSONException {

        String resource = new String(tmpFileHelper.readResource("translations/test-messages/messages_ru.json"));

        TankerJsonKeysetDoc doc = (new ObjectMapper()).readValue(resource, TankerJsonKeysetDoc.class);
        doReturn(doc).when(tankerClient).downloadKeyset("test_messages", "ru");

        translationDownloader.downloadTranlsations(List.of("ru"));

        String result = new String(tmpFileHelper.readTmpFile("translations/test-messages/messages_ru.json"));

        JSONAssert.assertEquals(resource, result, true);
    }

    @Test
    public void download404Test() throws TankerClientException, IOException, MsgNotEnumException, JSONException {
        doReturn(null).when(tankerClient).downloadKeyset("test_messages", "ru");

        translationDownloader.downloadTranlsations(List.of("ru"));

        String result = new String(tmpFileHelper.readTmpFile("translations/test-messages/messages_ru.json"));

        JSONAssert.assertEquals("{\"ru\": {}}", result, true);
    }

}
