package ru.yandex.partner.libs.i18n.sync;

import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;

import org.json.JSONException;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.BeforeAll;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.skyscreamer.jsonassert.JSONAssert;
import org.springframework.util.FileSystemUtils;

import ru.yandex.partner.libs.i18n.escaper.StringEscaper;
import ru.yandex.partner.libs.i18n.resolver.bundle.DashResourceBundleNameResolver;
import ru.yandex.partner.libs.i18n.resolver.bundle.ResourceBundleNameResolver;
import ru.yandex.partner.libs.i18n.resolver.key.GettextMessageKeyResolver;
import ru.yandex.partner.libs.i18n.tanker.escaper.TankerPOStringEscaper;
import ru.yandex.partner.libs.i18n.tanker.resolver.TankerGettextMessageKeyResolver;
import ru.yandex.partner.libs.i18n.tanker.sync.TranslationPatchGenerator;
import ru.yandex.partner.libs.i18n.testing.MessagesCollector;
import ru.yandex.partner.libs.i18n.testing.MsgNotEnumException;
import ru.yandex.partner.libs.i18n.translation.source.factory.ResourceFileTranslationSourceFactory;
import ru.yandex.partner.libs.i18n.translation.source.factory.TranslationSourceFactory;

public class TranslationPatchGeneratorTest {

    private static TmpFileHelper tmpFileHelper;

    private static MessagesCollector messagesCollector;
    private static File resourceDirectory;
    private static ResourceBundleNameResolver resourceBundleNameResolver;
    private static GettextMessageKeyResolver gettextMessageKeyResolver;
    private static StringEscaper stringEscaper;
    private static TranslationSourceFactory translationSourceFactory;

    private static TranslationPatchGenerator translationPatchGenerator;

    @BeforeAll
    public static void init() throws IOException {
        tmpFileHelper = new TmpFileHelper();

        messagesCollector = new MessagesCollector("ru.yandex.partner.libs.i18n.testmsg");
        resourceDirectory = tmpFileHelper.getTmpDir();
        resourceBundleNameResolver = new DashResourceBundleNameResolver("translations");
        gettextMessageKeyResolver = new TankerGettextMessageKeyResolver();
        stringEscaper = new TankerPOStringEscaper();
        translationSourceFactory = new ResourceFileTranslationSourceFactory(resourceDirectory,
                resourceBundleNameResolver, gettextMessageKeyResolver, stringEscaper);

        translationPatchGenerator = new TranslationPatchGenerator(messagesCollector, translationSourceFactory,
                gettextMessageKeyResolver);
    }

    @BeforeEach
    @AfterEach
    public void cleanup() {
        FileSystemUtils.deleteRecursively(resourceDirectory);
    }

    @Test
    public void allTranslationsExistTest() throws IOException, MsgNotEnumException {
        tmpFileHelper.copyResourceToTmp("translations/test-messages/messages_ru.json",
                "translations/test-messages/messages_ru.json");

        Path patchFilePath = Path.of(resourceDirectory.getAbsolutePath(), "translation_patch.json");
        translationPatchGenerator.generatePatch("ru", patchFilePath);

        Assertions.assertFalse(Files.exists(patchFilePath));
    }

    @Test
    public void noTranslationsExistTest() throws IOException, MsgNotEnumException, JSONException {

        tmpFileHelper.writeTmpFile("translations/test-messages/messages_ru.json", "{\"ru\": {}}".getBytes());

        Path patchFilePath = Path.of(resourceDirectory.getAbsolutePath(), "translation_patch.json");
        translationPatchGenerator.generatePatch("ru", patchFilePath);

        Assertions.assertTrue(Files.exists(patchFilePath));

        String expectedPatch = new String(
                tmpFileHelper.readResource("test/patch-generator/no-translations-exist/patch.json"));
        String actualPatch = Files.readString(patchFilePath);

        JSONAssert.assertEquals(expectedPatch, actualPatch, true);
    }

    @Test
    public void someTranslationsExistTest() throws IOException, MsgNotEnumException, JSONException {

        tmpFileHelper.copyResourceToTmp("test/patch-generator/some-translations-exist/resource.json",
                "translations/test-messages/messages_ru.json");
        tmpFileHelper.copyResourceToTmp("test/patch-generator/some-translations-exist/existing-patch.json",
                "translation_patch.json");

        Path patchFilePath = Path.of(resourceDirectory.getAbsolutePath(), "translation_patch.json");
        translationPatchGenerator.generatePatch("ru", patchFilePath);

        Assertions.assertTrue(Files.exists(patchFilePath));

        String expectedPatch = new String(
                tmpFileHelper.readResource("test/patch-generator/some-translations-exist/patch.json"));
        String actualPatch = Files.readString(patchFilePath);

        JSONAssert.assertEquals(expectedPatch, actualPatch, true);
    }


}
