package ru.yandex.partner.libs.i18n.sync;

import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;

import com.fasterxml.jackson.databind.ObjectMapper;
import org.json.JSONException;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.BeforeAll;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.mockito.ArgumentCaptor;
import org.skyscreamer.jsonassert.JSONAssert;
import org.springframework.util.FileSystemUtils;

import ru.yandex.partner.libs.i18n.escaper.StringEscaper;
import ru.yandex.partner.libs.i18n.resolver.bundle.DashResourceBundleNameResolver;
import ru.yandex.partner.libs.i18n.resolver.bundle.ResourceBundleNameResolver;
import ru.yandex.partner.libs.i18n.resolver.key.GettextMessageKeyResolver;
import ru.yandex.partner.libs.i18n.tanker.client.TankerClient;
import ru.yandex.partner.libs.i18n.tanker.client.TankerClientException;
import ru.yandex.partner.libs.i18n.tanker.dto.tjson.TankerDetailedJsonDoc;
import ru.yandex.partner.libs.i18n.tanker.escaper.TankerPOStringEscaper;
import ru.yandex.partner.libs.i18n.tanker.resolver.TankerGettextMessageKeyResolver;
import ru.yandex.partner.libs.i18n.tanker.sync.TranslationPatchUploader;
import ru.yandex.partner.libs.i18n.testing.MessagesCollector;
import ru.yandex.partner.libs.i18n.testing.MsgNotEnumException;
import ru.yandex.partner.libs.i18n.translation.source.factory.ResourceFileTranslationSourceFactory;
import ru.yandex.partner.libs.i18n.translation.source.factory.TranslationSourceFactory;

import static org.mockito.ArgumentMatchers.any;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.reset;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;

public class TranslationPatchUploaderTest {

    private static TmpFileHelper tmpFileHelper;

    private static MessagesCollector messagesCollector;
    private static TankerClient tankerClient;
    private static File resourceDirectory;
    private static ResourceBundleNameResolver resourceBundleNameResolver;
    private static GettextMessageKeyResolver gettextMessageKeyResolver;
    private static StringEscaper stringEscaper;
    private static TranslationSourceFactory translationSourceFactory;

    private static TranslationPatchUploader translationPatchUploader;

    @BeforeAll
    public static void init() throws IOException {
        tmpFileHelper = new TmpFileHelper();

        messagesCollector = new MessagesCollector("ru.yandex.partner.libs.i18n.testmsg");
        tankerClient = mock(TankerClient.class);
        resourceDirectory = tmpFileHelper.getTmpDir();
        resourceBundleNameResolver = new DashResourceBundleNameResolver("translations");
        gettextMessageKeyResolver = new TankerGettextMessageKeyResolver();
        stringEscaper = new TankerPOStringEscaper();
        translationSourceFactory = new ResourceFileTranslationSourceFactory(resourceDirectory,
                resourceBundleNameResolver, gettextMessageKeyResolver, stringEscaper);

        translationPatchUploader = new TranslationPatchUploader(messagesCollector, translationSourceFactory,
                gettextMessageKeyResolver, stringEscaper, tankerClient);
    }

    @BeforeEach
    @AfterEach
    public void cleanup() {
        FileSystemUtils.deleteRecursively(resourceDirectory);
        reset(tankerClient);
    }

    @Test
    public void someTranslationsExistTest() throws IOException, MsgNotEnumException, TankerClientException,
            JSONException {
        tmpFileHelper.copyResourceToTmp("test/patch-uploader/some-translations-exist/resource.json",
                "translations/test-messages/messages_ru.json");
        tmpFileHelper.copyResourceToTmp("test/patch-uploader/some-translations-exist/existing-patch.json",
                "translation_patch.json");

        Path patchFilePath = Path.of(resourceDirectory.getAbsolutePath(), "translation_patch.json");
        translationPatchUploader.uploadPatch("ru", patchFilePath);

        ArgumentCaptor<TankerDetailedJsonDoc> captor = ArgumentCaptor.forClass(TankerDetailedJsonDoc.class);
        verify(tankerClient, times(1)).uploadTranslations(captor.capture());

        String expectedDoc = new String(
                tmpFileHelper.readResource("test/patch-uploader/some-translations-exist/sent.json"));
        String sentDoc = (new ObjectMapper()).writeValueAsString(captor.getValue());

        JSONAssert.assertEquals(expectedDoc, sentDoc, true);

        Assertions.assertFalse(Files.exists(patchFilePath));
    }

    @Test
    public void noPatchFileTest() throws IOException, MsgNotEnumException, TankerClientException {
        tmpFileHelper.copyResourceToTmp("translations/test-messages/messages_ru.json",
                "translations/test-messages/messages_ru.json");

        Path patchFilePath = Path.of(resourceDirectory.getAbsolutePath(), "translation_patch.json");
        translationPatchUploader.uploadPatch("ru", patchFilePath);

        verify(tankerClient, times(0)).uploadTranslations(any());
    }

    @Test
    public void allTranslationsExistTest() throws IOException, MsgNotEnumException, TankerClientException {
        tmpFileHelper.copyResourceToTmp("translations/test-messages/messages_ru.json",
                "translations/test-messages/messages_ru.json");
        tmpFileHelper.writeTmpFile("translation_patch.json", "{}".getBytes());

        Path patchFilePath = Path.of(resourceDirectory.getAbsolutePath(), "translation_patch.json");
        translationPatchUploader.uploadPatch("ru", patchFilePath);

        verify(tankerClient, times(0)).uploadTranslations(any());
    }


}
