package ru.yandex.partner.libs.memcached;

import java.util.concurrent.ExecutionException;

import net.spy.memcached.MemcachedClient;
import net.spy.memcached.internal.OperationFuture;
import net.spy.memcached.transcoders.Transcoder;

public class MemcachedServiceImpl implements MemcachedService {
    private final MemcachedClient memcachedClient;
    private final Transcoder<Object> transcoder;
    private final MemcachedKeyMaker memcachedKeyMaker;

    public MemcachedServiceImpl(MemcachedClient memcachedClient,
                                Transcoder<Object> transcoder,
                                MemcachedKeyMaker memcachedKeyMaker
    ) {
        this.memcachedClient = memcachedClient;
        this.memcachedKeyMaker = memcachedKeyMaker;
        this.transcoder = transcoder;
    }

    @Override
    public boolean delete(String prefix, String key) {
        OperationFuture<Boolean> operationFuture = memcachedClient.delete(memcachedKeyMaker.makeKey(prefix, key));
        try {
            return operationFuture.get();
        } catch (InterruptedException | ExecutionException e) {
            throw new MemcachedException("Memcached DELETE operation has failed", e);
        }
    }

    @Override
    public boolean flush() {
        OperationFuture<Boolean> operationFuture = memcachedClient.flush();
        try {
            return operationFuture.get();
        } catch (InterruptedException | ExecutionException e) {
            throw new MemcachedException("Memcached FLUSH operation has failed", e);
        }
    }

    /**
     * Уменьшить значение (prefix, key) на 1
     *
     * @return новое значение,
     * 0 - если текущее значение меньше value
     * -1 - если (prefix, key) не найден
     */
    public long decr(String prefix, String key) {
        return decr(prefix, key, 1L);
    }

    /**
     * Уменьшить значение (prefix, key) на value
     *
     * @param value на сколько уменшить текущее значение
     * @return новое значение,
     * 0 - если текущее значение меньше value
     * -1 - если (prefix, key) не найден
     */
    public long decr(String prefix, String key, long value) {
        return memcachedClient.decr(memcachedKeyMaker.makeKey(prefix, key), value);
    }

    /**
     * Увеличить значение (prefix, key) на 1
     *
     * @return новое значение,
     * -1 - если (prefix, key) не найден
     */
    public long incr(String prefix, String key) {
        return incr(prefix, key, 1L);
    }

    /**
     * Увеличить значение (prefix, key) на value
     *
     * @param value на сколько увеличить текущее значение
     * @return новое значение,
     * -1 - если (prefix, key) не найден
     */
    public long incr(String prefix, String key, long value) {
        return memcachedClient.incr(memcachedKeyMaker.makeKey(prefix, key), value);
    }

    public String get(String prefix, String key) {
        return (String) memcachedClient.get(memcachedKeyMaker.makeKey(prefix, key), transcoder);
    }

    /**
     * @param value Только строка. Числа нельзя потому что perl не считывает java закодированные числа
     *              (Возможно надо будет пожружить в будущем)
     */
    public boolean set(String prefix, String key, String value, int expireTime) {
        OperationFuture<Boolean> operationFuture =
                memcachedClient.set(memcachedKeyMaker.makeKey(prefix, key), expireTime, value, transcoder);

        try {
            return operationFuture.get();
        } catch (InterruptedException | ExecutionException e) {
            throw new MemcachedException("Memcached SET operation has failed", e);
        }
    }
}
