package ru.yandex.partner.libs.multistate.action;

import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Predicate;

import ru.yandex.direct.model.ModelProperty;
import ru.yandex.partner.core.multistate.Multistate;
import ru.yandex.partner.core.multistate.StateFlag;
import ru.yandex.partner.libs.i18n.GettextMsg;

import static ru.yandex.partner.libs.multistate.MultistatePredicates.any;

public class ActionEntry<M, T extends StateFlag> {

    private final GettextMsg titleMsg;
    private final List<ActionCheckId> checks;
    private final Predicate<Multistate<T>> predicate;
    private final Set<ModelProperty<? super M, ?>> requiredProperties;
    private final Map<T, Boolean> setFlags;

    private ActionEntry(GettextMsg titleMsg,
                        List<ActionCheckId> checks,
                        Predicate<Multistate<T>> predicate,
                        Set<ModelProperty<? super M, ?>> requiredProperties,
                        Map<T, Boolean> setFlags) {
        this.titleMsg = titleMsg;
        this.checks = checks;
        this.predicate = predicate;
        this.requiredProperties = Collections.unmodifiableSet(requiredProperties);
        this.setFlags = setFlags;
    }

    public GettextMsg getTitleMsg() {
        return titleMsg;
    }

    public List<ActionCheckId> getChecks() {
        return checks;
    }

    public Predicate<Multistate<T>> getPredicate() {
        return predicate;
    }

    public Set<ModelProperty<? super M, ?>> getRequiredProperties() {
        return requiredProperties;
    }

    public Map<T, Boolean> getSetFlags() {
        return setFlags;
    }

    public static class Builder<M, T extends StateFlag> {
        private final GettextMsg titleMsg;
        private List<ActionCheckId> checks;
        private Predicate<Multistate<T>> predicate;
        private Set<ModelProperty<? super M, ?>> requiredProperties;
        private Map<T, Boolean> setFlags;

        public Builder(GettextMsg titleMsg) {
            this.titleMsg = titleMsg;
        }

        public Builder<M, T> setChecks(List<ActionCheckId> checks) {
            this.checks = checks;
            return this;
        }

        public Builder<M, T> setPredicate(Predicate<Multistate<T>> predicate) {
            this.predicate = predicate;
            return this;
        }

        public Builder<M, T> addRequiredProperties(Set<ModelProperty<? super M, ?>> requiredProperties) {
            this.requiredProperties.addAll(requiredProperties);
            return this;
        }

        public Builder<M, T> setRequiredProperties(Set<ModelProperty<? super M, ?>> requiredProperties) {
            this.requiredProperties = new HashSet<>(requiredProperties);
            return this;
        }

        public Builder<M, T> setSetFlags(Map<T, Boolean> setFlags) {
            this.setFlags = setFlags;
            return this;
        }

        public Builder<M, T> setSetFlags(T flag, Boolean value) {
            this.setFlags = Map.of(flag, value);
            return this;
        }

        public ActionEntry<M, T> build() {
            return new ActionEntry<>(
                    titleMsg,
                    Optional.ofNullable(checks).orElse(List.of()),
                    Optional.ofNullable(predicate).orElse(any()),
                    Optional.ofNullable(requiredProperties).orElse(Set.of()),
                    Optional.ofNullable(setFlags).orElse(Collections.emptyMap())
            );
        }

    }
}
