package ru.yandex.partner.libs.multistate.graph;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Deque;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import one.util.streamex.EntryStream;

import ru.yandex.direct.model.ModelProperty;
import ru.yandex.partner.core.action.Action;
import ru.yandex.partner.core.multistate.Multistate;
import ru.yandex.partner.core.multistate.StateFlag;
import ru.yandex.partner.libs.i18n.GettextMsg;
import ru.yandex.partner.libs.multistate.action.ActionCheckId;
import ru.yandex.partner.libs.multistate.action.ActionEntry;
import ru.yandex.partner.libs.multistate.action.ActionNameHolder;
import ru.yandex.partner.libs.utils.StreamUtils;

@ParametersAreNonnullByDefault
public abstract class AbstractMultistateGraph<M, T extends StateFlag> implements MultistateGraph<M, T> {

    private final Map<String, ActionEntry<M, T>> actionEntryMap = new HashMap<>();
    private final Set<Multistate<T>> reachableMultistates = new HashSet<>();
    private Map<Multistate<T>, Map<String, Multistate<T>>> allTransitions;
    private Map<String, List<Long>> multistatesByAction;

    public AbstractMultistateGraph() {
        for (Map.Entry<ActionNameHolder, ActionEntry<M, T>> entry :
                createGraph().entrySet()) {
            actionEntryMap.put(entry.getKey().getActionName(), entry.getValue());
        }
        initReachableMultistates();
    }

    @Override
    public Map<String, List<Boolean>> checkActionsAllowed(Collection<String> actionNames, List<? extends M> models) {
        Map<String, List<Boolean>> result = Maps.newHashMapWithExpectedSize(actionNames.size());
        Set<String> filteredActionNames = Sets.newHashSetWithExpectedSize(actionNames.size());
        actionNames.forEach(actionName -> {
            if (!actionEntryMap.containsKey(actionName)) {
                result.put(actionName, Collections.nCopies(models.size(), false));
            } else {
                filteredActionNames.add(actionName);
            }
        });

        result.putAll(checkActionsInternal(filteredActionNames, models));

        return result;
    }

    @Override
    public Map<? extends M, Boolean> isActionAllowed(String actionName, List<? extends M> models) {
        return EntryStream.zip(models, checkActionAllowed(actionName, models))
                .toMap();
    }

    @Override
    public List<? extends M> filteredByActionAllowed(String actionName, List<? extends M> models) {
        return EntryStream.of(isActionAllowed(actionName, models))
                .filterValues(isAllowed -> isAllowed)
                .map(Map.Entry::getKey)
                .toList();
    }

    @Override
    public Map<String, Boolean> checkActionsAllowed(Collection<String> actionNames, M model) {
        return checkActionsAllowed(actionNames, List.of(model)).entrySet().stream()
                .collect(Collectors.toMap(Map.Entry::getKey, e -> e.getValue().get(0)));
    }

    @Override
    public List<Boolean> checkActionAllowed(String actionName, List<? extends M> models) {
        return checkActionsAllowed(List.of(actionName), models).get(actionName);
    }

    @Override
    public Boolean checkActionAllowed(String actionName, M model) {
        return checkActionAllowed(actionName, List.of(model)).get(0);
    }

    @Override
    public List<Set<Action>> getAllowedActions(List<? extends M> models) {

        List<Set<Action>> result = models.stream()
                .map(m -> new HashSet<Action>()).collect(Collectors.toList());

        checkActionsInternal(actionEntryMap.keySet(), models).forEach((actionName, checkResult) -> {
            Action action = new Action(actionName, actionEntryMap.get(actionName).getTitleMsg());
            for (int i = 0; i < checkResult.size(); i++) {
                if (checkResult.get(i)) {
                    result.get(i).add(action);
                }
            }
        });

        return result;
    }

    @Override
    public Set<Action> getAllowedActions(M model) {
        return getAllowedActions(List.of(model)).get(0);
    }

    @Override
    public Set<String> getAllAvailableActionNames() {
        return actionEntryMap.keySet();
    }

    @Override
    public Map<T, Boolean> getFlagModificationsForAction(String actionName) {
        return Optional.of(actionEntryMap.get(actionName)).map(ActionEntry::getSetFlags).map(HashMap::new)
                .orElseGet(HashMap::new);
    }

    @Override
    public Set<Multistate<T>> getReachableMultistates() {
        return Set.copyOf(reachableMultistates);
    }

    @Override
    public Set<Multistate<T>> getMultistatesForPredicate(Predicate<Multistate<T>> predicate) {
        return reachableMultistates.stream().filter(predicate).collect(Collectors.toSet());
    }

    @Override
    public Map<Multistate<T>, Map<String, Multistate<T>>> getAllTransitions() {
        return this.allTransitions;
    }

    private Map<Multistate<T>, Map<String, Multistate<T>>> calculateAllTransitions() {
        return reachableMultistates.stream().collect(Collectors.toMap(
                Multistate::copy, this::getAllTransitionsForMultistate
        ));
    }

    @Override
    public Set<ModelProperty<? super M, ?>> getAllRequiredProperties() {
        return actionEntryMap.values().stream().map(ActionEntry::getRequiredProperties)
                .flatMap(Set::stream).collect(Collectors.toSet());
    }

    @Override
    public Set<ModelProperty<? super M, ?>> getRequiredPropertiesByActionName(String actionName) {
        return actionEntryMap.get(actionName).getRequiredProperties();
    }

    protected Map<String, List<Boolean>> checkActionsInternal(Set<String> actionNames, List<? extends M> models) {
        // вычисляем, какие действия применимы к каким моделям по мультистатусу
        Map<String, List<Boolean>> actionsApplicability = actionNames.stream().collect(Collectors.toMap(
                Function.identity(),
                actionName -> checkActionApplicability(actionName, models)
        ));

        // список действий, применимых к каким-либо моделям
        List<String> applicableActionNames = actionNames.stream()
                .filter(actionName -> actionsApplicability.get(actionName).stream().anyMatch(Predicate.isEqual(true)))
                .collect(Collectors.toList());

        // список необходимых проверок со списком действий, для которых они нужны
        Map<ActionCheckId, Set<String>> actionsByChecks = getNecessaryChecks(applicableActionNames);

        // если никакие проверки не нужны
        if (actionsByChecks.isEmpty()) {
            return actionsApplicability;
        }

        // применимость для всех необходимых проверок
        Map<? extends ActionCheckId, List<Boolean>> checksApplicability = actionsByChecks.entrySet().stream()
                .collect(Collectors.toMap(
                        Map.Entry::getKey,
                        entry -> getApplicabilityForSetOfActions(models.size(), actionsApplicability, entry.getValue())
                ));

        // производим проверки
        Map<? extends ActionCheckId, List<Boolean>> checkResults = checksApplicability.entrySet().stream()
                .collect(Collectors.toMap(
                        Map.Entry::getKey,
                        entry -> performCheckForApplicable(entry.getKey(), models, entry.getValue())
                ));

        // соединяем доступность дейстивий и результаты проверок
        return actionNames.stream().collect(Collectors.toMap(
                Function.identity(),
                actionName -> combineAvailabilityAndCheckResults(actionName, actionsApplicability, checkResults)
        ));

    }

    /**
     * Проверяет, к каким из моделей можно применить действие, исходя из мультистатусов
     * Этот метод можно переопределить для добавления кастомных условий,
     * которые нужно применить для всех дейстивий и всех моделей.
     */
    protected List<Boolean> checkActionApplicability(String actionName, List<? extends M> models) {
        ActionEntry<M, T> actionEntry = actionEntryMap.get(actionName);
        Predicate<Multistate<T>> predicate = actionEntry.getPredicate();
        return models.stream().map(model -> predicate.test(getMultistateFromModel(model)))
                .collect(Collectors.toList());
    }

    protected ActionEntry.Builder<M, T> getActionEntryBuilder(GettextMsg titleMsg,
                                                              Set<ModelProperty<? super M, ?>> multistateProperties) {
        return new ActionEntry.Builder<M, T>(titleMsg)
                .setRequiredProperties(multistateProperties);
    }

    protected abstract Map<ActionNameHolder, ActionEntry<M, T>> createGraph();

    /**
     * Переопределяется для конкретных классов
     */
    protected List<Boolean> performCheck(ActionCheckId check, List<? extends M> models) {
        return Collections.nCopies(models.size(), false);
    }

    protected abstract Multistate<T> getMultistateForValue(Long multistateValue);

    // dfs traversal of the graph
    private void initReachableMultistates() {
        Multistate<T> emptyMultistate = getMultistateForValue(0L);

        Deque<Multistate<T>> vertexQueue = new LinkedList<>();
        vertexQueue.add(emptyMultistate);

        reachableMultistates.add(emptyMultistate);

        List<ActionEntry<M, T>> stateChangingActionEntries = actionEntryMap.values().stream()
                .filter(actionEntry -> !actionEntry.getSetFlags().isEmpty())
                .collect(Collectors.toList());

        while (!vertexQueue.isEmpty()) {
            Multistate<T> currentMultistate = vertexQueue.pollFirst();
            stateChangingActionEntries.stream()
                    .filter(actionEntry -> actionEntry.getPredicate().test(currentMultistate))
                    .forEach(actionEntry -> {
                        Multistate<T> newMultistate = copyAndApplyFlags(currentMultistate, actionEntry.getSetFlags());
                        if (!reachableMultistates.contains(newMultistate)) {
                            reachableMultistates.add(newMultistate);
                            vertexQueue.add(newMultistate);
                        }
                    });
        }

        this.allTransitions = this.calculateAllTransitions();

        this.multistatesByAction = this.calculateMultistatesByAction();
    }

    @Nonnull
    private Multistate<T> copyAndApplyFlags(Multistate<T> currentMultistate, Map<T, Boolean> flags) {
        Multistate<T> newMultistate = currentMultistate.copy();
        flags.forEach(newMultistate::setFlag);
        return newMultistate;
    }

    @Nonnull
    private Map<String, Multistate<T>> getAllTransitionsForMultistate(Multistate<T> multistate) {
        return actionEntryMap.entrySet().stream()
                .filter(e -> e.getValue().getPredicate().test(multistate))
                .collect(Collectors.toMap(
                        Map.Entry::getKey,
                        e -> copyAndApplyFlags(multistate, e.getValue().getSetFlags())));
    }

    private List<Boolean> performCheckForApplicable(ActionCheckId check, List<? extends M> models,
                                                    List<Boolean> applicability) {
        List<Boolean> result = new ArrayList<>(models.size());
        List<M> filteredModels = new ArrayList<>(models.size());
        StreamUtils.forEachPair(models, applicability, (model, isApplicable) -> {
            result.add(isApplicable);
            if (isApplicable) {
                filteredModels.add(model);
            }
        });
        if (!filteredModels.isEmpty()) {
            List<Boolean> checkResult = performCheck(check, filteredModels);
            int j = 0;
            for (int i = 0; i < result.size(); i++) {
                if (result.get(i)) {
                    result.set(i, checkResult.get(j));
                    j++;
                }
            }
        }
        return result;
    }

    private List<Boolean> combineAvailabilityAndCheckResults(String actionName,
                                                             Map<String, List<Boolean>> actionsApplicability,
                                                             Map<? extends ActionCheckId, List<Boolean>> checkResults) {
        List<? extends ActionCheckId> checks =
                Optional.ofNullable(actionEntryMap.get(actionName).getChecks()).orElse(List.of());
        List<List<Boolean>> results = checks.stream()
                .map(checkResults::get)
                .filter(Objects::nonNull)
                .collect(Collectors.toList());
        return StreamUtils.reduceLists(actionsApplicability.get(actionName), results, Boolean::logicalAnd);
    }

    private List<Boolean> getApplicabilityForSetOfActions(int size, Map<String, List<Boolean>> actionsApplicability,
                                                          Set<String> actionNamesForCheck) {
        List<List<Boolean>> availabilities = actionNamesForCheck.stream()
                .map(actionsApplicability::get)
                .collect(Collectors.toList());
        return StreamUtils.reduceLists(Collections.nCopies(size, false), availabilities, Boolean::logicalOr);
    }

    private Map<ActionCheckId, Set<String>> getNecessaryChecks(List<String> applicableActionNames) {
        Map<ActionCheckId, Set<String>> actionsByChecks = new HashMap<>();
        applicableActionNames.forEach(actionName -> {
            List<? extends ActionCheckId> checks = actionEntryMap.get(actionName).getChecks();
            if (checks == null) {
                return;
            }
            checks.forEach(check -> actionsByChecks.computeIfAbsent(check, anEnum -> new HashSet<>()).add(actionName));
        });
        return actionsByChecks;
    }

    public Boolean containsAction(String actionName) {
        return actionEntryMap.containsKey(actionName);
    }

    public ActionEntry<M, T> getActionEntry(String actionName) {
        return actionEntryMap.get(actionName);
    }

    @Override
    public List<Long> getMultistatesByAction(String actionName) {
        return this.multistatesByAction.get(actionName);
    }

    private Map<String, List<Long>> calculateMultistatesByAction() {
        Map<Multistate<T>, Map<String, Multistate<T>>> graph = this.getAllTransitions();

        Map<String, List<Long>> result = new HashMap<>();
        for (Map.Entry<Multistate<T>, Map<String, Multistate<T>>> transition : graph.entrySet()) {
            Map<String, Multistate<T>> newMultistateByAction = transition.getValue();

            for (Map.Entry<String, Multistate<T>> entry : newMultistateByAction.entrySet()) {
                result.computeIfAbsent(entry.getKey(), e -> new ArrayList<>()).add(transition.getKey().toMultistateValue());
            }
        }

        return result;
    }
}
