package ru.yandex.partner.libs.multistate.expression;

import java.util.List;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import java.util.stream.LongStream;

import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.params.ParameterizedTest;
import org.junit.jupiter.params.provider.Arguments;
import org.junit.jupiter.params.provider.MethodSource;

import ru.yandex.partner.core.multistate.Multistate;
import ru.yandex.partner.core.multistate.user.UserMultistate;
import ru.yandex.partner.core.multistate.user.UserStateFlag;

import static java.util.function.Predicate.not;
import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.junit.jupiter.api.Assertions.assertThrows;
import static ru.yandex.partner.libs.multistate.MultistatePredicates.empty;
import static ru.yandex.partner.libs.multistate.MultistatePredicates.has;

public class MultistateExpressionParserTest {

    private MultistateExpressionParser<UserStateFlag> expressionParser;

    @BeforeEach
    public void init() {
        expressionParser = new MultistateExpressionParser<>(UserStateFlag.class);
    }

    @ParameterizedTest(name = "{0}")
    @MethodSource("testData")
    public void TestParseExpression(String expression, Predicate<Multistate<UserStateFlag>> expectedPredicate) {

        List<UserMultistate> userMultistates =
                LongStream.range(0, 16).boxed().map(UserMultistate::new).collect(Collectors.toList());

        Predicate<Multistate<UserStateFlag>> result = expressionParser.parseExpression(expression);
        userMultistates.forEach(multistate -> {
            assertEquals(expectedPredicate.test(multistate), result.test(multistate));
        });

    }

    @Test
    public void testBadExpression() {
        List<String> badExpressions = List.of(
                "",
                "contacts_provided OR",
                "(contacts_provided",
                "no contacts_provided",
                "N O T contacts_provided"
        );

        badExpressions.forEach(s -> {
            Throwable e = assertThrows(ExpressionParsingException.class,
                    () -> expressionParser.parseExpression(s));
            assertEquals(e.getMessage(), String.format("Syntax error in expression \"%s\"", s));
        });
    }

    @Test
    public void testBadParameter() {
        String badExpression = "contacts_providedddd";

        Throwable e = assertThrows(ExpressionParsingException.class,
                () -> expressionParser.parseExpression(badExpression));
        assertEquals(e.getMessage(), "Status \"contacts_providedddd\" does not exists");
    }

    private static List<Arguments> testData() {
        return List.of(
                Arguments.of("__EMPTY__", empty()),
                Arguments.of("contacts_provided", has(UserStateFlag.CONTACTS_PROVIDED)),
                Arguments.of("NOT contacts_provided", not(has(UserStateFlag.CONTACTS_PROVIDED))),
                Arguments.of("not contacts_provided", not(has(UserStateFlag.CONTACTS_PROVIDED))),
                Arguments.of("contacts_provided AND need_yan_contract",
                        has(UserStateFlag.CONTACTS_PROVIDED).and(has(UserStateFlag.NEED_YAN_CONTRACT))),
                Arguments.of("__EMPTY__ OR (contacts_provided AND need_yan_contract)",
                        has(UserStateFlag.CONTACTS_PROVIDED).and(has(UserStateFlag.NEED_YAN_CONTRACT)).or(empty()))
        );
    }

}
