package ru.yandex.partner.libs.rbac.group;

import java.util.Set;
import java.util.stream.Collectors;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import static com.google.common.base.Preconditions.checkNotNull;

@Service
public class GroupService {
    private final Set<Group> groups;
    private final Set<String> groupNames;
    private final Set<Long> groupIds;

    @Autowired
    public GroupService(GroupRepository groupRepository) {
        this.groups = groupRepository.getGroups();
        this.groupNames = this.groups.stream()
                .map(Group::getName)
                .map(String::toUpperCase)
                .collect(Collectors.toUnmodifiableSet());
        this.groupIds = this.groups.stream()
                .map(Group::getId)
                .collect(Collectors.toUnmodifiableSet());
    }

    public Set<String> getGroupNames() {
        return this.groupNames;
    }

    public Set<Long> getGroupIds() {
        return this.groupIds;
    }

    public Set<Group> getGroups() {
        return this.groups;
    }

    public boolean isGroupExists(String groupName) {
        checkNotNull(groupName);
        return this.groupNames.contains(groupName.toUpperCase());
    }

    public boolean isGroupExists(long groupId) {
        return this.groupIds.contains(groupId);
    }

    private void checkGroupName(String name) {
        if (!isGroupExists(name)) {
            throw new GroupNotExistsException(String.format("%s group not exists", name));
        }
    }

    private void checkGroupId(long id) {
        if (!isGroupExists(id)) {
            throw new GroupNotExistsException(String.format("group with id=%d not exists", id));
        }
    }

    public Group getGroupByName(String groupName) {
        checkGroupName(groupName);
        return Group.valueOf(groupName);
    }

    public Group getGroupById(long groupId) {
        checkGroupId(groupId);
        return groups.stream().filter(v -> v.getId() == groupId).findFirst().orElseThrow();
    }
}
