package ru.yandex.partner.libs.rbac.role;

import java.util.Collection;
import java.util.Collections;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.partner.core.role.Role;
import ru.yandex.partner.libs.rbac.group.Group;
import ru.yandex.partner.libs.rbac.right.Right;

import static com.google.common.base.Preconditions.checkNotNull;

@Service
public class RoleService {
    private static final Set<Role> ROLES = RoleRepository.getRoles();
    private final Map<Role, Set<Right>> roleToRightsMap;

    @Autowired
    public RoleService(RoleRepository roleRepository) {
        this.roleToRightsMap = roleRepository.getRoleToRightsMap();
    }

    public boolean isRoleExists(Role role) {
        checkNotNull(role);
        return RoleService.ROLES.contains(role);
    }

    public boolean isRolesExists(Set<Role> roles) {
        checkNotNull(roles);
        return RoleService.ROLES.containsAll(roles);
    }

    public void checkRoleExists(Role role) {
        if (!isRoleExists(role)) {
            throw new RoleNotExistsException();
        }
    }

    public void checkRolesExists(Set<Role> roles) {
        if (!isRolesExists(roles)) {
            throw new RoleNotExistsException();
        }
    }

    public boolean isConflictedRolesValid(Set<Role> roles) {
        // Check conflicted roles
        checkNotNull(roles);
        Set<Role> conflictedRolesIntersection = roles.stream()
                .map(Role::getConflictRoleForSet)
                .filter(Objects::nonNull)
                .flatMap(Collection::stream)
                .filter(roles::contains)
                .collect(Collectors.toSet());
        return conflictedRolesIntersection.isEmpty();
    }

    //todo: объеденить методы с помощь ссылки Role::
    public boolean isRequiredRolesValid(Set<Role> roles) {
        // Check required roles
        checkNotNull(roles);
        Set<Role> requiredRoles = roles.stream()
                .map(Role::getRequiredRoleForSet)
                .filter(Objects::nonNull)
                .flatMap(Collection::stream)
                .filter(roles::contains)
                .collect(Collectors.toSet());
        return roles.containsAll(requiredRoles);
    }

    public boolean isRoleSetValid(Set<Role> roles) {
        return isConflictedRolesValid(roles) && isRequiredRolesValid(roles);
    }

    public void checkRoleValid(Set<Role> roles) {
        if (!isRoleSetValid(roles)) {
            throw new RoleSetInvalidException();
        }
    }

    public Set<Right> getRoleRights(Role role) {
        checkRoleExists(role);
        return roleToRightsMap.getOrDefault(role, Collections.emptySet());
    }

    public Set<String> getRoleRightNames(Role role) {
        return getRoleRights(role).stream()
                .map(Right::getName)
                .collect(Collectors.toSet());
    }

    public static Set<Role> getRoles() {
        return RoleService.ROLES;
    }

    public static boolean hasRolesFromGroup(Set<Role> roles, Group group) {
        Set<Role> groupRoles = getRoles().stream()
                .filter(r -> r.getGroup() == group.getId())
                .collect(Collectors.toSet());
        return roles.stream().anyMatch(groupRoles::contains);
    }
}
