package ru.yandex.partner.libs.rbac.userrole;

import java.util.Collection;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.partner.core.role.Role;
import ru.yandex.partner.libs.rbac.right.Right;
import ru.yandex.partner.libs.rbac.role.RoleService;

import static com.google.common.base.Preconditions.checkNotNull;

@Service
public class UserRoleServiceImpl implements UserRoleService {
    private final UserRoleRepository userRoleRepository;
    private final RoleService roleService;

    @Autowired
    public UserRoleServiceImpl(UserRoleRepository userRoleRepository, RoleService roleService) {
        this.userRoleRepository = userRoleRepository;
        this.roleService = roleService;
    }

    @Override
    public Set<Role> getUserRoles(long userId) {
        return this.userRoleRepository.fetchUserRoles(userId);
    }

    @Override
    public Map<Long, Set<Role>> getUserRolesMap(Collection<Long> userId) {
        return this.userRoleRepository.fetchUserRolesMap(userId);
    }

    @Override
    public Set<Right> getUserRightsFromRoles(Set<Role> rolesSet) {
        return rolesSet.stream()
                .map(roleService::getRoleRights)
                .flatMap(Set::stream)
                .collect(Collectors.toUnmodifiableSet());
    }

    @Override
    public Set<Right> getUserRights(long userId) {
        return getUserRoles(userId).stream()
                .flatMap(u -> roleService.getRoleRights(u).stream())
                .collect(Collectors.toUnmodifiableSet());
    }

    @Override
    public Set<String> getUserRightNames(long userId) {
        return getUserRights(userId).stream()
                .map(Right::getName)
                .collect(Collectors.toUnmodifiableSet());
    }

    @Override
    public boolean hasRole(long userId, Role role) {
        checkNotNull(role);
        return getUserRoles(userId).contains(role);
    }

    @Override
    public boolean hasRoles(long userId, Set<Role> roles) {
        checkNotNull(roles);
        return getUserRoles(userId).containsAll(roles);
    }

    @Override
    public boolean hasRight(long userId, Right right) {
        checkNotNull(right);
        return getUserRights(userId).contains(right);
    }

    @Override
    public boolean hasRights(long userId, Set<Right> rights) {
        checkNotNull(rights);
        return getUserRights(userId).containsAll(rights);
    }

    @Override
    public boolean hasRightByName(long userId, String rightName) {
        checkNotNull(rightName);
        return getUserRightNames(userId).contains(rightName);
    }

    @Override
    public boolean hasRightsByNames(long userId, Set<String> rightNames) {
        checkNotNull(rightNames);
        return getUserRightNames(userId).containsAll(rightNames);
    }

    @Override
    public void setUserRoles(long userId, Set<Role> roles) {
        checkNotNull(roles);
        this.roleService.checkRolesExists(roles);
        this.roleService.checkRoleValid(roles);
        this.userRoleRepository.updateUserRoleIds(userId, roles);
    }
}

