package ru.yandex.partner.libs.rbac.right;

import java.util.Set;
import java.util.stream.Collectors;

import org.junit.jupiter.api.BeforeAll;
import org.junit.jupiter.api.Test;
import org.springframework.core.env.Environment;

import ru.yandex.partner.core.role.Role;
import ru.yandex.partner.libs.rbac.role.RoleRepository;
import ru.yandex.partner.libs.rbac.role.RoleService;

import static org.assertj.core.api.Assertions.assertThat;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;
import static ru.yandex.partner.libs.rbac.constants.Constants.PRODUCTION;
import static ru.yandex.partner.libs.rbac.role.RoleSet.ADBLOCK_PARTNER;
import static ru.yandex.partner.libs.rbac.role.RoleSet.ADFOX;
import static ru.yandex.partner.libs.rbac.role.RoleSet.ASSESSOR;
import static ru.yandex.partner.libs.rbac.role.RoleSet.BUSINESS_UNIT;
import static ru.yandex.partner.libs.rbac.role.RoleSet.DEVELOPER;
import static ru.yandex.partner.libs.rbac.role.RoleSet.DISTRIBUTION_MANAGER;
import static ru.yandex.partner.libs.rbac.role.RoleSet.DSP_MANAGER;
import static ru.yandex.partner.libs.rbac.role.RoleSet.DSP_PARTNER;
import static ru.yandex.partner.libs.rbac.role.RoleSet.DSP_VIEWER;
import static ru.yandex.partner.libs.rbac.role.RoleSet.INDOOR_PARTNER;
import static ru.yandex.partner.libs.rbac.role.RoleSet.INTERNAL_YAN_ADMINISTRATOR;
import static ru.yandex.partner.libs.rbac.role.RoleSet.INTERNAL_YAN_MANAGER;
import static ru.yandex.partner.libs.rbac.role.RoleSet.INTERNAL_YAN_VIEWER;
import static ru.yandex.partner.libs.rbac.role.RoleSet.MOBILE_PARTNER;
import static ru.yandex.partner.libs.rbac.role.RoleSet.OUTDOOR_PARTNER;
import static ru.yandex.partner.libs.rbac.role.RoleSet.PROTECTED_PAGES_EDITOR;
import static ru.yandex.partner.libs.rbac.role.RoleSet.ROBOT_ASSISTANT;
import static ru.yandex.partner.libs.rbac.role.RoleSet.SITE_PARTNER;
import static ru.yandex.partner.libs.rbac.role.RoleSet.TUTBY;
import static ru.yandex.partner.libs.rbac.role.RoleSet.VIDEO_PARTNER;
import static ru.yandex.partner.libs.rbac.role.RoleSet.YAN_MANAGER;
import static ru.yandex.partner.libs.rbac.role.RoleSet.YAN_MODERATOR;
import static ru.yandex.partner.libs.rbac.role.RoleSet.YAN_PARTNER_ASSISTANT;
import static ru.yandex.partner.libs.rbac.role.RoleSet.YAN_VIEWER;

public class RoleRightTest {
    private static RightService rightService;
    private static RoleService roleService;

    @BeforeAll
    static void prepare() {
        Environment env = mock(Environment.class);
        when(env.getActiveProfiles()).thenReturn(new String[]{PRODUCTION});

        rightService = new RightService(new RightRepository(), env);
        roleService = new RoleService(new RoleRepository(rightService));
    }

    @Test
    void checkRightExists() {
        assertThat(rightService.isRightExists("widget_statistics_view_as_default__internal_context_report")).isTrue();
        assertThat(rightService.isRightExists("statistics_payment_view_field__login")).isTrue();
    }

    @Test
    void checkRightNotExists() {
        assertThat(rightService.isRightExists("")).isFalse();
        assertThat(rightService.isRightExists("rbac_test_block_presets_view")).isFalse();
        assertThat(rightService.isRightExists("not_exists_right")).isFalse();
    }

    @Test
    void checkRightRoleIds() {
        assertThat(
                rightService.getRightRoles("search_on_site_campaign_edit").stream()
                        .map(Role::getRoleId).collect(Collectors.toSet())
        ).containsExactlyInAnyOrder(2L, 9L, 15L, 37L, 39L);
    }

    @Test
    void checkRoles() {
        assertThat(RoleService.getRoles()).containsExactlyInAnyOrder(
                DEVELOPER,
                INTERNAL_YAN_MANAGER,
                INTERNAL_YAN_ADMINISTRATOR,
                DSP_PARTNER,
                DSP_MANAGER,
                SITE_PARTNER,
                YAN_MANAGER,
                YAN_MODERATOR,
                YAN_VIEWER,
                DSP_VIEWER,
                INTERNAL_YAN_VIEWER,
                VIDEO_PARTNER,
                MOBILE_PARTNER,
                BUSINESS_UNIT,
                ADBLOCK_PARTNER,
                YAN_PARTNER_ASSISTANT,
                TUTBY,
                INDOOR_PARTNER,
                ADFOX,
                OUTDOOR_PARTNER,
                ROBOT_ASSISTANT,
                ASSESSOR,
                PROTECTED_PAGES_EDITOR,
                DISTRIBUTION_MANAGER
        );
    }

    @Test
    void checkRoleSetValid() {
        assertThat(
                roleService.isRoleSetValid(
                        Set.of(DEVELOPER, INTERNAL_YAN_MANAGER, INTERNAL_YAN_ADMINISTRATOR)
                )
        ).isTrue();
        assertThat(
                roleService.isRoleSetValid(
                        Set.of(YAN_VIEWER, YAN_MANAGER)
                )
        ).isFalse();
        assertThat(
                roleService.isRoleSetValid(
                        Set.of(BUSINESS_UNIT, YAN_MANAGER)
                )
        ).isTrue();
        assertThat(
                roleService.isRoleSetValid(
                        Set.of(BUSINESS_UNIT, SITE_PARTNER)
                )
        ).isTrue();

        //conflict role for set
        assertThat(
                roleService.isRoleSetValid(
                        Set.of(BUSINESS_UNIT, SITE_PARTNER, ROBOT_ASSISTANT, ASSESSOR)
                )
        ).isTrue();

        assertThat(
                roleService.isRoleSetValid(
                        Set.of(BUSINESS_UNIT, SITE_PARTNER, ROBOT_ASSISTANT, ASSESSOR, DEVELOPER)
                )
        ).isTrue();

        assertThat(
                roleService.isRoleSetValid(
                        Set.of(BUSINESS_UNIT, SITE_PARTNER, ROBOT_ASSISTANT, ASSESSOR, DEVELOPER, YAN_VIEWER,
                                YAN_MANAGER)
                )
        ).isFalse();
    }

    @Test
    void checkHasRight() {
        assertThat(
                roleService.getRoleRightNames(DEVELOPER).containsAll(
                        Set.of(
                                "statistics_reports_delete",
                                "statistics_reports_view",
                                "statistics_search_on_site_mcb_view_entity_field__login",
                                "statistics_search_on_site_view_entity_field__login",
                                "queue_view"
                        )
                )
        ).isTrue();
        assertThat(
                roleService.getRoleRightNames(DEVELOPER).contains(
                        "rbac_test_block_presets_view"
                )
        ).isFalse();
    }
}
