package ru.yandex.partner.test.http.json.model;

import java.io.IOException;
import java.net.URI;
import java.net.URLEncoder;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Map;
import java.util.stream.Collectors;

import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.annotation.JsonRawValue;
import com.fasterxml.jackson.core.JsonPointer;
import com.fasterxml.jackson.core.filter.FilteringParserDelegate;
import com.fasterxml.jackson.core.filter.TokenFilter;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.annotation.JsonDeserialize;
import org.junit.platform.commons.util.StringUtils;
import org.springframework.http.HttpMethod;
import org.springframework.http.ResponseEntity;
import org.springframework.web.reactive.function.BodyInserters;
import org.springframework.web.reactive.function.client.WebClient;

import ru.yandex.partner.test.http.json.JsonObjectDeserializer;

public class TestHttpRequest {
    private String method;
    private String name;
    private String token;
    private String url;
    @JsonDeserialize(using = JsonObjectDeserializer.class)
    @JsonRawValue
    private String body;
    private Map<String, String> cookies;
    private Map<String, JsonNode> options = new HashMap<>();
    @JsonProperty("mock_features")
    private JsonNode mockedFeatures;

    public String getMethod() {
        return method;
    }

    public void setMethod(String method) {
        this.method = method;
    }

    public String getName() {
        return name;
    }

    public void setName(String name) {
        this.name = name;
    }

    public String getToken() {
        return token;
    }

    public void setToken(String token) {
        this.token = token;
    }

    public String getUrl() {
        return url;
    }

    public void setUrl(String url) {
        this.url = url;
    }

    public String getBody() {
        return body;
    }

    public void setBody(String body) {
        this.body = body;
    }

    public Map<String, String> getCookies() {
        return cookies;
    }

    public void setCookies(Map<String, String> cookies) {
        this.cookies = cookies;
    }

    public JsonNode getMockedFeatures() {
        return mockedFeatures;
    }

    public void setMockedFeatures(JsonNode mockedFeatures) {
        this.mockedFeatures = mockedFeatures;
    }

    public ResponseEntity<String> perform(String host) {
        HttpMethod httpMethod;
        switch (getMethod().toLowerCase()) {
            case "get":
                httpMethod = HttpMethod.GET;
                break;
            case "post":
                httpMethod = HttpMethod.POST;
                break;
            case "patch":
                httpMethod = HttpMethod.PATCH;
                break;
            default:
                throw new IllegalArgumentException("Method is invalid: " + getMethod());
        }

        WebClient.RequestBodySpec webClient = WebClient.create()
                .method(httpMethod)
                .uri(getUri(host));

        if (httpMethod == HttpMethod.POST || httpMethod == HttpMethod.PATCH) {
            webClient.header("Content-type", "application/vnd.api+json");
        }

        String requestBody = getBody();
        if (requestBody != null) {
            webClient.body(BodyInserters.fromValue(requestBody));
        }

        if (StringUtils.isNotBlank(getToken())) {
            String token = getToken();
            webClient.header("Authorization", "token " + token);

        }

        if (getCookies() != null && !getCookies().isEmpty()) {
            String cookies = getCookies()
                    .entrySet()
                    .stream()
                    .map(entry -> String.format("%s=%s", entry.getKey(), entry.getValue()))
                    .collect(Collectors.joining("; "));
            webClient.header("Cookie", cookies);
        }

        webClient.header("Accept", "application/vnd.api+json");

        return webClient
                // Чтобы не бросало эксепшены при 5хх и 4хх
                .exchangeToMono(clientResponse -> clientResponse.toEntity(String.class))
                .map(response -> {
                    if (!response.hasBody()) {
                        return response;
                    }

                    var jsonPaths = getOptions().get("json_path");
                    if (jsonPaths != null) {
                        if (!jsonPaths.isArray()) {
                            throw new IllegalStateException("json_path should be array of string paths");
                        }

                        MultiPointerFilter jsonPathFilter = getJsonPathFilter(jsonPaths);

                        try {
                            return new ResponseEntity<>(
                                    String.valueOf(new FilteringParserDelegate(
                                            new ObjectMapper().createParser(response.getBody()),
                                            jsonPathFilter,
                                            TokenFilter.Inclusion.INCLUDE_ALL_AND_PATH,
                                            false
                                    ).readValueAsTree()),
                                    response.getHeaders(),
                                    response.getStatusCode()
                            );
                        } catch (IOException e) {
                            throw new RuntimeException(e);
                        }
                    } else {
                        return response;
                    }
                })
                .block();
    }

    private MultiPointerFilter getJsonPathFilter(JsonNode jsonPaths) {
        var pointers = new ArrayList<JsonPointer>();
        pointers.add(JsonPointer.compile("/errors"));

        for (JsonNode jsonPath : jsonPaths) {
            pointers.add(JsonPointer.compile(jsonPath.asText().replace('.', '/')));
        }

        return new MultiPointerFilter(pointers);
    }

    private URI getUri(String host) {
        String url = getUrl();

        String[] paths = url.split("\\?", 2);

        StringBuilder stringBuilder = new StringBuilder(host);
        stringBuilder.append(paths[0]);

        if (paths.length > 1) {
            stringBuilder.append("?");

            String[] queryParamsArr = paths[1].split("&");
            int queryParamsCount = queryParamsArr.length;
            for (String queryParam : queryParamsArr) {
                String[] nameAndValue = queryParam.split("=", 2);

                stringBuilder.append(URLEncoder.encode(nameAndValue[0], StandardCharsets.UTF_8));
                if (nameAndValue.length > 1) {
                    stringBuilder.append("=");
                    stringBuilder.append(URLEncoder.encode(nameAndValue[1], StandardCharsets.UTF_8));
                }

                queryParamsCount--;
                if (queryParamsCount > 0) {
                    stringBuilder.append("&");
                }
            }
        }

        return URI.create(stringBuilder.toString());
    }

    public Map<String, JsonNode> getOptions() {
        return options;
    }

    public void setOptions(Map<String, JsonNode> options) {
        this.options = options;
    }
}
