package ru.yandex.partner.test.http.json.model;

import java.util.Objects;
import java.util.Optional;
import java.util.StringJoiner;

import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.annotation.JsonRawValue;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.ObjectWriter;
import com.fasterxml.jackson.databind.annotation.JsonDeserialize;
import org.json.JSONException;
import org.junit.jupiter.api.Assertions;
import org.opentest4j.AssertionFailedError;
import org.skyscreamer.jsonassert.Customization;
import org.skyscreamer.jsonassert.JSONAssert;
import org.skyscreamer.jsonassert.JSONCompareMode;
import org.skyscreamer.jsonassert.ValueMatcher;
import org.skyscreamer.jsonassert.comparator.CustomComparator;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.ResponseEntity;
import org.springframework.util.MimeType;
import org.springframework.web.util.UriComponents;
import org.springframework.web.util.UriComponentsBuilder;

import ru.yandex.partner.test.http.json.JsonObjectDeserializer;
import ru.yandex.partner.test.utils.TestUtils;

public class TestHttpResponse {
    private static final Logger LOGGER = LoggerFactory.getLogger(TestHttpResponse.class);

    private int status;
    @JsonProperty("content_type")
    private String contentType;
    @JsonDeserialize(using = JsonObjectDeserializer.class)
    @JsonRawValue
    private String body;

    public TestHttpResponse() {
        // jsonb reflection
    }

    public TestHttpResponse(ResponseEntity<String> responseEntity) {
        this.status = responseEntity.getStatusCode().value();
        this.contentType = Optional.ofNullable(responseEntity.getHeaders().getContentType())
                .map(MimeType::toString)
                .orElse(null);
        this.body = responseEntity.getBody();
    }

    public int getStatus() {
        return status;
    }

    public void setStatus(int status) {
        this.status = status;
    }

    public String getContentType() {
        return contentType;
    }

    public void setContentType(String contentType) {
        this.contentType = contentType;
    }

    public String getBody() {
        return body;
    }

    public void setBody(String body) {
        this.body = body;
    }

    public void assertEquals(TestHttpResponse actual, String message) {
        ObjectMapper objectMapper = TestUtils.getObjectMapper();
        try {
            LinksValueMatcher linksValueMatcher = new LinksValueMatcher();
            JSONAssert.assertEquals(
                    message,
                    objectMapper.writeValueAsString(this),
                    objectMapper.writeValueAsString(actual),
                    new CustomComparator(JSONCompareMode.NON_EXTENSIBLE,
                            new Customization("body.links.first", linksValueMatcher),
                            new Customization("body.links.prev", linksValueMatcher),
                            new Customization("body.links.self", linksValueMatcher),
                            new Customization("body.links.related", linksValueMatcher),
                            new Customization("body.links.next", linksValueMatcher),
                            new Customization("body.links.last", linksValueMatcher),
                            new Customization("body.data.links.self", linksValueMatcher),
                            new Customization("body.data.links.related", linksValueMatcher),
                            new Customization("body.data.relationships.*.links.self", linksValueMatcher),
                            new Customization("body.data.relationships.*.links.related", linksValueMatcher),
                            new Customization("body.data[*].links.self", linksValueMatcher),
                            new Customization("body.data[*].links.related", linksValueMatcher),
                            new Customization("body.data[*].relationships.*.links.self", linksValueMatcher),
                            new Customization("body.data[*].relationships.*.links.related", linksValueMatcher),
                            new Customization("body.links.add_fields", linksValueMatcher),
                            new Customization("body.included[*].links.self", linksValueMatcher)
                    )
            );
        } catch (JSONException | JsonProcessingException e) {
            Assertions.fail("Error during convert to json. " + message, e);
        } catch (AssertionError e) {
            ObjectMapper mapper = new ObjectMapper();
            ObjectWriter writer = mapper.writerWithDefaultPrettyPrinter();
            try {
                throw new AssertionFailedError(e.getMessage(),
                        writer.writeValueAsString(objectMapper.readValue(body, Object.class)),
                        writer.writeValueAsString(objectMapper.readValue(actual.getBody(), Object.class))
                );
            } catch (JsonProcessingException ex) {
                ex.printStackTrace();
            }
        }
    }

    private static class LinksValueMatcher implements ValueMatcher<Object> {
        @Override
        public boolean equal(Object expected, Object actual) {
            if (!(expected instanceof String) || !(actual instanceof String)) {
                return false;
            }

            try {
                UriComponents expectedUri = UriComponentsBuilder.fromUriString((String) expected).build();
                UriComponents actualUri = UriComponentsBuilder.fromUriString((String) actual).build();

                return Objects.equals(expectedUri.getHost(), actualUri.getHost())
                        && Objects.equals(expectedUri.getPath(), actualUri.getPath())
                        && Objects.equals(expectedUri.getQueryParams(), actualUri.getQueryParams());
            } catch (RuntimeException ex) {
                LOGGER.error("Could not parse some uri in test {} or {}", expected, actual, ex);
                return false;
            }

        }
    }

    private static class OkValueMatcher implements ValueMatcher<Object> {
        @Override
        public boolean equal(Object expected, Object actual) {
            return true;
        }
    }

    @Override
    public String toString() {
        return new StringJoiner(", ", TestHttpResponse.class.getSimpleName() + "[", "]")
                .add("status=" + status)
                .add("contentType='" + contentType + "'")
                .add("body='" + body + "'")
                .toString();
    }
}
