package ru.yandex.partner.test.http.json.utils;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.Iterator;
import java.util.Map;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.util.DefaultIndenter;
import com.fasterxml.jackson.core.util.DefaultPrettyPrinter;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ArrayNode;
import com.fasterxml.jackson.databind.node.ObjectNode;
import com.fasterxml.jackson.databind.node.POJONode;
import org.junit.platform.commons.util.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.ResponseEntity;
import org.springframework.web.reactive.function.client.WebClient;
import org.springframework.web.reactive.function.client.WebClientException;

import ru.yandex.partner.test.http.json.exceptions.JsonUpdateException;
import ru.yandex.partner.test.http.json.model.JsonTestModel;
import ru.yandex.partner.test.http.json.model.TestHttpCase;
import ru.yandex.partner.test.http.json.model.TestHttpExchange;
import ru.yandex.partner.test.http.json.model.TestHttpMeta;
import ru.yandex.partner.test.utils.TestUtils;

public final class TestCaseManager {
    private static final Logger LOGGER = LoggerFactory.getLogger(TestCaseManager.class);
    private static final ObjectMapper OBJECT_MAPPER = TestUtils.getObjectMapper();
    private static final DefaultPrettyPrinter PRINTER = defaultPrettyPrinter();


    private TestCaseManager() {
    }

    public static boolean needGitDownload() {
        return StringUtils.isNotBlank(System.getenv("GIT_DOWNLOAD"));
    }

    public static boolean needGitUpdate() {
        return TestUtils.isTrueEnvironment("GIT_UPDATE");
    }

    public static boolean needSelfUpdate() {
        return TestUtils.needSelfUpdate();
    }

    public static void gitDownload(String localResourceDirPath, String gitDirPath) {
        String gitUrls = System.getenv("GIT_DOWNLOAD");
        for (String gitUrl : gitUrls.split("\\s+")) {
            LOGGER.info("Uploading file from git {}", gitUrl);
            String gitPath =
                    gitUrl.substring(gitUrl.indexOf(gitDirPath) + gitDirPath.length())
                            .toLowerCase().replace("tjson", "json");
            String relativePath = localResourceDirPath + (localResourceDirPath.endsWith("/") ? gitPath : "/" + gitPath);

            String absolutePath = TestUtils.getAbsolutePath(relativePath);
            TestHttpMeta meta = new TestHttpMeta();
            meta.setGitUrl(gitUrl);

            JsonTestModel JsonTestModel = new JsonTestModel();
            JsonTestModel.setMeta(meta);
            JsonTestModel.setData(getGitJsonNode(gitUrl));

            File file = TestUtils.prepareFile(absolutePath);
            try {
                OBJECT_MAPPER.writer(PRINTER).writeValue(file, JsonTestModel);
            } catch (IOException e) {
                throw new JsonUpdateException("Error writing json to file", e);
            }
        }
    }

    /**
     * Обновляет тег data
     *
     * @throws JsonUpdateException possible
     */
    public static void gitUpdate(String resourcePath) {
        LOGGER.info("Updating from git file {}", resourcePath);
        String absolutePath = TestUtils.getAbsolutePath(resourcePath);

        JsonTestModel JsonTestModel = prepareJsonModelUpdate(absolutePath);
        File file = TestUtils.prepareFile(absolutePath);
        try {
            OBJECT_MAPPER.writeValue(file, JsonTestModel);
        } catch (IOException e) {
            throw new JsonUpdateException("Error writing json to file", e);
        }
    }

    /**
     * обновляет тег response
     *
     * @throws JsonUpdateException possible
     */
    public static void selfUpdate(String resourcePath, TestHttpCase testHttpCase) throws IOException {
        LOGGER.info("Self updating file {}", resourcePath);
        File file = TestUtils.prepareFile(TestUtils.getAbsolutePath(resourcePath));

        int index = 0;
        for (TestHttpExchange testHttpExchange : testHttpCase.getTests()) {
            try {
                JsonTestModel jsonTestModel = readFromFile(file.getAbsolutePath());
                ((ObjectNode) jsonTestModel
                        .getData()
                        .get("tests")
                        .get(index++))
                        .set("response", sortJsonNode(OBJECT_MAPPER.valueToTree(testHttpExchange.getResponse())));

                String json = OBJECT_MAPPER.writeValueAsString(jsonTestModel);
                JsonNode jsonNode = OBJECT_MAPPER.readTree(json);
                OBJECT_MAPPER.writer(PRINTER).writeValue(file, jsonNode);
            } catch (IOException e) {
                throw new JsonUpdateException("Error writing json to file", e);
            }
        }

        try (var writer = new FileWriter(file, true)) {
            writer.append("\n");
        }
    }

    public static JsonNode sortJsonNode(JsonNode jsonNode) throws JsonProcessingException {
        if (jsonNode instanceof ObjectNode objectNode) {
            return sortObjectNode(objectNode);
        } else if (jsonNode instanceof ArrayNode arrayNode) {
            return sortArrayNode(arrayNode);
        } else if (jsonNode instanceof POJONode pojoNode) {
            return sortJsonNode(OBJECT_MAPPER.readTree(pojoNode.toString()));
        } else {
            return jsonNode;
        }
    }

    public static ArrayNode sortArrayNode(ArrayNode arrayNode) throws JsonProcessingException {
        if (arrayNode.isEmpty()) {
            return arrayNode;
        }

        if (!arrayNode.get(0).isValueNode()) {
            var newArrayNode = OBJECT_MAPPER.createArrayNode();
            for (JsonNode jsonNode : arrayNode) {
                newArrayNode.add(sortJsonNode(jsonNode));
            }
            return newArrayNode;
        } else if (arrayNode.get(0).isTextual()) {
            var newArrayNode = OBJECT_MAPPER.createArrayNode();
            var list = new ArrayList<JsonNode>( arrayNode.size());
            for (JsonNode jsonNode : arrayNode) {
                list.add(jsonNode);
            }
            list.sort(Comparator.comparing(JsonNode::textValue));
            for (JsonNode jsonNode : list) {
                newArrayNode.add(jsonNode);
            }
            return newArrayNode;
        } else {
            // сюда можно добавлять различные сортировки
            return arrayNode;
        }
    }

    public static ObjectNode sortObjectNode(ObjectNode objectNode) throws JsonProcessingException {
        Iterator<Map.Entry<String, JsonNode>> iter = objectNode.fields();
        var newObjectNode = OBJECT_MAPPER.createObjectNode();
        while (iter.hasNext()) {
            Map.Entry<String, JsonNode> entry = iter.next();
            newObjectNode.set(entry.getKey(), sortJsonNode(entry.getValue()));
        }
        return newObjectNode;
    }


    private static JsonTestModel prepareJsonModelUpdate(String absolutePath) {
        LOGGER.debug("Prepare json from {}", absolutePath);
        JsonTestModel jsonSource = readFromFile(absolutePath);

        JsonTestModel jsonUpdate = new JsonTestModel();
        jsonUpdate.setMeta(jsonSource.getMeta());
        jsonUpdate.setData(getGitJsonNode(jsonSource.getMeta().getGitUrl()));
        return jsonUpdate;
    }

    private static JsonNode getGitJsonNode(String gitUrl) {
        try {
            return OBJECT_MAPPER.readTree(getGitString(gitUrl));
        } catch (IOException e) {
            throw new JsonUpdateException("Error during read json", e);
        }
    }

    private static String getGitString(String gitUrl) {
        ResponseEntity<String> responseEntity;
        try {
            responseEntity = WebClient.create(gitUrl)
                    .get()
                    .retrieve()
                    .toEntity(String.class)
                    .block();
            if (responseEntity == null) {
                throw new JsonUpdateException("Git HTTP response is null");
            }

            if (!responseEntity.getStatusCode().is2xxSuccessful()) {
                throw new JsonUpdateException("Git HTTP wrong status code. StatusCode = " +
                        responseEntity.getStatusCode().value());
            }

        } catch (WebClientException e) {
            throw new JsonUpdateException("Git HTTP error", e);
        }

        String response = responseEntity.getBody();

        if (response == null) {
            throw new JsonUpdateException("Git HTTP response body is null");
        }

        int index = response.indexOf('{');
        if (index < 0) {
            throw new JsonUpdateException("Git HTTP response body does not contain open bracket '{'");
        }
        return response.substring(index);
    }

    private static JsonTestModel readFromFile(String filePath) {
        LOGGER.debug("Reading from file {}", filePath);
        try (InputStream inputStream = new FileInputStream(filePath)) {
            return OBJECT_MAPPER.readValue(inputStream, JsonTestModel.class);
        } catch (IOException e) {
            throw new JsonUpdateException("Error during read from file " + filePath, e);
        }
    }

    private static DefaultPrettyPrinter defaultPrettyPrinter() {
        DefaultPrettyPrinter.Indenter indenter =
                new DefaultIndenter("    ", DefaultIndenter.SYS_LF);
        DefaultPrettyPrinter printer = new DefaultPrettyPrinter();
        printer.indentObjectsWith(indenter);
        printer.indentArraysWith(indenter);
        return printer;
    }
}
