package ru.yandex.partner.test.utils;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.nio.file.FileVisitResult;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.attribute.BasicFileAttributes;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;
import java.util.concurrent.ConcurrentHashMap;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.JavaType;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.MapperFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;
import com.fasterxml.jackson.databind.node.JsonNodeFactory;
import com.fasterxml.jackson.databind.node.ObjectNode;
import com.fasterxml.jackson.databind.type.TypeFactory;
import com.fasterxml.jackson.datatype.jsr310.JavaTimeModule;
import com.fasterxml.jackson.datatype.jsr310.deser.LocalDateDeserializer;
import com.fasterxml.jackson.datatype.jsr310.deser.LocalDateTimeDeserializer;
import com.fasterxml.jackson.datatype.jsr310.ser.LocalDateSerializer;
import com.fasterxml.jackson.datatype.jsr310.ser.LocalDateTimeSerializer;
import org.apache.commons.io.FileUtils;
import org.json.JSONException;
import org.junit.platform.commons.util.StringUtils;
import org.skyscreamer.jsonassert.JSONAssert;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.core.io.ClassPathResource;

public class TestUtils {
    private static final Logger LOGGER = LoggerFactory.getLogger(TestUtils.class);
    private static final Pattern TABLE_NAME_PATTERN = Pattern.compile("(from|insert\\sinto|update)\\s`(\\S+)`");
    private static final Pattern TABLE_NAME_OF_WRITE_QUERY_PATTERN =
            Pattern.compile("(insert\\sinto|update|delete\\sfrom)\\s`(\\S+)`");
    private static final Pattern SELECT_FIELDS_PATTERN = Pattern.compile("select\\n((?:\\s+\\S+,?\\n)+)from");
    private static final Pattern ACTION_LOG_JSON_PATTERN = Pattern.compile("'\\{.+\\}'");
    private static final Pattern INSERT_FIELDS_PATTERN = Pattern.compile("insert\\s+into.+?\\n((?:\\s+\\S+,?\\n)+)\\)" +
            "\\n(values\\s+(?:\\(\\n(?:\\s+.+\\n)+\\)(?:,|;)\\s+)+)");

    private TestUtils() {
        // Utils
    }

    static DateTimeFormatter dateTimeFormatter = DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss");
    static DateTimeFormatter dateFormatter = DateTimeFormatter.ofPattern("yyyy-MM-dd");

    private static final ObjectMapper OBJECT_MAPPER = new ObjectMapper()
            .setNodeFactory(new SortingNodeFactory())
            .registerModule(new JavaTimeModule()
                    .addDeserializer(LocalDateTime.class, new LocalDateTimeDeserializer(dateTimeFormatter))
                    .addSerializer(LocalDateTime.class, new LocalDateTimeSerializer(dateTimeFormatter))
                    .addDeserializer(LocalDate.class, new LocalDateDeserializer(dateFormatter))
                    .addSerializer(LocalDate.class, new LocalDateSerializer(dateFormatter))
            )
            .enable(SerializationFeature.INDENT_OUTPUT)
            .enable(MapperFeature.SORT_PROPERTIES_ALPHABETICALLY);

    private static final Pattern PATTERN = Pattern.compile("/partner/(.+?)/ut/");

    private static final Map<String, String> normalizedSqlCache = new ConcurrentHashMap<>();

    public static boolean needSelfUpdate() {
        return isTrueEnvironment("SELF_UPDATE");
    }

    public static boolean isTrueEnvironment(String environment) {
        String value = System.getenv(environment);
        return "1".equals(value) || "true".equalsIgnoreCase(value);
    }

    public static ObjectMapper getObjectMapper() {
        return OBJECT_MAPPER;
    }

    public static String getTestDataAndUpdateIfNeeded(String fileName, Object dataForSelfUpdate, Class<?> clazz) {
        String resourceDirPath = clazz.getName().toLowerCase().replace('.', '/') + '/';
        final String resourcePath = resourceDirPath.concat(fileName);

        return getTestDataAndUpdateIfNeeded(dataForSelfUpdate, resourcePath);
    }

    public static String getTestDataAndUpdateIfNeeded(Object dataForSelfUpdate, String resourcePath) {
        if (needSelfUpdate()) {
            doSelfUpdate(dataForSelfUpdate, getAbsolutePath(resourcePath));
        }

        LOGGER.debug("filePath = {}", resourcePath);

        try {
            return readDataFromFile(Paths.get(getAbsolutePath(resourcePath)));
        } catch (IOException e) {
            throw new SelfUpdateException("Error during read file", e);
        }
    }

    public static List<String> getSqlTestData(String resourcePath) {
        var directory = Paths.get(getAbsolutePath(resourcePath));
        try {
            return readDataFromDirectory(directory);
        } catch (IOException e) {
            throw new RuntimeException("Error during read files from directory", e);
        }
    }


    private static List<String> readDataFromDirectory(Path directory) throws IOException {
        var result = new ArrayList<String>();
        if (Files.exists(directory)) {
            Files.walkFileTree(directory, new SimpleFileVisitor<>() {
                @Override
                public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                    result.add(readDataFromFile(file));
                    return FileVisitResult.CONTINUE;
                }
            });
        }
        return result;
    }

    private static String readDataFromFile(Path filePath) throws IOException {
        StringBuilder stringBuilder = new StringBuilder();
        try (BufferedReader bufferedReader = Files.newBufferedReader(filePath)) {
            String input;
            while ((input = bufferedReader.readLine()) != null) {
                stringBuilder.append(input).append("\n");
            }
            return stringBuilder.toString();
        }
    }

    private static void doSelfUpdate(Object dataForSelfUpdate, String absolutePath) {
        LOGGER.debug("trying self update");
        File file = prepareFile(absolutePath);
        updateFile(dataForSelfUpdate, file);
        LOGGER.debug("self update complete");
    }

    public static void doSQLSelfUpdate(List<String> sqlList, String resourcePath) {
        Path directory = Paths.get(TestUtils.getAbsolutePath(resourcePath));
        try {
            Map<String, String> queryMap = createQueryMap(sqlList);
            createDirectories(directory.toString().concat("/"));
            Files.walkFileTree(directory, new SimpleFileVisitor<>() {
                @Override
                public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                    if (!Files.isDirectory(file)) {
                        var fileName = file.getFileName().toString();
                        fileName = fileName.substring(0, fileName.lastIndexOf("."));
                        if (queryMap.containsKey(fileName)) {
                            updateFile(prettifyQueryBeforeWrite(queryMap.get(fileName), fileName), file.toFile());
                            queryMap.remove(fileName);
                        } else {
                            Files.delete(file);
                        }
                    }
                    return FileVisitResult.CONTINUE;
                }
            });

            for (Map.Entry<String, String> pair : queryMap.entrySet()) {
                doSelfUpdate(
                        prettifyQueryBeforeWrite(pair.getValue(), pair.getKey()),
                        directory.toString().concat("/").concat(pair.getKey()).concat(".sql")
                );
            }

        } catch (IOException e) {
            throw new SelfUpdateException("Error during self update", e);
        }
    }

    /**
     * Вернет bar_table_2 -> query
     * foo_table_1 -> query
     * bar_table_3 -> query
     * ...
     * bar_table_2 -> query
     * foo_table_2 -> query
     * ...
     * Порядок нумерации ключей соответствует хронологическому порядку запросов к таблице
     */
    private static Map<String, String> createQueryMap(List<String> sqlQueries) {
        Map<String, List<String>> tableQueries = new LinkedHashMap<>(sqlQueries.size());
        for (String sqlQuery : sqlQueries) {
            String tableName = extractTableName(sqlQuery);
            tableQueries.computeIfAbsent(tableName, k -> new ArrayList<>()).add(sqlQuery);
        }
        return tableQueries.entrySet()
                .stream()
                .flatMap(e -> {
                    List<String> queriesGroup = e.getValue();
                    return IntStream.range(0, queriesGroup.size())
                            .mapToObj(index -> {
                                String tableName = e.getKey();
                                String sqlQuery = queriesGroup.get(index);
                                return Map.entry(enumeratedTableName(tableName, index, queriesGroup.size()), sqlQuery);
                            });
                })
                .collect(Collectors.toMap(Map.Entry::getKey, Map.Entry::getValue));
        // Дальше порядок не гарантирован, т.к. toMap() вернет HashMap
    }

    private static String enumeratedTableName(String tableName, int index, int size) {
        if (size == 1) {
            return tableName;
        } else {
            return tableName + "_" + (index + 1);
        }
    }

    public static List<String> normalizeQueries(List<String> sqlQueries) {
        return sqlQueries.stream()
                .map(q -> prettifyQueryBeforeWrite(q, extractTableName(q)))
                .sorted()
                .collect(Collectors.toList());
    }

    public static boolean isReadOnlyQuery(String sqlQuery) {
        return !TABLE_NAME_OF_WRITE_QUERY_PATTERN.matcher(sqlQuery)
                .find();
    }

    public static String extractTableName(String sqlQuery) {
        var matcher = TABLE_NAME_PATTERN.matcher(sqlQuery);
        if (matcher.find()) {
            return matcher.group(2);
        }
        throw new IllegalStateException("Can't extract table name from sql query string!");
    }

    /**
     * Сортируем поля, чтобы тесты не флапали. Но нужно учитывать,
     * что сбивается порядковое соответствие запрошенных полей с их значениями
     */
    private static String getSqlWithSortedFields(String sqlQuery) {
        var selectMatcher = SELECT_FIELDS_PATTERN.matcher(sqlQuery);
        if (selectMatcher.find()) {
            String fields = selectMatcher.group(1);
            String sortedFields = Arrays.stream(fields.split("\n"))
                    .map(line -> !line.endsWith(",") ? line + "," : line)
                    .sorted()
                    .collect(Collectors.joining("\n"));

            sortedFields = sortedFields.substring(0, sortedFields.length() - 1) + "\n";

            return sqlQuery.replace(fields, sortedFields);
        }

        return sortFieldsForInsert(sqlQuery);
    }

    private static String sortFieldsForInsert(String sqlQuery) {
        var insertMatcher = INSERT_FIELDS_PATTERN.matcher(sqlQuery);
        if (insertMatcher.find()) {
            //field -> old index
            Map<String, Integer> fieldsIndexes = new TreeMap<>();
            //old index -> new index
            Map<Integer, Integer> valuesIndexes = new HashMap<>();

            //replace fields
            String queryFields = insertMatcher.group(1);

            List<String> fields = Arrays.stream(queryFields.split("\n"))
                    .map(line -> !line.endsWith(",") ? line + "," : line)
                    .toList();

            int fieldsCount = fields.size();

            int index = 0;
            for (String field : fields) {
                fieldsIndexes.put(field, index++);
            }

            index = 0;
            for (String field : fieldsIndexes.keySet()) {
                valuesIndexes.put(fieldsIndexes.get(field), index++);
            }

            String sortedFieldsStr = String.join("\n", fieldsIndexes.keySet());

            sortedFieldsStr = sortedFieldsStr.substring(0, sortedFieldsStr.length() - 1) + "\n";

            sqlQuery = sqlQuery.replace(queryFields, sortedFieldsStr);

            //replace values
            String queryValues = insertMatcher.group(2);

            List<String> parts = Arrays.stream(queryValues.split(",?\s*\n")).toList();
            //parts = [
            //    "values (",
            //    "  1",
            //    "  \"a\"",
            //    "), (",
            //    "  2",
            //    "  \"b\"",
            //    ");"
            //];

            StringBuilder sortedValues = new StringBuilder();
            sortedValues.append(parts.get(0)).append("\n");

            index = 1;
            while (index < parts.size()) {
                String[] rec = new String[fieldsCount];
                for (int i = 0; i < fieldsCount; i++) {
                    rec[valuesIndexes.get(i)] = parts.get(index++);
                }

                sortedValues.append(String.join(",\n", rec)).append("\n").append(parts.get(index++)).append("\n");
            }

            return sqlQuery.replace(queryValues, sortedValues.toString());
        }

        return sqlQuery;
    }

    private static String prettifyQueryBeforeWrite(String sqlQuery, String tableName) {
        return normalizedSqlCache.computeIfAbsent(sqlQuery, q -> {
            String result = q;
            result = replaceActionLogOpts(result, tableName);
            result = getSqlWithSortedFields(result);
            return result;
        });
    }

    private static String replaceActionLogOpts(String sqlQuery, String tableName) {
        if (tableName.contains("action_log") && sqlQuery.contains("insert")) {
            sqlQuery = sqlQuery.replaceAll(ACTION_LOG_JSON_PATTERN.pattern(), "'{\"sql_tests\":\"some_opts\"}'");
        }
        return sqlQuery;
    }

    private static void updateFile(Object dataForSelfUpdate, File file) {
        try {
            if (dataForSelfUpdate instanceof String) {
                try (FileWriter out = new FileWriter(file)) {
                    out.write((String) dataForSelfUpdate);
                }
            } else {
                OBJECT_MAPPER.writeValue(file, dataForSelfUpdate);
            }
        } catch (IOException e) {
            throw new SelfUpdateException("Error during write data", e);
        }
    }

    public static void compareToDataFromFile(Object data, Class<?> clazz, String fileName) throws Exception {
        LOGGER.debug("data = {}", data);
        LOGGER.debug("fileName = {}", fileName);

        String json = data instanceof String ? (String) data : OBJECT_MAPPER.writeValueAsString(data);
        String expectedJson = getTestDataAndUpdateIfNeeded(fileName, json, clazz);

        LOGGER.debug("compareToDataFromFile expectedJson = {}", expectedJson);
        LOGGER.debug("compareToDataFromFile json = {}", json);

        JSONAssert.assertEquals(expectedJson, json, true);
    }

    public static void compareToDataFromFile(Object data, String filePath) throws IOException, JSONException {
        String json = data instanceof String ? (String) data : OBJECT_MAPPER.writeValueAsString(data);
        String expectedJson = FileUtils.readFileToString(new ClassPathResource(filePath).getFile());
        LOGGER.debug("compareToDataFromFile json = \n{}", json);
        LOGGER.debug("compareToDataFromFile expectedJson = \n{}", expectedJson);
        JSONAssert.assertEquals(expectedJson, json, true);
    }

    public static String getAbsolutePath(String resourcePath) {
        String arcadiaPath = getArcadiaPathByYaMake();
        if (arcadiaPath == null) {
            arcadiaPath = getArcadiaPathByIde();
            if (arcadiaPath == null) {
                throw new ResourceException("Can't find environment variables [ORIGINAL_SOURCE_ROOT and TEST_WORK_PATH]"
                        + " or property [user.dir] ");
            }
        }

        return Paths.get(arcadiaPath, resourcePath).toString();
    }

    /**
     * Создает директории и файлы (если они отсутствовали) по полученным путям
     *
     * @param absolutePath абсолютный путь до файлов
     * @return файл
     */
    public static File prepareFile(String absolutePath) {
        LOGGER.debug("Preparing file {}", absolutePath);
        createDirectories(absolutePath);

        File file = new File(absolutePath);
        if (!file.exists()) {
            try {
                boolean isCreated = file.createNewFile();

                if (isCreated && LOGGER.isDebugEnabled()) {
                    LOGGER.debug("File has been created: {}", absolutePath);
                }
            } catch (IOException e) {
                throw new ResourceException("Error during create file " + absolutePath);
            }
        }

        return file;
    }

    public static File prepareJsonFile(String absPath) {
        var file = prepareFile(absPath);
        try {
            return file.length() != 0 ? file : Files.write(file.toPath(), List.of("{}")).toFile();
        } catch (IOException e) {
            throw new RuntimeException("Can't write to file", e);
        }
    }

    private static void createDirectories(String absolutePath) {
        File dir = new File(absolutePath.substring(0, absolutePath.lastIndexOf("/")));
        boolean isMkdir = dir.mkdirs();
        if (isMkdir && LOGGER.isDebugEnabled()) {
            LOGGER.debug("Directories have been created: {}", absolutePath);
        }
    }

    private static String getArcadiaPathByYaMake() {
        String originalSourceRoot = ru.yandex.devtools.test.Paths.getSourcePath();
        String testWorkPath = ru.yandex.devtools.test.Paths.getWorkPath();

        if (StringUtils.isBlank(originalSourceRoot) || StringUtils.isBlank(testWorkPath)) {
            return null;
        }

        LOGGER.info("TEST_WORK_PATH: " + testWorkPath);
        String dirPath = originalSourceRoot + "/partner/";
        Matcher matcher = PATTERN.matcher(testWorkPath);
        if (matcher.find()) {
            dirPath += matcher.group(1);
        } else {
            throw new ResourceException("Can't find path ORIGINAL_SOURCE_ROOT = " + originalSourceRoot
                    + "; TEST_WORK_PATH = " + testWorkPath);
        }

        return dirPath + "/src/test/resources";
    }

    private static String getArcadiaPathByIde() {
        String ideaDir = System.getProperty("user.dir");
        if (StringUtils.isBlank(ideaDir)) {
            return null;
        }
        String relativePathToResource = "/src/test/resources";
        if (ideaDir.endsWith("/")) {
            relativePathToResource = relativePathToResource.substring(1);
        }

        return ideaDir + relativePathToResource;
    }

    public static <T> T parseOption(Map<String, JsonNode> options, String name, Class<T> tClass) {
        return parseOption(options, name, TypeFactory.defaultInstance().constructType(tClass));
    }

    public static <T> T parseOption(Map<String, JsonNode> options, String name, JavaType type) {
        if (options == null || !options.containsKey(name)) {
            return null;
        }
        return parseNode(options.get(name), name, type);
    }

    public static <T> T parseNode(JsonNode node, String name, Class<T> tClass) {
        try {
            return TestUtils.getObjectMapper().treeToValue(node, tClass);
        } catch (JsonProcessingException e) {
            throw new IllegalStateException("Error during parse option " + name, e);
        }
    }

    public static <T> T parseNode(JsonNode node, String name, JavaType type) {
        try {
            return TestUtils.getObjectMapper().readValue(
                    TestUtils.getObjectMapper().treeAsTokens(node),
                    type
            );
        } catch (IOException e) {
            throw new IllegalStateException("Error during parse option " + name, e);
        }
    }

    /**
     * Сортирует по ключам
     */
    private static class SortingNodeFactory extends JsonNodeFactory {
        @Override
        public ObjectNode objectNode() {
            return new ObjectNode(this, new TreeMap<>());
        }
    }
}
