package ru.yandex.partner.unifiedagent;

import java.time.Duration;
import java.util.Collections;
import java.util.Map;
import java.util.concurrent.ExecutionException;
import java.util.stream.Collectors;

import javax.annotation.PreDestroy;

import com.google.common.collect.ImmutableMap;
import com.google.common.net.HostAndPort;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.logbroker.agent.client.Client;
import ru.yandex.logbroker.agent.client.LogMessage;
import ru.yandex.logbroker.agent.client.Session;
import ru.yandex.monlib.metrics.primitives.Rate;
import ru.yandex.monlib.metrics.registry.MetricRegistry;

import static ru.yandex.logbroker.agent.client.SessionMetrics.ACKNOWLEDGED_BYTES;
import static ru.yandex.logbroker.agent.client.SessionMetrics.ACKNOWLEDGED_MESSAGES;
import static ru.yandex.logbroker.agent.client.SessionMetrics.DROPPED_BYTES;
import static ru.yandex.logbroker.agent.client.SessionMetrics.DROPPED_MESSAGES;
import static ru.yandex.logbroker.agent.client.SessionMetrics.ERRORS_COUNT;
import static ru.yandex.logbroker.agent.client.SessionMetrics.GRPC_CALLS;
import static ru.yandex.logbroker.agent.client.SessionMetrics.GRPC_CALLS_INITIALIZED;
import static ru.yandex.logbroker.agent.client.SessionMetrics.GRPC_INFLIGHT_BYTES;
import static ru.yandex.logbroker.agent.client.SessionMetrics.GRPC_INFLIGHT_MESSAGES;
import static ru.yandex.logbroker.agent.client.SessionMetrics.GRPC_WRITE_BATCH_REQUESTS;
import static ru.yandex.logbroker.agent.client.SessionMetrics.INFLIGHT_BYTES;
import static ru.yandex.logbroker.agent.client.SessionMetrics.INFLIGHT_MESSAGES;
import static ru.yandex.logbroker.agent.client.SessionMetrics.RECEIVED_BYTES;
import static ru.yandex.logbroker.agent.client.SessionMetrics.RECEIVED_MESSAGES;

public class SimpleUnifiedAgentService implements UnifiedAgentService {
    private static final Logger LOGGER = LoggerFactory.getLogger(SimpleUnifiedAgentService.class);
    private final Client client;
    private final MetricRegistry metricRegistry;
    private final int segmentSize;
    private final Duration minAckDelay;
    private final Rate droppedMessages;
    private final LogbrokerDtoValidator validator;
    private Session session;

    public SimpleUnifiedAgentService(
            UnifiedAgentProperties unifiedAgentProperties,
            MetricRegistry metricRegistry
    ) {
        LOGGER.info("Start with properties: {}", unifiedAgentProperties);

        //noinspection UnstableApiUsage
        HostAndPort unifiedAgentConnection = HostAndPort.fromParts(
                unifiedAgentProperties.getHost(),
                unifiedAgentProperties.getPort()
        );

        //noinspection UnstableApiUsage
        this.client = Client.newClient(unifiedAgentConnection)
                .setMaxInflightBytes(unifiedAgentProperties.getMaxInflightBytes())
                .setGrpcMaxMessageSize(unifiedAgentProperties.getMaxMessageSizeBytes())
                .setCloseTimeout(Duration.ofMillis(unifiedAgentProperties.getCloseTimeoutMillis()))
                .setGrcpReconnectDelay(Duration.ofMillis(unifiedAgentProperties.getReconnectDelayMillis()))
                .setMetricRegistry(metricRegistry)
                .build();

        this.segmentSize = unifiedAgentProperties.getSegmentSize();
        this.minAckDelay = Duration.ofMillis(unifiedAgentProperties.getMinAckDelayMillis());

        this.session = createSession();

        // при пересоздании сессии метрика остается та же
        this.metricRegistry = this.client.getClientParameters().getMetricRegistry().subRegistry("session", "default");

        this.droppedMessages = metricRegistry.rate(DROPPED_MESSAGES);
        this.validator = new LogbrokerDtoValidator(
                Collections.unmodifiableSet(unifiedAgentProperties.getTopics()),
                unifiedAgentProperties.getMaxMessageSizeBytes()
        );
    }

    @Override
    public UnifiedAgentResult sendToLogbroker(LogbrokerDto logbrokerDto) {
        String errorDetails = validator.validate(logbrokerDto);
        if (errorDetails != null) {
            return UnifiedAgentResult.validationError(errorDetails);
        }

        Map<String, String> meta = ImmutableMap.of("topic", logbrokerDto.getTopic());
        long millis = System.currentTimeMillis();
        getSession().send(
                logbrokerDto.getMessages()
                        .stream()
                        .map(msg -> new LogMessage(msg, millis, meta))
                        .collect(Collectors.toList())
        );

        logMetrics();
        return UnifiedAgentResult.ok();
    }

    private Session getSession() {
        long droppedMessageCount = droppedMessages.get();
        if (droppedMessageCount > 0L) {
            LOGGER.warn("Found dropped message. Count {}", droppedMessageCount);
            // TODO try to find dropped message and resend
            session = createSession();
        }

        return session;
    }

    private Session createSession() {
        return client.newSession()
                .enableSentMessageIndex(segmentSize, minAckDelay)
                .build();
    }

    private void logMetrics() {
        if (LOGGER.isDebugEnabled()) {
            String metrics =
                    "metricRegistry.counter(RECEIVED_MESSAGES) = "
                            + metricRegistry.rate(RECEIVED_MESSAGES).get() + "\n" +
                            "metricRegistry.rate(RECEIVED_BYTES) = "
                            + metricRegistry.rate(RECEIVED_BYTES).get() + "\n" +
                            "metricRegistry.rate(ACKNOWLEDGED_MESSAGES) = "
                            + metricRegistry.rate(ACKNOWLEDGED_MESSAGES).get() + "\n" +
                            "metricRegistry.rate(ACKNOWLEDGED_BYTES) = "
                            + metricRegistry.rate(ACKNOWLEDGED_BYTES).get() + "\n" +
                            "metricRegistry.gaugeInt64(INFLIGHT_MESSAGES) = "
                            + metricRegistry.gaugeInt64(INFLIGHT_MESSAGES).get() + "\n" +
                            "metricRegistry.gaugeInt64(INFLIGHT_BYTES) = "
                            + metricRegistry.gaugeInt64(INFLIGHT_BYTES).get() + "\n" +
                            "metricRegistry.rate(GRPC_WRITE_BATCH_REQUESTS) = "
                            + metricRegistry.rate(GRPC_WRITE_BATCH_REQUESTS).get() + "\n" +
                            "metricRegistry.gaugeInt64(GRPC_INFLIGHT_MESSAGES) = "
                            + metricRegistry.gaugeInt64(GRPC_INFLIGHT_MESSAGES).get() + "\n" +
                            "metricRegistry.gaugeInt64(GRPC_INFLIGHT_BYTES) = "
                            + metricRegistry.gaugeInt64(GRPC_INFLIGHT_BYTES).get() + "\n" +
                            "metricRegistry.rate(GRPC_CALLS) = "
                            + metricRegistry.rate(GRPC_CALLS).get() + "\n" +
                            "metricRegistry.rate(GRPC_CALLS_INITIALIZED) = "
                            + metricRegistry.rate(GRPC_CALLS_INITIALIZED).get() + "\n" +
                            "metricRegistry.rate(DROPPED_MESSAGES) = "
                            + metricRegistry.rate(DROPPED_MESSAGES).get() + "\n" +
                            "metricRegistry.rate(DROPPED_BYTES) = "
                            + metricRegistry.rate(DROPPED_BYTES).get() + "\n" +
                            "metricRegistry.rate(ERRORS_COUNT) = "
                            + metricRegistry.rate(ERRORS_COUNT).get() + "\n";
            LOGGER.debug(metrics);
        }
    }

    @PreDestroy
    public void preDestroy() {
        if (session != null) {
            try {
                session.close();
            } catch (ExecutionException ignored) {
                LOGGER.error("ExecutionException during session close");
            } catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                LOGGER.error("InterruptedException during session close");
                throw new RuntimeException("Interrupted", e);
            } finally {
                if (client != null) {
                    client.close();
                }
            }
        }
    }
}
