package ru.yandex.partner.libs.utils;

import java.io.IOException;
import java.util.Set;

import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.core.JsonToken;
import com.fasterxml.jackson.databind.DeserializationContext;
import com.fasterxml.jackson.databind.deser.std.StdDeserializer;

public class BooleanCompatDeserializer extends StdDeserializer<Boolean> {
    public static final Set<String> VALID_BOOLEAN_STRINGS =
            Set.of(JsonToken.VALUE_FALSE.asString(), JsonToken.VALUE_TRUE.asString());

    public BooleanCompatDeserializer() {
        super(Boolean.class);
    }

    @Override
    public Boolean deserialize(JsonParser p, DeserializationContext ctxt) throws IOException {
        JsonToken token = p.currentToken();
        if (token == JsonToken.VALUE_FALSE) {
            return Boolean.FALSE;
        }

        if (token == JsonToken.VALUE_TRUE) {
            return Boolean.TRUE;
        }

        if (token.isNumeric()) {
            return p.readValueAs(Integer.class) != 0;
        }

        if (token == JsonToken.VALUE_STRING) {
            var str = p.readValueAs(String.class);

            if (str.isEmpty()) {
                return (Boolean) ctxt.handleWeirdStringValue(Boolean.class, str,
                        "Empty string can't be parsed as boolean");
            }

            if (Character.isAlphabetic(str.charAt(0))) {
                if (VALID_BOOLEAN_STRINGS.contains(str)) {
                    return Boolean.parseBoolean(str);
                } else {
                    return (Boolean) ctxt.handleWeirdStringValue(Boolean.class, str,
                            "Invalid alphabetic string, must contain true|false");
                }
            } else {
                try {
                    return Integer.parseInt(str) != 0;
                } catch (RuntimeException e) {
                    return (Boolean) ctxt.handleWeirdStringValue(Boolean.class, str, "Invalid boolean string");
                }
            }
        }

        return (Boolean) ctxt.handleUnexpectedToken(getValueType(), token, p,
                "Unexpected token type while parsing boolean");
    }
}
