package ru.yandex.partner.libs.utils;

import java.io.IOException;
import java.io.InputStream;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.function.BiConsumer;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.JavaType;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.jooq.DSLContext;
import org.jooq.Field;
import org.jooq.Record;
import org.jooq.Result;
import org.jooq.SelectFieldOrAsterisk;
import org.jooq.Table;
import org.jooq.TableRecord;

public class JooqUtils {
    private static final ObjectMapper OBJECT_MAPPER = new ObjectMapper();
    private static final JavaType MAP_JAVA_TYPE =
            OBJECT_MAPPER.getTypeFactory().constructMapType(Map.class, String.class, Object.class);
    private static final JavaType LIST_OF_MAP_JAVA_TYPE =
            OBJECT_MAPPER.getTypeFactory().constructCollectionType(List.class, MAP_JAVA_TYPE);

    private JooqUtils() {
        // utils
    }

    public static <R extends Record> void insertRecords(DSLContext dslContext, Table<R> table, List<R> records) {
        if (records != null && !records.isEmpty()) {
            dslContext.insertInto(table, table.fields()).valuesOfRecords(records).execute();
        }
    }

    public static <R extends Record> Result<R> insertRecordsWithReturning(DSLContext dslContext,
                                                                          Table<R> table,
                                                                          List<R> records,
                                                                          SelectFieldOrAsterisk... fields) {
        if (records != null && !records.isEmpty()) {
            return dslContext.insertInto(table, table.fields()).valuesOfRecords(records).returning(fields).fetch();
        }
        throw new RuntimeException("Wrong records to insert!");
    }

    public static void applyPatchToRecord(TableRecord<?> record,
                                          Map<String, Field<?>> fieldTypes,
                                          Map<String, Object> patch) {
        for (var entry : patch.entrySet()) {
            if (fieldTypes.containsKey(entry.getKey())) {
                Field field = fieldTypes.get(entry.getKey());
                if (entry.getValue() == null) {
                    record.set(field, null);
                } else {
                    Class fieldType = field.getDataType().getType();
                    if (Enum.class.isAssignableFrom(fieldType)) {
                        record.set(field, (Enum.valueOf(fieldType, (String) entry.getValue())));
                    } else {
                        try {
                            record.set(field, fieldType.cast(entry.getValue()));
                        } catch (ClassCastException e) {
                            ObjectMapper objectMapper = new ObjectMapper();
                            try {
                                record.set(field,
                                        objectMapper.treeToValue(objectMapper.valueToTree(entry.getValue()), fieldType)
                                );
                            } catch (JsonProcessingException ex) {
                                throw e;
                            }
                        }
                    }
                }
            }
        }
    }

    public static void applyPatchToRecord(TableRecord<?> record,
                                          Map<String, Object> patch) {
        Map<String, Field<?>> fields = Arrays.stream(record.getTable().fields())
                .collect(Collectors.toMap(Field::getName, f -> f));

        applyPatchToRecord(record, fields, patch);
    }


    public static <TR extends TableRecord<TR>> List<TR> jsonToRecords(String pathToJsonResource, Supplier<TR> creator) {
        try (InputStream inputStream = JooqUtils.class.getResourceAsStream(pathToJsonResource)) {

            JsonNode jsonNode = OBJECT_MAPPER.readTree(inputStream);
            List<Map<String, Object>> list;
            if (jsonNode.isArray()) {
                list = OBJECT_MAPPER.readValue(
                        OBJECT_MAPPER.treeAsTokens(jsonNode),
                        LIST_OF_MAP_JAVA_TYPE
                );
            } else {
                Map<String, Object> map =
                        OBJECT_MAPPER.readValue(
                                OBJECT_MAPPER.treeAsTokens(jsonNode),
                                MAP_JAVA_TYPE
                        );

                list = List.of(map);
            }

            Map<String, BiConsumer<TR, Object>> fieldMap = Stream.of(creator.get().fields())
                    .collect(Collectors.toMap(Field::getName,
                            field -> (dspRecord, o) -> dspRecord.set((Field) field, o)));

            return list.stream()
                    .map(
                            dataMap -> {
                                TR tableRecord = creator.get();
                                for (Map.Entry<String, Object> entry : dataMap.entrySet()) {
                                    var biConsumer = fieldMap.get(entry.getKey());
                                    if (biConsumer == null) {
                                        throw new IllegalStateException("Unknown field. Field = " + entry.getKey() +
                                                " Table = " + tableRecord.getTable().getName());
                                    }
                                    biConsumer.accept(tableRecord, entry.getValue());
                                }
                                return tableRecord;
                            }
                    ).collect(Collectors.toList());
        } catch (IOException e) {
            throw new IllegalStateException("Error while reading " + pathToJsonResource, e);
        }
    }
}
