package ru.yandex.partner.libs.utils;

import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.function.BiConsumer;
import java.util.function.BiFunction;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import java.util.stream.Stream;

import static java.util.Objects.requireNonNull;

public class StreamUtils {

    private StreamUtils() {
    }

    /**
     * Создаёт поток объектов, полученных применением переданной функции к каждой паре
     * соответствующих элементов двух массивов
     * Если массивы разной длины, то кратчайший дополняется null-ами
     * Если какой-либо из массивов null, будет NullPointerException
     *
     * @param l1
     * @param l2
     * @param mergeFunction
     * @param <A>
     * @param <B>
     * @param <C>
     * @return
     */
    public static <A, B, C> Stream<C> mergedStream(List<? extends A> l1, List<? extends B> l2,
                                                   BiFunction<A, B, C> mergeFunction) {
        requireNonNull(l1);
        requireNonNull(l2);
        requireNonNull(mergeFunction);
        return IntStream.range(0, getMaxSize(l1, l2))
                .mapToObj(i -> mergeFunction.apply(getElement(l1, i), getElement(l2, i)));
    }

    /**
     * Производит переданное действие над каждой парой соответствующих элементов из двух массивов
     * Если массивы разной длины, то кратчайший дополняется null-ами
     * Если какой-либо из массивов null, будет NullPointerException
     *
     * @param l1
     * @param l2
     * @param consumer
     * @param <A>
     * @param <B>
     */
    public static <A, B> void forEachPair(List<? extends A> l1, List<? extends B> l2,
                                          BiConsumer<A, B> consumer) {
        requireNonNull(l1);
        requireNonNull(l2);
        requireNonNull(consumer);
        IntStream.range(0, getMaxSize(l1, l2))
                .forEach(i -> consumer.accept(getElement(l1, i), getElement(l2, i)));
    }

    public static <T> List<T> reduceLists(List<T> base, Collection<List<T>> lists,
                                          BiFunction<T, T, T> elementFunction) {
        return lists.stream().reduce(base,
                (l1, l2) -> StreamUtils.mergedStream(l1, l2, elementFunction).collect(Collectors.toList())
        );
    }

    private static <A, B> int getMaxSize(List<? extends A> l1, List<? extends B> l2) {
        return Math.max(l1.size(), l2.size());
    }

    private static <T> T getElement(List<? extends T> list, int index) {
        return index < list.size() ? list.get(index) : null;
    }

}
