package ru.yandex.partner.scripts.bkdata

import ch.qos.logback.classic.Level
import com.fasterxml.jackson.databind.ObjectMapper
import com.fasterxml.jackson.databind.node.ArrayNode
import org.apache.commons.cli.CommandLine
import org.apache.commons.cli.Options
import org.json.JSONObject
import org.slf4j.Logger
import org.slf4j.LoggerFactory
import org.springframework.core.io.buffer.DataBufferUtils
import org.springframework.http.HttpHeaders
import org.springframework.http.HttpMethod
import org.springframework.web.reactive.function.BodyExtractors
import org.springframework.web.reactive.function.client.WebClient
import org.springframework.web.util.UriComponentsBuilder
import reactor.core.publisher.toFlux
import reactor.util.retry.Retry
import ru.yandex.partner.libs.bs.json.PageBkDataComparator
import ru.yandex.partner.libs.cli.CliApp
import java.io.File
import java.nio.file.Files
import java.nio.file.Path
import java.time.Duration
import java.util.concurrent.locks.ReentrantLock
import kotlin.concurrent.withLock

/**
 * ## Примеры:
 * ### Продолжить валидацию с дефолтным фильтром (не в архиве) с указанной страницы
 * ```
 * ./enrich-check.sh \
 * -H "$(cat cookies)" -H "Accept: application/vnd.api+json" \
 * -H "Host: partner2-test.yandex.ru" \
 * --ref https://dev3.partner.yandex.ru:8624/restapi \
 * --test http://localhost:8080 \
 * -p 1704
 * ```
 *
 * ### Валидировать страницы по списку идентификаторов пейджей
 * ```
 * ./enrich-check.sh \
 * -H "$(cat cookies)" -H "Accept: application/vnd.api+json" \
 * -H "Host: partner2-test.yandex.ru" \
 * --ref https://dev3.partner.yandex.ru:8624/restapi \
 * --test http://localhost:8080 \
 * -pageId 1704,1820,17500
 * ```
 * ### Валидировать отдельный тип модели
 * ```
 * ./enrich-check.sh \
 * -H "$(cat cookies)" -H "Accept: application/vnd.api+json" \
 * -H "Host: partner2-test.yandex.ru" \
 * -H "X-Ya-Service-Ticket: $(cat .tvm_ticket)"
 * --ref https://dev3.partner.yandex.ru:8624/restapi \
 * --test http://localhost:8080 \
 * --model=internal_context_on_site_campaign --pageId 238158,265882
 * ```
 */
class BkDataEnrichCompareApp : CliApp() {
    companion object {
        private val LOGGER: Logger = LoggerFactory.getLogger(BkDataEnrichCompareApp::class.java)
        private val OPTIONS = Options()
        private val REF_API = OPTIONS.addOption("ref", "ref", true, "Reference api")
            .getOption("ref")
        private val TEST_API = OPTIONS.addOption("test", "test", true, "Test api")
            .getOption("test")
        private val MODEL = OPTIONS.addOption(
            "m", "model", true,
            "Page model (defaults to context_on_site_campaign)"
        )
            .getOption("m")
        private val FILTER = OPTIONS.addOption("f", "filter", true, "Filter")
            .getOption("filter")
        private val HEADER = OPTIONS.addOption("H", "header", true, "Header")
            .getOption("H")
        private val FROM_PAGE = OPTIONS.addOption(
            "p", "page", true,
            "Pagination starting page for current filter"
        )
            .getOption("p")
        private val PAGE_ID = OPTIONS.addOption(
            "pageId", "pageId", true,
            "Overrides page filter with exact page ids, comma separated"
        )
            .getOption("pageId")

        private val LOGGING_LEVEL = OPTIONS.addOption(
            "ll", "loggingLevel", true,
            "Specifies logging level for the script class"
        ).getOption("ll")

        init {
            // INFO уровень на root, чтобы не мешал лог http клиента
            val root = LoggerFactory.getLogger(Logger.ROOT_LOGGER_NAME) as ch.qos.logback.classic.Logger
            root.level = Level.INFO
        }

        @JvmStatic
        fun main(args: Array<String>) {
            BkDataEnrichCompareApp().run(args)
        }
    }

    override fun getOptions() = OPTIONS

    override fun configure(cmd: CommandLine) {
        refApi = cmd.getOptionValue(REF_API.opt)
        require(refApi.isNotEmpty()) { "ref" }
        testApi = cmd.getOptionValue(TEST_API.opt)
        require(testApi.isNotEmpty()) { "test" }
        filter = cmd.getOptionValue(FILTER.opt) ?: filter

        filter = cmd.getOptionValue(PAGE_ID.opt)?.let { "{\"page_id\":[\"IN\", [${it}]]}" }
            ?: filter

        headers = cmd.getOptionValues(HEADER.opt)
            // if somehow multiline - multiple headers
            ?.flatMap { it.split("\n") }
            ?.filter { it.isNotEmpty() }
            ?.associate {
                val (name, value) = it.split(":", limit = 2)
                name.trim() to value.trim()
            } ?: mapOf()

        model = cmd.getOptionValue(MODEL.opt) ?: model

        fromPage = cmd.getOptionValue(FROM_PAGE.opt)?.toInt() ?: 1

        val level = Level.valueOf(cmd.getOptionValue(LOGGING_LEVEL.opt))
        (LoggerFactory.getLogger(BkDataEnrichCompareApp::class.java)
            as ch.qos.logback.classic.Logger).level = level

        // если передать уровень < INFO (например WARN), то сообщение, очевидно, не отобразится ¯\_(ツ)_/¯
        LOGGER.info("Running ${this.javaClass.simpleName} with logging level: $level")
    }

    private lateinit var refApi: String
    private lateinit var testApi: String

    /**
     * Example for exact page "{\"page_id\":[\"=\", 2115]}"
     */
    private var filter: String? = "{\"multistate\":\"not deleted\"}"
    private var headers: Map<String, String> = mapOf()
    private var model: String = "context_on_site_campaign"
    private var fromPage: Int = 1 // порядковый номер "страницы" с сущностями (иными словами batch id)
    private var totalCount: Int = 0
    private var errorsCount: Int = 0

    private val webClient = WebClient.builder()
        .build()
    private val objectMapper = ObjectMapper()
    private val pageBkDataComparator = PageBkDataComparator()

    override fun run() {
        val commonHeaders: (t: HttpHeaders) -> Unit = { h ->
            headers.forEach { header -> h.add(header.key, header.value) }
        }

        while (true) {
            val pageIds = webClient
                .get().uri(
                    UriComponentsBuilder.fromUriString(refApi)
                        .path("/v1/$model")
                        .queryParam("fields[$model]", "public_id")
                        .queryParam("page[number]", fromPage)
                        .queryParam("page[size]", 100)
                        .queryParam("filter", "{filter}")
                        .build(mapOf("filter" to filter))
                )
                .headers(commonHeaders)
                .exchangeToFlux { response ->
                    response.toEntity(String::class.java)
                        .filter { it.statusCode.is2xxSuccessful }
                        .map { objectMapper.readTree(it.body) }
                        .map { it.path("data") }
                        .filter { !it.isMissingNode }
                        .flatMapMany { (it as ArrayNode).toFlux() }
                        .map { it.get("id").textValue() }
                }
                .collectList().block()!!
            if (pageIds.isEmpty()) {
                LOGGER.warn("Got empty page ids from $refApi for model $model")
                break
            }
            LOGGER.info("Page: {}, ids: {}", fromPage, pageIds)
            fromPage++

            val lock = ReentrantLock()
            var comparedCount = 0

            pageIds.parallelStream()
                .forEach { pageId ->
                    try {
                        val mono = webClient.get().uri(bkDataUri(refApi, pageId))
                            .headers(commonHeaders)
                            .exchangeToMono {
                                if (!it.statusCode().is2xxSuccessful) {
                                    it.releaseBody()
                                    throw IllegalStateException("Non-successfull status on bkdata request: ${it.statusCode()}")
                                }
                                DataBufferUtils.join(it.body(BodyExtractors.toDataBuffers()))
                            }
                            .retryWhen(Retry.backoff(3, Duration.ofSeconds(5)))
                            .map { objectMapper.readTree(it.asInputStream(true)) }
                            .map { it.path("data").toString() }

                        val refJson = mono.block()!!
                        LOGGER.debug("got refJson")
                        if (LOGGER.isDebugEnabled) {
                            File(".d").mkdirs()
                            Files.writeString(Path.of(".d", "$pageId-ref.json"), refJson)
                            LOGGER.debug("dumped ref bk-data to .d/$pageId-ref.json")
                        }

                        val enrichedJson = webClient.method(HttpMethod.POST).uri(
                            UriComponentsBuilder.fromUriString(testApi)
                                .path("/v1/api/bkdata/enrich_page")
                                .build().toUri()
                        )
                            .headers(commonHeaders)
                            .bodyValue(refJson)
                            .exchangeToMono {
                                if (!it.statusCode().is2xxSuccessful) {
                                    it.releaseBody()
                                    throw IllegalStateException("Non-successfull status on enrich: ${it.statusCode()}")
                                }
                                DataBufferUtils.join(it.body(BodyExtractors.toDataBuffers()))
                            }
                            .retryWhen(Retry.backoff(3, Duration.ofSeconds(5)))
                            .map { buffer ->
                                buffer.toString(Charsets.UTF_8)
                                    .also { DataBufferUtils.release(buffer) }
                            }

                        val actualJson =  enrichedJson.block()!!
                        LOGGER.debug("got actualJson")
                        if (LOGGER.isDebugEnabled) {
                            Files.writeString(Path.of(".d", "$pageId-actual.json"), actualJson, )
                            LOGGER.debug("dumped actual bk-data to .d/$pageId-actual.json")
                        }
                        val compareResult = pageBkDataComparator.compareJSON(JSONObject(refJson), JSONObject(actualJson))

                        lock.withLock {
                            comparedCount++ // инкрементим счетчик внутри page
                            totalCount++ // инкрементим общее число сверок
                            if (!compareResult.passed()) {
                                LOGGER.info("$comparedCount/${pageIds.size} Bk data comparison failed " +
                                    "for page $pageId: {}", compareResult
                                )
                                errorsCount++
                            } else {
                                LOGGER.info("$comparedCount/${pageIds.size} Successfully compared $pageId")
                            }
                            LOGGER.info("Total errors ratio: $errorsCount/$totalCount")
                        }

                    } catch (ex: java.lang.Exception) {
                        LOGGER.error("Compared page $pageId with exception {}", ex)
                    } catch (ex: AssertionError) {
                        LOGGER.error("Compared page $pageId with exception {}", ex)
                    }
                }
        }
    }

    private fun bkDataUri(url: String, pageId: String?) =
        UriComponentsBuilder.fromUriString(url)
            .path("/v1/bkdata/$pageId")
            .build().toUri()
}
