package ru.yandex.partner.scripts.duplicate;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.PrintStream;
import java.math.BigDecimal;
import java.math.RoundingMode;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import javax.net.ssl.SSLException;

import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.annotation.JsonRawValue;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.ObjectWriter;
import com.fasterxml.jackson.databind.SerializationFeature;
import com.fasterxml.jackson.databind.node.ArrayNode;
import com.fasterxml.jackson.databind.node.ObjectNode;
import io.netty.handler.ssl.SslContextBuilder;
import io.netty.handler.ssl.util.InsecureTrustManagerFactory;
import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.Options;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import org.skyscreamer.jsonassert.FieldComparisonFailure;
import org.skyscreamer.jsonassert.JSONCompare;
import org.skyscreamer.jsonassert.JSONCompareMode;
import org.skyscreamer.jsonassert.JSONCompareResult;
import org.skyscreamer.jsonassert.comparator.DefaultComparator;
import org.springframework.http.ResponseEntity;
import org.springframework.http.client.reactive.ReactorClientHttpConnector;
import org.springframework.lang.Nullable;
import org.springframework.web.reactive.function.client.WebClient;
import reactor.core.publisher.Mono;
import reactor.netty.http.client.HttpClient;

import ru.yandex.partner.libs.cli.CliApp;
import ru.yandex.partner.libs.cli.CliAppException;


/**
 * ./duplicate-check.sh -ref_url https://perl-host/restapi -test_url https://java-host/restapi -output_dir /tmp \
 * -filter '["AND",[{"id":["=","R-A-1086-1"]},{"multistate":["=","not deleted"]}]]' \
 * -H 'Cookie: {Cookie}'
 */
public class DuplicateComparisonApp extends CliApp {
    private static final Pattern DESIGN_ID_PATTERN = Pattern.compile("data\\.attributes\\.design_templates\\[.+]\\.id");

    private static final String REF = "ref";
    private static final String TEST = "test";

    private final ObjectMapper objectMapper;
    private final ObjectWriter objectWriter;
    private final Map<String, String> urlAliasMap;
    private final Map<String, String> aliasUrlMap;
    private final Map<String, String> aliasHostMap;
    private final Map<String, WebClient> webClientMap;
    private final ExecutorService executorService;

    private String refUrl;
    private String testUrl;
    private String filter;
    private String outputDir;
    private Map<String, String> headers;
    private long start;
    private long limit;
    private long count;

    public DuplicateComparisonApp() {
        this.objectMapper = new ObjectMapper();
        this.objectMapper.configure(SerializationFeature.FAIL_ON_EMPTY_BEANS, false);
        this.objectWriter = objectMapper.writerWithDefaultPrettyPrinter();
        this.urlAliasMap = new HashMap<>();
        this.aliasUrlMap = new HashMap<>();
        this.aliasHostMap = new HashMap<>();
        this.webClientMap = new HashMap<>();
        this.executorService = Executors.newFixedThreadPool(2);
    }

    public static void main(String[] args) throws Exception {
        (new DuplicateComparisonApp()).run(args);
    }

    @Override
    protected Options getOptions() {
        var options = new Options();
        for (DuplicateCliOptions value : DuplicateCliOptions.values()) {
            options.addOption(value.getOption());
        }
        return options;
    }

    @Override
    protected void configure(CommandLine cmd) throws CliAppException {
        refUrl = cmd.getOptionValue(DuplicateCliOptions.REF_URL.getOption().getOpt());
        testUrl = cmd.getOptionValue(DuplicateCliOptions.TEST_URL.getOption().getOpt());
        filter = cmd.getOptionValue(DuplicateCliOptions.FILTER.getOption().getOpt());
        outputDir = cmd.getOptionValue(DuplicateCliOptions.OUTPUT_DIR.getOption().getOpt());
        headers = Arrays.stream(cmd.getOptionValues(DuplicateCliOptions.HEADER.getOption().getOpt()))
                .map(s -> {
                    int index = s.indexOf(":");
                    if (index < 0) {
                        throw new RuntimeException("Invalid header: " + s);
                    }

                    return Map.entry(s.substring(0, index).trim(), s.substring(index + 1).trim());
                })
                .collect(Collectors.toMap(Map.Entry::getKey, Map.Entry::getValue));

        start = Optional.ofNullable(cmd.getOptionValue(DuplicateCliOptions.START.getOption().getOpt()))
                .map(Long::valueOf)
                .orElse(1L);


        limit = Optional.ofNullable(cmd.getOptionValue(DuplicateCliOptions.LIMIT.getOption().getOpt()))
                .map(Long::valueOf)
                .orElse(1_000L);

        count = Optional.ofNullable(cmd.getOptionValue(DuplicateCliOptions.COUNT.getOption().getOpt()))
                .map(Long::valueOf)
                .orElse(1_039_939L);
        if (refUrl.equals(testUrl)) {
            throw new CliAppException(DuplicateCliOptions.REF_URL.getOption().getOpt()
                    + " cannot be equal " + DuplicateCliOptions.TEST_URL.getOption().getOpt());
        }

        urlAliasMap.put(refUrl, REF);
        urlAliasMap.put(testUrl, TEST);

        aliasUrlMap.put(REF, refUrl);
        aliasUrlMap.put(TEST, testUrl);

        for (var entry : aliasUrlMap.entrySet()) {
            try {
                var url = new URL(entry.getValue());
                aliasHostMap.put(entry.getKey(), url.getHost());
            } catch (MalformedURLException e) {
                throw new CliAppException("Cannot parse host " + entry.getValue(), e);
            }
        }

        try {
            // прогрев кеша
            getWebClient(refUrl);
            getWebClient(testUrl);
        } catch (SSLException e) {
            throw new CliAppException("Cannot initialize WebClient", e);
        }
    }

    @Override
    protected void run() throws Exception {
        long pageNumber = start - 1L;
        count = limit * pageNumber + count;
        // todo беты не вывозят этот запрос
        //    long count = getCount(refUrl, filter);

        while ((pageNumber * limit) < count) {
            pageNumber++;

            List<String> blocksForDuplicate = getIds(refUrl, filter, pageNumber, limit);
            for (String sourcePublicBlockId : blocksForDuplicate) {
                try {
                    var getResponseBefore = get(refUrl, sourcePublicBlockId, List.of("available_fields"),
                            getRequestLogName(null, sourcePublicBlockId, "get_available_fields.json"));
                    List<String> availableFields = extractAvailableFields(getResponseBefore);
                    CompletableFuture<Responses> refResponsesFuture = CompletableFuture
                            .supplyAsync(() -> test(refUrl, sourcePublicBlockId, availableFields), executorService);
                    CompletableFuture<Responses> testResponsesFuture = CompletableFuture
                            .supplyAsync(() -> test(testUrl, sourcePublicBlockId, availableFields), executorService);


                    Responses refResponses = refResponsesFuture.get();
                    Responses testResponses = testResponsesFuture.get();

                    compareDuplicateResponse(sourcePublicBlockId,
                            refResponses.getPostDuplicateResponse(),
                            testResponses.getPostDuplicateResponse());
                    compareGetAfterResponse(sourcePublicBlockId,
                            refResponses.getGetDuplicatedResponse(),
                            testResponses.getGetDuplicatedResponse());
                    compareGetAfterResponse(sourcePublicBlockId,
                            refResponses.getGetSourceResponse(),
                            testResponses.getGetSourceResponse());
                } catch (Exception e) {
                    e.printStackTrace();
                    File file = new File(outputDir + "/errors/" + sourcePublicBlockId + "/error.log");
                    file.getParentFile().mkdirs();
                    if (!file.exists()) {
                        file.createNewFile();
                    }
                    try (var stream = new PrintStream(new FileOutputStream(file))) {
                        e.printStackTrace(stream);
                    }
                }

            }
        }

        executorService.shutdown();
    }

    private Responses test(String host, String sourcePublicBlockId, List<String> availableFields) {
        try {
            var postDuplicateResponse = duplicate(host, sourcePublicBlockId);

            if (postDuplicateResponse.getStatusCodeValue() == 401) {
                postDuplicateResponse = duplicate(host, sourcePublicBlockId);
            }

            if (!postDuplicateResponse.getStatusCode().is2xxSuccessful()) {
                return new Responses(postDuplicateResponse, null, null);
            }

            var newId = objectMapper.readTree(postDuplicateResponse.getBody()).get("data").get("id").asText();
            var getDuplicatedResponse = get(host, newId, availableFields, getRequestLogName(host, sourcePublicBlockId,
                    "get_duplicated.json"));

            var getSourceResponse = get(host, newId, availableFields, getRequestLogName(host, sourcePublicBlockId,
                    "get_source.json"));

            return new Responses(postDuplicateResponse, getSourceResponse, getSourceResponse);
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    private long getCount(String host, String filter) throws SSLException, JsonProcessingException {
        var webClient = getWebClient(host);

        ResponseEntity<String> responseEntity = doAndLogRequest(webClient.get()
                        .uri(uriBuilder -> uriBuilder
                                .path("/v1/context_on_site_rtb")
                                .queryParam("filter", "{filter}")
                                .queryParam("meta", "total")
                                .build(filter))
                        .header("Accept", "application/vnd.api+json")
                        .exchangeToMono(clientResponse -> clientResponse.toEntity(String.class)),
                outputDir + "/count.json");

        if (!responseEntity.getStatusCode().is2xxSuccessful()) {
            return -1L;
        }

        return objectMapper.readTree(responseEntity.getBody()).get("meta").get("found_rows").asLong();
    }

    private List<String> getIds(String host, String filter, long pageNumber, long pageSize)
            throws SSLException, JsonProcessingException {
        var webClient = getWebClient(host);

        ResponseEntity<String> responseEntity = doAndLogRequest(webClient.get()
                        .uri(uriBuilder -> uriBuilder
                                .path("/v1/context_on_site_rtb")
                                .queryParam("fields[context_on_site_rtb]", "public_id")
                                .queryParam("page[number]", pageNumber)
                                .queryParam("page[size]", pageSize)
                                .queryParam("filter", "{filter}")
                                .build(filter))
                        .header("Accept", "application/vnd.api+json")
                        .exchangeToMono(clientResponse -> clientResponse.toEntity(String.class)),
                outputDir + "/test_ids_page_" + pageNumber + "_size_" + pageSize + ".json");

        if (!responseEntity.getStatusCode().is2xxSuccessful()) {
            return List.of();
        }

        ArrayNode arrayNode = (ArrayNode) objectMapper.readTree(responseEntity.getBody()).get("data");
        var list = new ArrayList<String>(arrayNode.size());
        for (JsonNode jsonNode : arrayNode) {
            list.add(jsonNode.get("id").asText());
        }
        return list;
    }

    private ResponseEntity<String> get(String host, String publicBlockId, List<String> fields, String requestLogPath)
            throws SSLException {
        var webClient = getWebClient(host);

        return doAndLogRequest(webClient.get()
                        .uri("/v1/context_on_site_rtb/" + publicBlockId + "/?fields[context_on_site_rtb]=" +
                                String.join(",", fields))
                        .header("Accept", "application/vnd.api+json")
                        .exchangeToMono(clientResponse -> clientResponse.toEntity(String.class)),
                requestLogPath);
    }

    private ResponseEntity<String> duplicate(String host, String publicBlockId) throws SSLException {
        var webClient = getWebClient(host);

        return doAndLogRequest(webClient
                        .post()
                        .uri("/v1/context_on_site_rtb/" + publicBlockId + "/action/duplicate")
                        .header("Accept", "application/vnd.api+json")
                        .exchangeToMono(clientResponse -> clientResponse.toEntity(String.class)),
                getRequestLogName(host, publicBlockId, "duplicate.json"));

    }

    private WebClient getWebClient(String host) throws SSLException {
        var webClient = webClientMap.get(host);
        if (webClient == null) {
            webClient = getWebClientDefault(host);
            webClientMap.put(host, webClient);
        }
        return webClient;

    }

    private ResponseEntity<String> doAndLogRequest(Mono<ResponseEntity<String>> responseEntityMono, String logPath) {
        long startResponse = System.currentTimeMillis();
        var responseEntity = responseEntityMono.block();
        long execTime = System.currentTimeMillis() - startResponse;
        var responseDto = new ResponseDto(responseEntity.getStatusCodeValue(), execTime,
                responseEntity.getBody());
        try {
            var file = new File(logPath);
            file.getParentFile().mkdirs();
            objectWriter.writeValue(file, responseDto);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
        return responseEntity;
    }

    private WebClient getWebClientDefault(String host) throws SSLException {
        var sslContext = SslContextBuilder
                .forClient()
                .trustManager(InsecureTrustManagerFactory.INSTANCE)
                .build();
        var httpClient = HttpClient.create().secure(t -> t.sslContext(sslContext));
        WebClient.Builder builder = WebClient.builder()
                .baseUrl(host)
                .clientConnector(new ReactorClientHttpConnector(httpClient));

        for (Map.Entry<String, String> entry : headers.entrySet()) {
            builder.defaultHeader(entry.getKey(), entry.getValue());
        }

        return builder.build();
    }

    private List<String> extractAvailableFields(ResponseEntity<String> getResponseBefore)
            throws JsonProcessingException {
        Map<String, Boolean> availableFieldsMap =
                objectMapper.treeToValue(objectMapper.readTree(getResponseBefore.getBody()).get("data").get(
                        "attributes").get("available_fields"), Map.class);
        return availableFieldsMap.entrySet()
                .stream()
                .filter(Map.Entry::getValue)
                .map(Map.Entry::getKey)
                .collect(Collectors.toList());
    }

    private String getRequestLogName(@Nullable String host, String publicBlockId, String name) {
        StringBuilder sb = new StringBuilder(outputDir);
        if (!outputDir.endsWith("/")) {
            sb.append('/');
        }
        sb.append(publicBlockId);
        sb.append('/');
        if (host != null) {
            sb.append(urlAliasMap.get(host));
            sb.append('/');
        }

        sb.append(name);
        return sb.toString();
    }

    private void compareDuplicateResponse(String sourceId, ResponseEntity<String> ref, ResponseEntity<String> test)
            throws IOException, JSONException {
        if (ref.getStatusCodeValue() != test.getStatusCodeValue()) {
            File file = new File(outputDir + "/errors/" + sourceId + "/duplicate.json");
            var map = Map.of("expectedStatus", ref.getStatusCodeValue(),
                    "actualStatus", test.getStatusCodeValue());
            file.getParentFile().mkdirs();
            objectWriter.writeValue(file, map);
        } else {
            compareJsonsAndWriteIfError("duplicate", sourceId, ref.getBody(), test.getBody());
        }
    }

    private void compareGetAfterResponse(
            String sourceId,
            ResponseEntity<String> ref,
            ResponseEntity<String> test) throws IOException, JSONException {
        if (ref == null || test == null) {
            return;
        }

        if (ref.getStatusCodeValue() != test.getStatusCodeValue()) {
            File file = new File(outputDir + "/errors/" + sourceId + "/get_after.json");
            var map = Map.of("expectedStatus", ref.getStatusCodeValue(),
                    "actualStatus", test.getStatusCodeValue());
            file.getParentFile().mkdirs();
            objectWriter.writeValue(file, map);
        } else {
            String refBody = ref.getBody().replace(aliasUrlMap.get(REF), "url");
            String testBody = test.getBody().replace(aliasUrlMap.get(TEST), "url");

            JsonNode refBkNode =
                    ((ObjectNode) objectMapper.readTree(refBody).get("data").get("attributes")).remove("bk_data");
            JsonNode testBkNode =
                    ((ObjectNode) objectMapper.readTree(testBody).get("data").get("attributes")).remove("bk_data");

            // в отдельную папку тк там айдишки расходятся
            compareJsonsAndWriteIfError("errors_bk_data", "bk_data", sourceId, refBkNode.asText(), testBkNode.asText());
            compareJsonsAndWriteIfError("get_after", sourceId, refBody, testBody);
        }
    }

    private void compareJsonsAndWriteIfError(String logName, String sourceId, String refJson, String testJson)
            throws JSONException, IOException {
        compareJsonsAndWriteIfError("errors", logName, sourceId, refJson, testJson);
    }

    private void compareJsonsAndWriteIfError(String dir, String logName, String sourceId, String refJson,
                                             String testJson)
            throws JSONException, IOException {
        var replacedRefJson = refJson.replace(aliasHostMap.get(REF), "host")
                .replaceAll("host:\\d+", "host")
                .replace("https", "http");
        var replacedTestJson = testJson.replace(aliasHostMap.get(TEST), "host")
                .replaceAll("host:\\d+", "host")
                .replace("https", "http");
        JSONCompareResult result = JSONCompare.compareJSON(replacedRefJson, replacedTestJson,
                new DefaultComparator(JSONCompareMode.NON_EXTENSIBLE));

        List<FieldComparisonFailure> failures = result.getFieldFailures()
                .stream()
                // фильтр не проверяемых полей
                .filter(
                        fieldComparisonFailure -> !fieldComparisonFailure.getField().contains("create_date")
                                && !fieldComparisonFailure.getField().contains("bk_data")
                                && !DESIGN_ID_PATTERN.matcher(fieldComparisonFailure.getField()).matches())
                // null и пустая строка равны
                // null и пустой массив
                .filter(predicate(FieldComparisonFailure::getActual, FieldComparisonFailure::getExpected))
                .collect(Collectors.toList());

        List<FieldComparisonFailure> unexpected = result.getFieldUnexpected().stream()
                // null и пустая строка равны
                // null и пустой массив
                .filter(predicate(FieldComparisonFailure::getActual, FieldComparisonFailure::getField))
                .collect(Collectors.toList());

        List<FieldComparisonFailure> missing = result.getFieldMissing().stream()
                // null и пустая строка равны
                // null и пустой массив
                .filter(predicate(FieldComparisonFailure::getField, FieldComparisonFailure::getExpected))
                .collect(Collectors.toList());

        List<FieldComparisonFailure> errors = new ArrayList<>(failures.size() + unexpected.size() + missing.size());
        errors.addAll(failures);
        errors.addAll(unexpected);
        errors.addAll(missing);


        if (!errors.isEmpty()) {
            File file = new File(outputDir + "/" + dir + "/" + sourceId + "/" + logName + ".json");
            file.getParentFile().mkdirs();
            objectWriter.writeValue(file, errors);
        }
    }

    private Predicate<FieldComparisonFailure> predicate(Function<FieldComparisonFailure, Object> getActual,
                                                        Function<FieldComparisonFailure, Object> getExpected) {
        return fieldComparisonFailure -> {
            var actual = getActual.apply(fieldComparisonFailure);
            var expected = getExpected.apply(fieldComparisonFailure);

            if (actual == null || expected == null
                    || JSONObject.NULL.equals(expected) || JSONObject.NULL.equals(actual)) {
                Object nonnullValue;
                if (actual == null || JSONObject.NULL.equals(actual)) {
                    nonnullValue = expected;
                } else {
                    nonnullValue = fieldComparisonFailure.getActual();
                }

                if ("".equals(nonnullValue)) {
                    return false;
                }
                if (nonnullValue instanceof Collection && ((Collection<?>) nonnullValue).isEmpty()) {
                    return false;
                }
                if (nonnullValue instanceof JSONArray && ((JSONArray) nonnullValue).length() == 0) {
                    return false;
                }
            }
            return true;
        };
    }


    private static class Responses {
        private final ResponseEntity<String> postDuplicateResponse;
        private final ResponseEntity<String> getDuplicatedResponse;
        private final ResponseEntity<String> getSourceResponse;

        Responses(ResponseEntity<String> postDuplicateResponse, ResponseEntity<String> getDuplicatedResponse,
                  ResponseEntity<String> getSourceResponse) {
            this.postDuplicateResponse = postDuplicateResponse;
            this.getDuplicatedResponse = getDuplicatedResponse;
            this.getSourceResponse = getSourceResponse;
        }

        ResponseEntity<String> getPostDuplicateResponse() {
            return postDuplicateResponse;
        }

        ResponseEntity<String> getGetDuplicatedResponse() {
            return getDuplicatedResponse;
        }

        public ResponseEntity<String> getGetSourceResponse() {
            return getSourceResponse;
        }
    }

    private static class ResponseDto {
        private static final BigDecimal THOUSAND = BigDecimal.valueOf(1000L);

        @JsonProperty
        private int status;
        @JsonProperty
        private BigDecimal execSeconds;
        @JsonRawValue
        private String body;

        ResponseDto(int status, long execMillis, String body) {
            this.status = status;
            this.execSeconds = BigDecimal.valueOf(execMillis).divide(THOUSAND, 3, RoundingMode.HALF_UP);
            this.body = body;
        }

        int getStatus() {
            return status;
        }

        public BigDecimal getExecSeconds() {
            return execSeconds;
        }

        String getBody() {
            return body;
        }
    }
}
