package Partner::DB;

use strict;
use warnings;

use Carp;
use Data::Dumper;
use English;
use Exporter;
use HTTP::Request;
use JSON::XS qw( decode_json encode_json );
use LWP::UserAgent;
use Net::INET6Glue::INET_is_INET6;

our @ISA       = qw(Exporter);
our @EXPORT_OK = qw(
  get_db_partner
  get_db_partnerlogs
  get_db_partner2
  get_db_partner_logs2
  get_db_clickhouse_mdb
  );
our @EXPORT = @EXPORT_OK;

my %deploy_servers = (
    autotest  => 'partner-autotest-stage',
    autotest2 => 'partner-autotest-stage-2',
    autotest3 => 'partner-autotest-stage-3',
    oneshot   => 'partner-oneshot-db',
    preprod   => 'partner-preprod-stage',
);

my $avaliable_servers = {
    dev    => 1,
    docker => 1,
    prod   => 1,
    ts     => 1,
    %deploy_servers,
};

my $PI_SECRETS_PATH = '/etc/pi-secrets.json';

sub get_db_partner2 {
    my (%params) = @_;

    my %fixed_params = _get_fixed_params(
        server            => $params{server},
        avaliable_servers => $avaliable_servers,
    );

    my $se;

    if ($fixed_params{server} eq 'prod') {
        $se = _get_db_prod_partner2();
        $se->{prod} = 1;
    } elsif ($fixed_params{server} eq 'dev') {
        $se = _get_db_dev_partner2();
    } elsif ($fixed_params{server} =~ 'docker') {
        $se = _get_db_docker_partner2($fixed_params{server});
    } elsif ($fixed_params{server} eq 'ts') {
        $se = _get_db_ts_partner2();
    } elsif ($deploy_servers{$fixed_params{server}}) {
        $se = _get_db_deploy_partner2($deploy_servers{$fixed_params{server}});
    } else {
        croak "Internal error. This should not happen";
    }

    return $se;
}

sub get_db_partner_logs2 {
    my (%params) = @_;

    my %fixed_params = _get_fixed_params(
        server            => $params{server},
        avaliable_servers => $avaliable_servers,
    );

    my $se;

    if ($fixed_params{server} eq 'prod') {
        $se = _get_db_prod_partner_logs2();
        $se->{prod} = 1;
    } elsif ($fixed_params{server} eq 'dev') {
        $se = _get_db_dev_partner_logs2();
    } elsif ($fixed_params{server} eq 'ts') {
        $se = _get_db_ts_partner_logs2();
    } elsif ($deploy_servers{$fixed_params{server}}) {
        $se = _get_db_deploy_partner_logs2($deploy_servers{$fixed_params{server}});
    } else {
        croak "Internal error. This should not happen";
    }

    return $se;
}

sub get_db_clickhouse_mdb {
    my (%params) = @_;

    my $port;
    my $passwd = '';
    my $user = 'partner';
    if ($params{server} eq 'ts') {
        $port   = 19001;
        $passwd = _get_secret('partner2-clickhouse-password-testing');
        $user   = _get_secret('partner2-clickhouse-user-testing');
    } elsif ($params{server} eq 'prod') {
        $port   = 19000;
        $passwd = _get_secret('partner2-clickhouse-password-production');
        $user   = _get_secret('partner2-clickhouse-user-production');
    }

    return {
        database => 'partner',
        user     => $user,
        password => $passwd,
        host     => 'localhost',
        port     => $port,
    };

}

############################

sub _get_fixed_params {
    my (%params) = @_;

    my %avaliable_servers = %{
        delete($params{avaliable_servers}) // {
            test => 1,
            prod => 1,
        }
      };

    my %fixed_params;

    if (defined $params{server}) {
        my $server = ($params{server} =~ m/(\w+)/)[0];
        if ($avaliable_servers{$server}) {

            $fixed_params{server} = $params{server};

        } else {
            croak "Got incorrect value for parameter 'server'. Expected '"
              . join("', '", keys %avaliable_servers)
              . "', but got '"
              . $params{server}
              . "'. Stopped";
        }

    } else {
        croak "Got no value for parameter 'server'. Stopped";
    }

    return %fixed_params;
}

sub _get_db_prod_partner2 {

    my $config = _get_secret('connection-to-partner2-prod-database');

    return {
        database => 'partner',
        host     => '127.0.0.1',
        %$config,
    };
}

sub _get_db_dev_partner2 {
    my $config = _get_secret('connection-to-partner2-development-database');

    return {
        database => 'partner',
        host     => '127.0.0.1',
        %$config,
    };
}

sub _get_db_docker_partner2 {
    my $port = (split /:/, shift)[1];
    return {
        database => 'partner',
        user     => 'root',
        password => '',
        host     => '127.0.0.1',
        port     => $port,
    };
}

sub _get_db_ts_partner2 {
    my $config = _get_secret('connection-to-partner2-testing-database');

    return {
        database => 'partner',
        host     => '127.0.0.1',
        %$config,
    };
}

sub _resolve_db_endpoint {
    my ($cluster, $stage_name) = @_;

    my $ua  = LWP::UserAgent->new();
    my $res = $ua->request(
        HTTP::Request->new(
            'POST',
            'http://sd.yandex.net:8080/resolve_endpoints/json',
            undef,
            encode_json(
                {
                    cluster_name    => $cluster,
                    endpoint_set_id => "${stage_name}.Database",
                    client_name     => 'robot-partner',
                }
            )
        )
    );

    my $data = decode_json($res->decoded_content)->{endpoint_set}{endpoints}[0];

    return wantarray ? ($data->{fqdn}, $data->{port}) : $data->{fqdn};
}

sub _resolve_any_db_endpoint {
    my ($stage_name) = @_;

    for my $cluster (qw(vla sas myt man)) {
        my ($host, $port) = _resolve_db_endpoint($cluster, $stage_name);

        return (wantarray ? ($host, $port) : $host)
          if $host;
    }

    croak "Internal error. Couldn't find $stage_name";
}

sub _get_db_deploy {
    my ($stage_name) = @_;

    my ($host, $port) = _resolve_any_db_endpoint($stage_name);

    return {
        host     => $host,
        port     => $port,
        user     => 'root',
        password => '',
    };
}

sub _get_db_deploy_partner2 {
    my ($stage_name) = @_;

    my $config = _get_db_deploy($stage_name);
    $config->{database} = 'partner';

    return $config;
}

sub _get_db_deploy_partner_logs2 {
    my ($stage_name) = @_;

    my $config = _get_db_deploy($stage_name);
    $config->{database} = 'partner_logs';

    return $config;
}

sub _get_db_prod_partner_logs2 {
    my $config = _get_secret('connection-to-partner2-prod-database');

    return {
        database => 'partner_logs',
        host     => '127.0.0.1',
        %$config,
    };
}

sub _get_db_dev_partner_logs2 {
    my $config = _get_secret('connection-to-partner2-development-database');

    return {
        database => 'partner_logs',
        host     => '127.0.0.1',
        %$config,
    };
}

sub _get_db_ts_partner_logs2 {
    my $config = _get_secret('connection-to-partner2-testing-database');

    return {
        database => 'partner_logs',
        host     => '127.0.0.1',
        %$config,
    };
}

sub _get_secret {
    my ($secret) = @_;

    open(my $fh, '<', $PI_SECRETS_PATH) or croak $!;

    my $config = join('', <$fh>);

    close($fh) or croak $!;

    my $data = decode_json($config)->{$secret};
    croak "Secret '$secret' not found" unless $data;

    return $data;
}

1;
