package Partner::DB::Script;

=encoding UTF-8

=head1 DESCRIPTION

Скрипт для подключени к DB

=head1 USAGE

mysql_<dbname> --server=<dev|prod|ts> -- <mysql_options>

=head1 OPTIONS

  server   - dev; prod; ts; docker
  silent   - не выводить команду подключения

=cut

use strict;
use warnings;

use Carp;
use Getopt::Long qw( GetOptions );
use Pod::Usage;
use Pod::Find qw(pod_where);
use Term::ANSIColor qw(colored);

use Partner::DB;

our @ISA       = qw(Exporter);
our @EXPORT_OK = qw( run );


sub run {
    my ($dbname) = @_;

    my ( $server, $silent ) = parse_command_line_options();

    my $method = 'get_db_' . $dbname;
    die "wrong dbname=$dbname" unless Partner::DB->can( $method );

    no strict 'refs';
    my $se = $method->( server => $server );

    my $cmd = '';
    if ($dbname =~ /^clickhouse/ ){
        $cmd = _get_ch_connection($se, \@ARGV);
    }
    else {
        $cmd = _get_mysql_connection($se, \@ARGV);
    }

    _print_connection($cmd, $silent);

    system $cmd;
}


sub _get_mysql_connection {
    my ($se, $args) = @_;

    my $user     = $se->{user};
    my $password = $se->{password};
    my $host     = $se->{host};
    my $db       = $se->{database};
    my $port     = $se->{port};

    my $cmd = "MYSQL_PWD=$password  mysql -u $user --host=$host --port=$port --default-character-set=utf8"
        . ($se->{prod} ? ' --prompt="PRODUCTION> "' : '')
        . ( $args ? join(' ', '', @$args) : '' )
        . ' ' . $db;

    return $cmd;
}

sub _get_ch_connection {
    my ($se, $args) = @_;

    my $user     = $se->{user};
    my $password = $se->{password};
    my $host     = $se->{host};
    my $db       = $se->{database};
    my $port     = $se->{port};

    my $cmd = "clickhouse-client --host=$host --user=$user --port=$port --password=$password --database=$db"
        . ( $args ? join(' ', '', @$args) : '' );

    return $cmd;
}

sub _print_connection {
    my ($cmd, $silent) = @_;

    if (-t STDIN && !$silent) {
        print colored($cmd, 'yellow') . "\n";
    }
    print "\n";

    return 1;
}

sub parse_command_line_options {

    my $help = 0;
    my $server = '';
    my $silent = 0;
    GetOptions(
        'server=s' => \$server,
        'silent!'  => \$silent,
        'help|?|h' => \$help
    ) or croak("Error in command line arguments. Stopped");

    pod2usage(-verbose => 2, -noperldoc => 1) if $help;


    croak("No info about server.") unless ($server);

    return ($server, $silent);
}


1;
