package Partner::PR;

=encoding UTF-8
=cut

=head1 DESCRIPTION

Это perl библиотека для работы со Pull Request-ами, кода Партнерского
интерфейса L<https://wiki.yandex-team.ru/partner>.

Сейчас Партнерский Интерфейс использует следующие системы для хранения кода и для
проведения code reiew:

  * L<https://github.yandex-team.ru>

=cut

use strict;
use warnings FATAL => 'all';
use feature 'say';
use utf8;

use Carp;
use Data::Dumper;
use Net::INET6Glue;
use LWP::UserAgent;
use IO::Socket::SSL qw(SSL_VERIFY_NONE);
use HTTP::Request;
use JSON::PP;
use Moment;
use List::Util qw(first);
use URI::Escape qw(uri_escape);

our $GITHUB_BASE_API_URL = 'https://github.yandex-team.ru/api/v3';

=head2 new

Констркутор. Создает объект Partner::PR. Конструктор частично валидирует
параметры, но полная проверка происходит только при вызове методов.

Конструктор не взаимодействует с github (запросы происходят
при вызове методов объекта).

    my $github_pp = Partner::PR->new(
        github_owner => 'partner',
        github_repo => 'yharnam',
    );

=cut

sub new {
    my ($class, @params) = @_;

    if (@params % 2 != 0) {
        croak 'Incorrect usage. new() must get hash like: `new( timestamp => 0 )`. Stopped';
    }

    my %params = @params;

    my $self = {};

    $self->{_github_owner} = delete $params{github_owner};
    $self->{_github_repo}  = delete $params{github_repo};
    $self->{_github_token} = delete $params{github_token};
    $self->{_lwp}          = LWP::UserAgent->new(
        ssl_opts => {
            verify_hostname => 0,
            SSL_verify_mode => SSL_VERIFY_NONE
        }
    );

    croak sprintf("Got unknown params: %s", join ', ', keys %params) if %params;

    bless $self, $class;

    return $self;
}

=head2 get_open_pull_requests_from_github

Метод возвращает открытые PRs - /pulls?state=open
см https://developer.github.com/v3/pulls/

Возвращает массив отсортированый по id.
    {
        assignee             => 'alex-vee',
        author               => 'sullenor',
        codereview_state     => 'success',
        from_branch          => 'DI-1505_the-new-service',
        id                   => 13,
        mergeable            => 1,
        tests_state          => 'success',
        timestamp_created_at => 1454582111,
        to_branch            => 'master',
        url                  => 'https://github.yandex-team.ru/partner/yharnam/pull/13',
    }

Также можно передать approve_status => 1, тогда добаввятся:
        approved_by          => <login>,
        disapproved_by       => <login>,

=cut

sub get_open_pull_requests_from_github {
    my ($self, %opts) = @_;

    croak "No github_owner" if not defined $self->{_github_owner};
    croak "No github_repo"  if not defined $self->{_github_repo};

    croak "approved_by is deprecated" if $opts{approved_by};

    my $prs = [];

    my $raw_json = _github_get_full_data($self, 'pulls?state=open');

    foreach my $element (sort {$a->{number} <=> $b->{number}} @{$raw_json}) {
        push @{$prs},
          {
            'id'                   => $element->{number},
            'url'                  => $element->{html_url},
            'author'               => $element->{user}->{login},
            'assignee'             => $element->{assignee}->{login},
            'timestamp_created_at' => Moment->new(iso_string => $element->{created_at})->get_timestamp(),
            'from_branch'          => $element->{head}->{ref},
            'to_branch'            => $element->{base}->{ref},
          };
    }

    # Дозаполняем структуру статусами пулл реквеста: пройденные тесты, код-ревью, конфиликты мёржа
    foreach my $pr (@$prs) {
        my $pr_statuses = $self->github_get_pr_statuses($pr->{id});
        map {$pr->{$_} = $pr_statuses->{$_}} qw( tests_state  codereview_state  mergeable  changes_requested );

        if ($opts{approve_status}) {
            my $user2status = $self->_github_get_users_approve_status($pr->{id});

            $pr->{approved_by}    = [];
            $pr->{disapproved_by} = [];

            foreach my $user (keys %{$user2status}) {
                if ($user2status->{$user} eq '+') {
                    push @{$pr->{approved_by}}, $user;
                } elsif ($user2status->{$user} eq '-') {
                    push @{$pr->{disapproved_by}}, $user;
                }
            }
        }
    }

    return $prs;
}

sub _github_get_full_data {
    my $self            = shift;
    my $api_url         = shift;
    my $is_custom_repo  = shift;

    my ( $response, $data );
    my $page      = 1;
    my $full_data = [];

    do {
        my $url = $api_url . ($api_url =~ /\?/ ? '&' : '?') . "per_page=100&page=$page";
        ( $response, $data ) = $self->__call(
            method          => 'GET',
            url_part        => $url,
            is_custom_repo => $is_custom_repo
        );
        return $data unless ref($data) eq 'ARRAY';

        push @$full_data, @$data;
        $page++;
    } while (exists $response->headers()->{link} && $response->headers()->{link} =~ /rel="next"/);

    return $full_data;
}

sub github_get_pr_statuses {
    my ($self, $pr_id) = @_;
    my $pr_statuses = {};

    # Commits
    my $pr_commits         = _github_get_full_data($self, sprintf 'pulls/%s/commits', $pr_id);

    # Last Commit
    my $last_pr_commit     = @{$pr_commits}[-1]->{sha};
    my $pr_commit_statuses = $self->github_get_commit_statuses($last_pr_commit);

    # Tests
    my $pr_commit_tests_state = first {$_->{context} =~ /Tests|teamcity/ || $_->{context} =~ m'\./t'} @$pr_commit_statuses;
    $pr_statuses->{tests_state} = $pr_commit_tests_state ? $pr_commit_tests_state->{state} : '';

    # CodeReview (Результат голосования)
    my $pr_commit_codereview_state = first {$_->{context} eq 'Code review'} @$pr_commit_statuses;
    $pr_statuses->{codereview_state} = $pr_commit_codereview_state ? $pr_commit_codereview_state->{state} : '';

    # Request changes
    my $pr_reviews  = _github_get_full_data($self, sprintf 'pulls/%s/reviews', $pr_id);
    my $logins_requested_changes = {};
    foreach my $row (@$pr_reviews){
        $logins_requested_changes->{ $row->{user}->{login} } = 1  if $row->{state} eq 'CHANGES_REQUESTED';
        delete $logins_requested_changes->{ $row->{user}->{login} }  if grep { $row->{state} eq $_ } qw( APPROVED DISMISSED );
    }
    $pr_statuses->{changes_requested} = join ', ', sort keys %$logins_requested_changes;

    $pr_statuses->{mergeable} = $self->github_get_pr_mergeable_flag($pr_id);

    return $pr_statuses;
}

sub github_get_pr_info {
    my ($self, $pr_id) = @_;
    my $pr_info = _github_get_full_data($self, sprintf('pulls/%s', $pr_id) );
    return $pr_info;
}
sub github_get_pr_mergeable_flag {
    my ($self, $pr_id) = @_;
    my $sleep = 1;
    my $pr_info;

    do {
        $pr_info = github_get_pr_info($self, $pr_id);
        sleep($sleep);

        # Check state according
        # http://stackoverflow.com/questions/30619549/why-does-github-api-return-an-unknown-mergeable-state-in-a-pull-request
    } while ($pr_info->{mergeable_state} eq 'unknown');

    $pr_info->{mergeable} ? 1 : 0;
}

sub github_get_commit_statuses {
    my ($self, $commit) = @_;

    my $statuses = _github_get_full_data($self, sprintf('commits/%s/statuses', $commit));

    return $statuses;
}

# https://developer.github.com/v3/repos/commits/#list-commits-on-a-repository
sub github_get_repo_commits {
    my ($self, $path) = @_;
    my $days_depth = 30;    #days

    my $commits = _github_get_full_data( $self,
        sprintf('commits?since=%s',
            Moment->now->minus(day => $days_depth)->get_iso_string . ($path ? "&path=$path" : "") )
    );

    return $commits;
}

sub github_get_repo_tags {
    my ($self) = @_;

    my $tags = _github_get_full_data( $self, 'tags');

    return $tags;
}

# https://developer.github.com/v3/pulls/#merge-a-pull-request-merge-button
sub github_merge_pr {
    my ($self, $pr_id) = @_;

    croak "No github_token" if not defined $self->{_github_token};

    my $api_url = sprintf 'pulls/%s/merge', $pr_id;
    my $data = $self->__call(
        method   => 'PUT',
        url_part => $api_url,
    );
    $data->{merged} ? 1 : 0;
}

sub __call {
    my ($self, %params) = @_;

    croak "Expected to recive 'method'. Stopped"   if not defined $params{method};
    croak "Expected to recive 'url_part'. Stopped" if not defined $params{url_part};

    $params{content} = encode_json($params{content}) if ref $params{content};

    my $sub_path = $params{is_custom_repo}
      ? ''
      : sprintf('repos/%s/%s', @$self{qw( _github_owner  _github_repo)});

    my $url = sprintf('%s/%s', $GITHUB_BASE_API_URL, join('/', grep {$_} $sub_path, $params{url_part}));
    my $request = HTTP::Request->new(
        $params{method} => $url,
        [
            'Content-Type'  => 'application/json',
            'Authorization' => sprintf('token %s', $self->{_github_token}),
        ],
        $params{content},
    );

    my $response = $self->{_lwp}->request($request);
    my $content  = $response->decoded_content;
    my $data = length($content // '')
      ? decode_json $content
      : $content;

    if ($response->is_success) {
        return ( $response, $data );
    } else {
        my $error = 'error';
        eval {
            if ($data->{error}->{error_code}) {
                $error = $data->{error}->{error_code};
            }
        };
        local $Data::Dumper::Indent   = 1;
        local $Data::Dumper::Sortkeys = 1;
        local $Data::Dumper::Terse    = 1;
        croak $error, "\n", Dumper({ api_url => $url, response =>  $data });
    }
}

# https://developer.github.com/v3/repos/branches/#list-branches
sub github_get_branches {
    my ($self) = @_;

    croak "No github_token" if not defined $self->{_github_token};

    my $api_url = sprintf 'branches';
    my $data = $self->__call(
        method   => 'GET',
        url_part => $api_url,
    );
    return {
        map {$_->{name} => $_ } @$data
    };
}

# https://developer.github.com/v3/git/refs/#delete-a-reference
sub github_delete_branch {
    my ($self, $branch_name) = @_;

    croak "No github_token" if not defined $self->{_github_token};

    my $api_url = sprintf 'git/refs/heads/%s', $branch_name;
    my ( $response, $data ) = $self->__call(
        method   => 'DELETE',
        url_part => $api_url,
    );
    return $response->status_line eq '204 No Content'
        ? 1
        : 0;
}

=head2 search_issues_from_github

Вызываем api метод /api/v3/search/issues в github

Возвращает массив:
    {
        'assignee' => undef,
        'author'   => 'zurom',
        'id'       => 259301,
        'number'   => 377,
        'state'    => 'closed',
        'title'    => ''
        'url'      => 'https://github.yandex-team.ru/api/v3/repos/partner/partner2/pulls/377'
    }

=cut

sub search_issues_from_github {
    my ($self, %opts) = @_;
    #%opts = (
    # см https://help.github.com/articles/searching-issues-and-pull-requests/
    #  substr      => 'INFRASTRUCTUREPI-926'
    #  repo        => 'partner/yharnam',
    #  in_title    => 1,
    #  in_body     => 1,
    #  in_comments => 1,
    #  is_merged   => undef,
    #  is_unmerged => undef,
    #  review      => undef, # [none;required;approved;changes_requested]
    #  author      => undef,
    #  assignee    => 'zurom',
    #  in_comments => 1,
    #  type        => 'pr',  # [pr;issue]
    #  state       => undef, # [open;closed]
    #  status      => undef, # [pending;success;failure]
    #  created     => undef, # YYYY-MM-DD.
    #  updated     => undef, # YYYY-MM-DD
    #  label       => '',
    #);

    my $substr = delete($opts{'substr'}) // '';
    $substr = sprintf('"%s"', $substr) if $substr;

    my $available_values = {
        review => [qw( none    required  approved  changes_requested )],
        type   => [qw( pr      issue  )],
        state  => [qw( open    closed )],                                  # state==merged
        status => [qw( pending success  failure )]
    };

    my @keywords = ();
    foreach my $key (keys %opts) {
        my $value = $opts{$key};
        next unless defined $value;
        if ($key =~ m/^(in|is)/) {
            (my $keyword = $key) =~ s/^(in|is)_/$1:/;
            push @keywords, $keyword;
        } else {
            my $available_values = $available_values->{$key};

            die sprintf('Unknown value "%s" for option "%s"', ($value // ''), $key)
              if defined($available_values) && !grep {$value eq $_} @$available_values;

            push @keywords, sprintf('%s:%s', $key, $value);
        }
    }

    my $query_str = join ' ', grep {$_} $substr, @keywords;
    my $url       = sprintf 'search/issues?q=%s', uri_escape($query_str);

    my $data = _github_get_full_data($self, $url, 1);

    my @prs = ();
    foreach my $pr (@{$data->{items}}) {
        push @prs, {
            author   => $pr->{user}->{login},
            assignee => $pr->{assignee}->{login},
            html_url => $pr->{pull_request}->{html_url},
            map {$_ => $pr->{$_}} qw( state  id  number title)
        };
    }

    return \@prs;
}

sub _github_get_users_approve_status {
    my ($self, $id) = @_;

    my $comments = _github_get_full_data($self, sprintf 'issues/%s/comments', $id);

    my %users;
    foreach my $comment (reverse @$comments) {
        my $login = $comment->{user}->{login};
        next if exists $users{$login};

        if (($comment->{body} =~ /^\s*👍/) || ($comment->{body} =~ /^\s*:\+1:/)) {
            $users{$login} = '+';
        } elsif (($comment->{body} =~ /^\s*👎/) || ($comment->{body} =~ /^\s*:\-1:/)) {
            $users{$login} = '-';
        }

    }

    return \%users;
}

1;
