package Yandex::Tanker;

=encoding UTF-8

=cut

# ABSTRACT: API for http://tanker.yandex-team.ru

use strict;
use warnings FATAL => 'all';
use 5.008;

use Carp;
use HTTP::Request;
use HTTP::Request::Common;
use Term::ANSIColor qw(:constants);
use Pod::Usage;
use LWP;

use Yandex::Tanker::JSON;

my $true = 1;
my $false = '';

our $PROD_URL = 'https://tanker-api.yandex-team.ru';
our $TEST_URL = 'https://tanker-api.test.yandex-team.ru';

=head1 SYNOPSIS

    use Yandex::Tanker;

    my $tanker = Yandex::Tanker->new({
        url         => $Yandex::Tanker::PROD_URL,
        project     => 'project_id',
    });

    my @keysets = $tanker->get_keysets();

=head1 DESCRIPTION

Yandex::Tanker version numbers uses Semantic Versioning standart.
Please visit L<http://semver.org/> to find out all about this great thing.

=head1 DOCS

L<http://doc.yandex-team.ru/Tanker/tanker-concepts/>.

L<http://doc.yandex-team.ru/Tanker/api-reference/>.

L<http://wiki.yandex-team.ru/nikitabrovikov/tanker/api>.

=head1 METHODS

=head2 new

Создает объект. В качестве значения url можно передать одну из определенных
переменных $PROD_URL, $TEST_URL.

    my $tanker = Yandex::Tanker->new({
        url         => $Yandex::Tanker::PROD_URL,   # обязательный параметр
        project     => 'project_id',                # обязательный параметр
        token       => '123456',                    # необязательный параметр
    });

=cut

sub new {
    my ($class, $self) = @_;

    _check_required_fields($self, [qw(url project)]);

    # Before release there is no $VERSION variable (is is added with
    # Dist::Zilla)
    my $version;
    if (defined $Yandex::Tanker::VERSION) {
        $version = $Yandex::Tanker::VERSION;
    } else {
        $version = '';
    }

    $self->{ua} = LWP::UserAgent->new;
    $self->{ua}->agent("Yandex::Tanker v.$version");
    $self->{ua}->timeout(600);
    if (defined $self->{token}) {
        $self->{ua}->default_header(AUTHORIZATION => 'OAuth ' . $self->{token});
    }

    bless($self, $class);
    return $self;
}

=head2 get_project_info

B<Get:> 1) $self 2) %opts - хеш с дополнительными настройками

B<Return:> 1) \%info, с описанием проекта

Может принимать опциональный параметр 'branch', значение которого - это имя ветки (по умолчанию исползуется имя 'master').

http://doc.yandex-team.ru/Tanker/api-reference/concepts/operations-projects.xml#projects-description

=cut

sub get_project_info {
    my ($self, %opts) = @_;

    $opts{branch} = 'master' if !defined($opts{branch});

    my $url =
        $self->{url} . "/admin"
        . "/project/" . $self->{project} . '/'
        . "?branch=" . $opts{branch};

    my $response = $self->_get_url($url);
    my $data = from_json($response);

    return $data->{data};
}

=head2 get_keysets

B<Get:> 1) $self 2) %opts - хеш с дополнительными настройками

B<Return:> 1) @ids со списком id всех кейсетов, которые есть в проекте

Возвращает список id кейсетов проекта. Может принимать опциональный параметр
'branch', значение которого - это имя ветки (по умолчанию исползуется имя
'master').

    my @keysets = $tanker->get_keysets(
        branch => 'some_name',
    );

http://doc.yandex-team.ru/Tanker/api-reference/concepts/operations-keysets.xml#keysets-list

=cut

sub get_keysets {
    my ($self, %opts) = @_;

    $opts{branch} = 'master' if !defined($opts{branch});

    my $url =
        $self->{url} . "/admin"
        . "/project/" . $self->{project}
        . "/keysets/"
        . "?branch=" . $opts{branch};

    my $response = $self->_get_url($url);
    my $data = from_json($response);

    my @ids = map {$_->{name}} @{$data->{data}->{items}};

    return @ids;
}

=head2 get_po_translation

B<Get:> 1) $self 2) %opts - хеш с дополнительными настройками

B<Return:> 1) $translation - скаляр с тектом перевода в указанном формате

С помощью этого метода можно получить перевод из танкера. Будут получены
перевод из всех кейсетов.

Минимальный возможный вариант:

    my $po_content = $tanker->get_translation(
        language => 'ru',     # язык, который следует выгрузить
        status => 'approved', # 'approved' или 'unapproved'
    );

Обязательные параметры:

=over

=item B<language> - код языка, который следует выгрузить (например, 'ru' или
'tr')

=item B<status> - Может принимать значения: 'approved' или 'unapproved'.

=back

Кроме этих обязательных параметров, возможно еще указать опциональные
параметры:

=over

=item B<branch> - имя ветви, или идентификатор ревизии проекта. Дефолтное
значение 'master'

=item B<safe> - если этот параметр указан, то вместо пустых переводов будут
выгружены ключи на первоначальном языке

=item B<no-comments> - при указании этого параметра выдача производится без
комментариев

=item B<fix_po_content> в том случае если в разных кейсетах танкера есть
одинаковые msgid, то полученный po файл будет содержать все эти msgid и из-за
этого он будет некорректным. При указании этого параметра, полученный из
танкера po файл будет исправлен: в po файле будут только уникальные msgid.

=back

http://doc.yandex-team.ru/Tanker/api-reference/concepts/operations-translations.xml#trans-download

=cut

sub get_po_translation {
    my ($self, %user_opts) = @_;

    _check_required_fields(
        \%user_opts,
        [
            qw(
                language
                status
            )
        ]
    );

    my %url_opts;

    if ($user_opts{status} eq 'approved') {
        # По дефолту выгружаются только approved, ничего делать не нужно
    } elsif ($user_opts{status} eq 'unapproved') {
        $url_opts{status} = 'unapproved';
    } else {
        croak "Incorrect value for status: '$user_opts{status}'."
            . " Expected 'approved' or 'unapproved'.";
    }

    $url_opts{'project-id'} = $self->{project};

    if (defined($user_opts{branch})) {
        $url_opts{'branch-id'} = $user_opts{branch};
    } else {
        $url_opts{'branch-id'} = 'master';;
    }

    my @avaliable_opts = qw(
        language
        safe
        no-comments
    );

    map {$url_opts{$_} = $user_opts{$_}} grep {defined $user_opts{$_}} @avaliable_opts;

    my $url =
        $self->{url}
        . "/projects/export/po/?"
        . (join '&', map {$_ . "=" . $url_opts{$_}} keys %url_opts);

    my $content = $self->_get_url($url);

    if ($url_opts{'language'} eq 'ru') {
        my $header_ru = '"Plural-Forms: nplurals=3; plural=(n%10==1 && n%100!=11 ? 0 : n%10>=2 && n%10<=4 && (n%100<10 || n%100>=20) ? 1 : 2)\n"';
        $content =~ s/\n\n/\n$header_ru\n\n/;
    }

    my $fixed_content;

    if ($user_opts{fix_po_content}) {
        $fixed_content = $self->_fix_po_from_tanker($content);
    } else {
        $fixed_content = $content;
    }

    return $fixed_content;
}

=head2 get_project_tjson

B<Get:> 1) $self 2) %opts - hash with advanced options

B<Return:> 1) $json - tanker responce with info about all project

    $tanker->get_project_tjson(
        status => 'approved',   # 'approved' или 'unapproved'
        branch => 'feature_x',  # optional, default is master
    );

L<http://doc.yandex-team.ru/Tanker/api-reference/concepts/operations-translations.xml>

L<http://doc.yandex-team.ru/Tanker/api-reference/concepts/formats.xml#format-tjson>

=cut

sub get_project_tjson {
    my ($self,  %user_opts) = @_;

    my $status;
    croak "Need to specify parameter 'status'" if not defined $user_opts{status};
    if ( $user_opts{status} eq 'approved' ) {
        $status = 'approved';
    } elsif ( $user_opts{status} eq 'unapproved' ) {
        $status = 'unapproved';
    } else {
        croak "Incorrect value for status: '$user_opts{status}'."
            . " Expected 'approved' or 'unapproved'.";
    }

    croak "You can't specify parameter 'branch-id' in get_project_tjson()" if defined $user_opts{'branch-id'};
    my $branch = delete $user_opts{branch};
    if (not defined $branch) {
        $branch = 'master';
    }

    my $url =
        $self->{url}
        . "/projects/export/tjson/"
        . "?project-id=" . $self->{project}
        . "&branch-id=" . $branch
        . "&status=" . $status
        ;

    my $content = $self->_get_url($url);

    return $content;
}

=head2 get_keyset_tjson

B<Get:> 1) $self 2) %opts - hash with advanced options

B<Return:> 1) $json - tanker responce with info about specified keyset

    $tanker->get_keyset_tjson(
        'keyset-id' => 'main',
        status => 'approved',   # 'approved' или 'unapproved'
        branch => 'feature_x',  # optional, default is master
    );

L<http://doc.yandex-team.ru/Tanker/api-reference/concepts/operations-translations.xml>

L<http://doc.yandex-team.ru/Tanker/api-reference/concepts/formats.xml#format-tjson>

=cut

sub get_keyset_tjson {
    my ($self,  %user_opts) = @_;

    _check_required_fields(
        \%user_opts,
        [
            qw(
                keyset-id
            )
        ]
    );

    my $status;
    croak "Need to specify parameter 'status'" if not defined $user_opts{status};
    if ( $user_opts{status} eq 'approved' ) {
        $status = 'approved';
    } elsif ( $user_opts{status} eq 'unapproved' ) {
        $status = 'unapproved';
    } else {
        croak "Incorrect value for status: '$user_opts{status}'."
            . " Expected 'approved' or 'unapproved'.";
    }

    croak "You can't specify parameter 'branch-id' in get_keyset_tjson()" if defined $user_opts{'branch-id'};
    my $branch = delete $user_opts{branch};
    if (not defined $branch) {
        $branch = 'master';
    }

    my $url =
        $self->{url}
        . "/keysets/tjson/"
        . "?project-id=" . $self->{project}
        . "&keyset-id=" . $user_opts{'keyset-id'}
        . "&branch-id=" . $branch
        . "&status=" . $status
        ;

    my $content = $self->_get_url($url);

    return $content;
}

sub get_tjson {
    croak "Method get_tjson() is deprecated. Please use get_keyset_tjson() or get_project_tjson().";
}

=head2 create_keyset

B<Параметры:> %opts с настройками

B<Возвращаемое значение:> -

Создает кейсет в танкере

    $tanker->create_keyset(
        file              => 'locale/ru/keysets/settings.po', # этот файл будет залит в танкер
        keyset            => 'settings',
        language          => 'ru',
        format            => 'po',
        'original-not-id' => '1',
    );

Обязательные параметры:

=over

=item B<file> - путь и имя файла, который будет загружен в кейсет

=item B<keyset> - id кейсета в который будет загружен файл, можно передать несколько в массиве

=item B<language> - двухбуквенный код языка

=item B<format> - формат загружаемого файла (android, osx, loc, po, xml, dtd)

=back

Кроме этих обязательных параметров, возможно еще указать опциональные
параметры:

=over

=item B<original-not-id> параметр имеет смысл только при загрузке po файла.
Если параметр не указан, то в msgid будет записана строка из поля
"первоначальный язык" (это поле иногда называют "оригинальный язык"), если же
параметр указан, то в msgid попадет ключ.

=item B<branch> - branch name where create keyset

=back

L<http://doc.yandex-team.ru/Tanker/api-reference/concepts/operations-translations.xml#trans-add>

=cut

sub create_keyset {
    my ($self, %opts) = @_;
    $self->_action_on_keyset(%opts, action => 'create');
}

=head2 replace_keyset

Gets the same parameters as the create_keyset() method.

Here is a description of what will be done in case of replace:

 * msgid is in the file and is in the tanker - after the merge the msgid in
   tanker will stay the same. It it was translated in tanker the translation
   will not be changed (But the translation will be marked as 'Expired').
 * msgid is in the file and is not in the tanker - msgid will be created in
   tanker and the field "original language" will also get the msgid string.
 * msgid is not in the file and is in the tanker - msgid will be deleted in
   tanker, no matter if it is translated or not.

=cut

sub replace_keyset {
    my ($self, %opts) = @_;

    croak "replace_keyset() does not accept parameter 'force-replace'" if exists $opts{'force-replace'};

    $self->_action_on_keyset(%opts, action => 'replace');
}

=head2 delete_keyset

B<Get:> 1) $self 2) %opts - хеш с дополнительными настройками

B<Return:> -

С помощью этого метода можно удалить кейсет из танкера.

Минимальный возможный вариант:

    $tanker->delete_keyset(
        keyset => 'main_page',  # id кейсета, который нужно удалить
    );

Обязательные параметры:

=over

=item B<keyset> - id кейсета, который нужно удалить

=back

Кроме этих обязательных параметров, возможно еще указать опциональные
параметры:

=over

=item B<branch> - имя ветви, или идентификатор ревизии проекта. Дефолтное
значение 'master'

L<http://doc.yandex-team.ru/Tanker/api-reference/concepts/operations-keysets.xml#keysets-delete>

=back

=cut

sub delete_keyset {
    my ($self, %opts) = @_;

    _check_required_fields(
        \%opts,
        [
            qw(
              keyset
              )
        ]
    );

    $opts{branch} = 'master' if !defined($opts{branch});

    my $url =
        $self->{url} . "/admin"
        . "/project/" . $self->{project}
        . "/keyset/" . $opts{keyset}
        . "/?branch=" . $opts{branch}
        ;

    my $request = HTTP::Request->new('DELETE', $url);
    my $response = $self->{ua}->request($request);

    if (!$response->is_success()) {
        $self->_print_debug_and_croak(
            url => $url,
            content => $response->content(),
        );
    }

    return '';
}

=head2 get_branches

B<Get:> 1) $self

B<Return:> 1) @branchNames со списком имен всех ветвей, которые есть в проекте

Возвращает список имен ветвей проекта.

    my @branches = $tanker->get_branches();

http://doc.yandex-team.ru/Tanker/api-reference/concepts/operations-branches.xml#branches-list

=cut

sub get_branches {
    my ($self) = @_;

    my $url =
        $self->{url} . "/admin"
        . "/project/" . $self->{project}
        . "/branches/";

    my $response = $self->_get_url($url);
    my $data = from_json($response);

    my @branchNames = map {$_->{name}} @{$data->{data}->{items}};

    return @branchNames;
}

=head2 create_branch

B<Параметры:> %opts с настройками

B<Возвращаемое значение:> -

Создает ветвь в танкере

    $tanker->create_branch(
        name => 'foo'
        ref => 'bar'
    );

Обязательные параметры:

=over

=item B<name> - имя новой ветви

=item B<ref> - ревизия, от которой ветка будет создана

=back

L<http://doc.yandex-team.ru/Tanker/api-reference/concepts/operations-branches.xml#branches-create>

=cut

sub create_branch {
    my ($self, %opts) = @_;

    _check_required_fields(
        \%opts,
        [
            qw(
              name
              ref
              )
        ]
    );

    my $url =
        $self->{url} . "/admin"
        . "/project/" . $self->{project}
        . "/branch/"
        ;

    my $data = to_json(\%opts);

    my $request = HTTP::Request->new('POST', $url);
    $request->header('Content-Type' => 'application/json');
    $request->content($data);

    my $res = $self->{ua}->request($request);

    if (!$res->is_success()) {
        $self->_print_debug_and_croak(
            url => $url,
            content => $res->content(),
        );
    }
}

=head2 delete_branch

B<Get:> 1) $self 2) %opts - хеш с дополнительными настройками

B<Return:> -

С помощью этого метода можно удалить ветвь из танкера.

Пример:

    $tanker->delete_branch(
        branch => 'foo'
    );

Обязательные параметры:

=over

=item B<branch> - имя ветви для удаления

=back

L<http://doc.yandex-team.ru/Tanker/api-reference/concepts/operations-branches.xml#branches-delete>

=cut

sub delete_branch {
    my ($self, %opts) = @_;

    _check_required_fields(
        \%opts,
        [
            qw(
              branch
              )
        ]
    );

    my $url =
        $self->{url} . "/admin"
        . "/project/" . $self->{project}
        . "/branch/" . $opts{branch} . "/";

    my $request = HTTP::Request->new('DELETE', $url);
    my $response = $self->{ua}->request($request);

    if (!$response->is_success()) {
        $self->_print_debug_and_croak(
            url => $url,
            content => $response->content(),
        );
    }

    return '';
}

=head2 get_tags

B<Get:> 1) $self

B<Return:> 1) @tagNames со списком всех тэгов, существующих в проекте

Возвращает список тэгов проекта.

    my @tags = $tanker->get_tags();

L<http://doc.yandex-team.ru/Tanker/api-reference/concepts/operations-tags.xml#tags-list>

=cut

sub get_tags {
    my ($self) = @_;

    my $url =
        $self->{url} . "/admin"
        . "/project/" . $self->{project}
        . "/tags/";

    my $response = $self->_get_url($url);
    my $data = from_json($response);

    my @tagNames = map {$_->{tag}} @{$data->{data}->{items}};

    return @tagNames;
}

=head2 create_tag

B<Параметры:> %opts с настройками

B<Возвращаемое значение:> -

Создает тэг в танкере

    $tanker->create_branch(
        tag => 'foo'
        ref => 'bar'
    );

Обязательные параметры:

=over

=item B<name> - имя нового тэга

=item B<ref> - ревизия, от которой будет создан тэг

=back

L<http://doc.yandex-team.ru/Tanker/api-reference/concepts/operations-tags.xml#tags-create>

=cut


sub create_tag {
    my ($self, %opts) = @_;

    _check_required_fields(
        \%opts,
        [
            qw(
              tag
              ref
              )
        ]
    );

    my $url =
        $self->{url} . "/admin"
        . "/project/" . $self->{project}
        . "/tag/"
        ;

    my $data = to_json(\%opts);

    my $request = HTTP::Request->new('POST', $url);
    $request->header('Content-Type' => 'application/json');
    $request->content($data);

    my $res = $self->{ua}->request($request);

    if (!$res->is_success()) {
        $self->_print_debug_and_croak(
            url => $url,
            content => $res->content(),
        );
    }
}

=head2 delete_tag

B<Get:> 1) $self 2) %opts - хеш с дополнительными настройками

B<Return:> -

С помощью этого метода можно удалить тэг из танкера.

Пример:

    $tanker->delete_tag(
        tag => 'foo'
    );

Обязательные параметры:

=over

=item B<tag> - имя тэга для удаления

=back

L<http://doc.yandex-team.ru/Tanker/api-reference/concepts/operations-tags.xml#tags-delete>

=cut

sub delete_tag {
    my ($self, %opts) = @_;

    _check_required_fields(
        \%opts,
        [
            qw(
              tag
              )
        ]
    );

    my $url =
        $self->{url} . "/admin"
        . "/project/" . $self->{project}
        . "/tag/" . $opts{tag} . "/";

    my $request = HTTP::Request->new('DELETE', $url);
    my $response = $self->{ua}->request($request);

    if (!$response->is_success()) {
        $self->_print_debug_and_croak(
            url => $url,
            content => $response->content(),
        );
    }

    return '';
}

=head2 merge

B<Параметры:> 1) $self 2) %opts с настройками

B<Возвращаемое значение:> 1) $status - результат мержа - true если мерж
завершился успешно и false если в результате мержа возникли конфликты 2)
$conflict_json - json в котором описаны все проблемы, которые возникли при мерже.
(задача исправить все эти ошибки и передать поправленный json на эту же ручку)

Обязательные параметры:

=over

=item B<source> - имя ветки-источника (из нее будут взяты данные и будут
залиты в ветку-приемник)

=item B<dest> - имя ветки-приемника. Именно в нее будут залиты изменения из
ветки-источника.

=back

Опциональные параметры:

=over

=item B<solution> - скаляр с json в котором разрешены все конфликты (это
дополныенный данными $conflict_json)

=back

L<http://doc.yandex-team.ru/Tanker/api-reference/concepts/operations-branches.xml?ncrnd=680708#branches-merge>

=cut

sub merge {
    my ($self, %opts) = @_;

    _check_required_fields(
        \%opts,
        [
            qw(
              source
              dest
              )
        ]
    );

    my $url =
        $self->{url}
        . "/branches"
        . "/merge"
        . "/?project-id=" . $self->{project}
        . "&source=" . $opts{source}
        . "&dest=" . $opts{dest}
        ;

    my $data = $opts{solution} || "{}";

    my $request = HTTP::Request->new('POST', $url);
    $request->header('Content-Type' => 'application/json');
    $request->content($data);

    my $res = $self->{ua}->request($request);

    if (!$res->is_success()) {
        $self->_print_debug_and_croak(
            url => $url,
            content => $res->content(),
        );
    }

    my $status;
    my $conflict_json = $res->content();

    if ($conflict_json eq "{}") {
        $status = $true;
    } else {
        $status = $false;
    }

    return $status, $conflict_json;
}

=head2 get_stat

B<Get:> 1) $self

B<Return:> 1) $stat - hashref with stat data

L<http://doc.yandex-team.ru/Tanker/api-reference/concepts/operations-statistics.xml>

It will return something like:

    {
        keysets => {
            keyset_name1    => {
                keys  => 3,
                stats => {
                    en => {
                        approved             => 3,
                        expired              => 0,
                        external_translation => 0,
                        generated            => 0,
                        requires_translation => 0,
                        translated           => 0
                    },
                    ru => {
                    ...
                    },
                    ...
                }
            },
            ...
        }
        summary => {
            en => {
                approved             => 1470,
                expired              => 21,
                external_translation => 0,
                generated            => 0,
                requires_translation => 1253,
                translated           => 704
            },
            ru => {
                ...
            },
            ...
        },
        updated => "13:45 12.10.2012"
    }

=cut

sub get_stat {
    my ($self, %opts) = @_;

    croak "You can't specify parameter 'branch-id' in get_stat()" if defined $opts{'branch-id'};
    my $branch = delete $opts{branch};
    if (not defined $branch) {
        $branch = 'master';
    }

    my $url =
        $self->{url} . "/stats/"
        . "?project-id=" . $self->{project}
        . "&branch-id=" . $branch
        . "&json"
        ;

    my $response = $self->_get_url($url);
    my $data = from_json($response);

    return $data;
}

sub _action_on_keyset {
    my ($self, %opts) = @_;

    _check_required_fields(
        \%opts,
        [
            qw(
              file
              keyset
              )
        ]
    );

    croak "Incorrect use of 'branch-id' parameter. Please use 'branch' insted" if defined($opts{'branch-id'});

    my $branch = delete $opts{branch};;
    if (not defined $branch) {
        $branch = 'master';
    }

    my $file = delete $opts{file};
    my $action = delete $opts{action};
    my $keysets = delete $opts{keyset};
    $keysets = [$keysets] if ref($keysets) ne 'ARRAY';

    my $form_ref = [
        %opts,
        'project-id' => $self->{project},
        file         => [$file],
        'branch-id'  => $branch,
        (map {+'keyset-id' => $_} @$keysets),
    ];

    my $url = $self->{url} . "/keysets/$action/";

    my $res = $self->{ua}->request(POST $url, $form_ref, Content_Type => 'multipart/form-data',);

    if (!$res->is_success()) {
        $self->_print_debug_and_croak(
            url => $url,
            content => $res->content(),
        );
    }

}

sub _get_url {
    my ($self, $url) = @_;

    my $res = $self->{ua}->request(GET $url);

    if ($res->is_success()) {
        return $res->content();
    } else {
        $self->_print_debug_and_croak(
            url => $url,
            content => $res->content(),
        );
    }
}

sub _check_required_fields {
    my ($opts_ref, $required_ref) = @_;

    my @defined_opts_keys = grep {defined $opts_ref->{$_}} keys %{$opts_ref};
    my @missing_fields = grep {not _is_in_array($_, \@defined_opts_keys)} @{$required_ref};

    if (@missing_fields) {
        croak "Missing required fields: '" . join("', '", @missing_fields) . "'";
    }
}

sub _is_in_array {
    my ($value, $array_ref) = @_;

    croak "Expected value" if not defined $value;
    croak "Expected array_ref" if not defined $array_ref;

    croak "Expecting first parameter to be scalar" if ref $value ne '';
    croak "Expecting second parameter to be ARRAYREF" if ref $array_ref ne 'ARRAY';

    return exists {map { $_ => 1 } @{$array_ref}}->{$value};
}

=begin comment _fix_po_from_tanker

B<Параметры:> 1) $self 2) $po_file_content

B<Возвращаемое значение:> 1) $fixed_po_file_content

Ручка получения для получения перевода всего проекта в формате po не
гарантирует уникальность msgid. В том случае если в разных кейсетах есть
строки с одинаковым msgid, то po файл полученный из танкера будем битым.

Правильный способ решения этой проблемы - это контролировать чтобы в разных
кейсетах не было строк с совпадающим msgid. Но пока правильный способ не
реализован, есть возможность исправлять содержимое po файла из танкера.

На входе этого метода содержимое po файла полученное от танкера, на выходе -
содержимое po файла из которого убраны повторящиеся msgid.

В результирующем po файле есть только строки у которых есть перевод в (поле
msgstr)

=end comment

=cut

sub _fix_po_from_tanker {
    my ($self, $po_content) = @_;

    my $first_block;
    my @fixed_blocks;

    my $block_id = 0;

    foreach my $block ($self->_split_po_to_blocks($po_content)) {

        if ( $self->_is_first_block($block) ) {
            $first_block = $block . "\n";
        } elsif ( $self->_block_has_translation($block) ) {
            my $msgid = $self->_get_msgid_from_block($block);
            my $msgctxt = $self->_get_msgctxt_from_block($block);

            # Этот блок я видел в случае, если:
            my $has_seen_this_block = $false;
            my $has_seen_in_block;
            foreach (@fixed_blocks) {
                if (defined $msgctxt) {
                    # Есть msgctxt, совпдают и msgid, и msgctxt
                    if ( $msgid eq $_->{msgid} and defined($_->{msgctxt}) and $msgctxt eq $_->{msgctxt}) {
                        $has_seen_this_block = $true;
                        $has_seen_in_block = $_->{block_id},
                    }
                } else {
                    # Нет msgctxt, совпадат msgid, а msgctxt отсутствует
                    if ($msgid eq $_->{msgid} and not defined $_->{msgctxt}) {
                        $has_seen_this_block = $true;
                        $has_seen_in_block = $_->{block_id},
                    }
                }
            }

            if (not $has_seen_this_block) {
                # Если раньше не видел эту совокупность msgid + msgctxt
                # Записываю что это блок нужно будет записать в po файл
                push @fixed_blocks, {
                    msgid => $msgid,
                    msgctxt => $msgctxt,
                    block => $block,
                    block_id => $block_id,
                };
                $block_id++;
            } else {
                # Если же раньше видел эту совокупность msgid + msgctxt
                # Дополняю уже сохраненный блок комментарием
                my $comment = $self->_get_comment_from_block($block);
                foreach (@fixed_blocks) {
                    if ($_->{block_id} == $has_seen_in_block) {
                        $_->{block} = $comment . $_->{block};
                    }
                }
            }

        }

    }

    my $fixed_po_content =
        $first_block
        . join ("\n", map {$_->{block}} @fixed_blocks)
        ;

    return $fixed_po_content;
}

sub _split_po_to_blocks {
    my ($self, $po_content) = @_;

    my @blocks;

    my @lines = split /\n/, $po_content;
    my $block_separator = '';
    my $block;

    foreach my $line (@lines) {

        if ($line eq $block_separator) {
            push @blocks, $block;
            $block = '';
        } else {
            $block .= $line . "\n";
        }

    }

    if (length $block > 0) {
        push @blocks, $block;
    }

    return @blocks;
}

sub _is_first_block {
    my ($self, $block) = @_;

    my $start_line = qq{# This file was generated by Yandex.Tanker project.};

    if ( ($block =~ /^$start_line/) or ($block =~ /msgid ""/s) ) {
        return 1;
    } else {
        return '';
    }
}

sub _block_has_translation {
    my ($self, $block) = @_;

    if ($block !~ /msgstr ""\s*$/m) {
        return 1;
    } else {
        return '';
    }
}

sub _get_msgid_from_block {
    my ($self, $block) = @_;

    $block =~ /msgid "(.*)"\n/m;
    my $msgid = $1;

    return $msgid;
}

sub _get_msgctxt_from_block {
    my ($self, $block) = @_;

    my $msgctxt;

    if ($block =~ /msgctxt "(.*)"\n/m) {
        $msgctxt = $1;
    }

    return $msgctxt;
}

sub _get_comment_from_block {
    my ($self, $block) = @_;

    my $comment;
    my @lines = split /\n/, $block;

    foreach (@lines) {
        $comment .= "$_\n" if /^#/;
    }

    $comment = "#\n" unless defined($comment);

    return $comment;
}

sub _print_debug_and_croak {
    my ($self, %opts) = @_;

    croak
        "Got some error.\n"
        . "The url was:\n"
        . "\n\t" . $opts{url} . "\n\n"
        . "Got content:\n"
        . "*"x78 . "\n"
        . $opts{content}
        . "\n"
        . "*"x78 . "\n"
        ;
}


sub _print_error_and_exit {
    my ($message) = @_;

    print "\n";
    print BOLD RED "Error: $message", RESET;
    print "\n\n";
    pod2usage();

    exit;
}

=head1 SOURCE CODE

The source code for this module is hosted on Yandex.GitHub
L<https://github.yandex-team.ru/bessarabov/Yandex-Tanker>

=head1 BUGS

Please report any bugs or feature requests in GitHub Issues
L<https://github.yandex-team.ru/bessarabov/Yandex-Tanker/issues>

=cut

1;
