package Yandex::Wiki;

# ABSTRACT: Perl interface for wiki.yandex-team.ru

use strict;
use warnings FATAL => 'all';
use 5.010;
use Carp;
use Net::INET6Glue::INET_is_INET6;
use LWP::UserAgent;
use IO::Socket::SSL qw(SSL_VERIFY_NONE);
use HTTP::Request;
use JSON::PP;

=encoding UTF-8

=cut

=head1 SYNOPSIS

    my $wiki = Yandex::Wiki->new(
        oauth_token => $oauth_token,
    );

    say $wiki->get_page('/HomePage');

    $wiki->save_page(
        url => '/tmp/Sandbox',
        title => 'Page title',
        content => 'Test page content',
    );

    my @children = $wiki->get_children('/partner/w');

Yandex::Wiki uses Semantic Versioning standart for version numbers. Please
visit L<http://semver.org/> to find out all about this great thing.

https://wiki.yandex-team.ru/wiki/dev/api/autodocs/

=head1 METHODS

=head2 new

The constructor will connect to yandex wiki and it will save auth marker in
the object. In case of any error the constructor will die;

    my $wiki = Yandex::Wiki->new(
        oauth_token => $oauth_token,
        url => "https://wiki-api.yandex-team.ru",   # optional
    );

=cut

sub new {
    my ($class, @params) = @_;

    if (ref($params[0]) eq 'HASH') {
        croak "Since version 2.0.0 the interface of Yandex::Wiki has changed."
            . " Please use key-value parameters in the method new(). Stopped"
    }

    my %params = @params;

    croak "Parameter 'login' is deprecated. Use 'oauth_token'" if $params{login};
    croak "Parameter 'password' is deprecated. Use 'oauth_token'" if $params{password};

    croak "You must specify parameter 'oauth_token'" if not $params{oauth_token};

    my $self = {};
    bless($self, $class);

    $self->{_oauth_token} = $params{'oauth_token'};
    $self->{_url} = ($params{url} || "https://wiki-api.yandex-team.ru") . "/_api/frontend";

    $self->{_lwp} = LWP::UserAgent->new(
        ssl_opts => {
            verify_hostname => 0,
            SSL_verify_mode => SSL_VERIFY_NONE
        }
    );

    return $self;
}

=head2 rest_get_page
L<https://wiki.yandex-team.ru/wiki/dev/api/autodocs/#poluchitdljaprosmotrastranicuiligrid.get->
Метод выдает огромную структуру данных с данными про страницу.
Пример:
    my $page_info = $wiki->rest_get_page('/partner/w/adfox/');
После этого в $page_info будет HASHREF с такой структоурой:
    {
        data => {
            access => {
                ...
            },
            actuality_status => 'actual',
            bemjson => {
                ...
            },
            bookmark => undef,
            breadcrumbs => [
                ...
            ],
            comments_count => 0,
            created_at => '2015-09-28T14:33:33',
            current_user_subscription => 'none',
            formatter_version => 30.0,
            is_official => ...,
            lang => '',
            last_author => {
                ...
            },
            modified_at => '2016-03-31T15:43:29',
            owner => {
                ...
            },
            page_type => 'article',
            qr_url => 'https://disk.yandex.net/qr/?text=https%3A%2F%2Fwiki.yandex-team.ru%2Fpartner%2Fw%2Fadfox%3FTn0iMTY7NF8yLlBdckJTSweniMwsNGUstCYSnUZnyhKungUhgBZyJURDIQ%253D%253D
            subpages_count => 0,
            supertag => 'partner/w/adfox',
            tag => 'partner/w/adfox',
            title => 'adfox',
            toc => {
                ...
            },
            url => '/partner/w/adfox',
            user_css => undef,
            version => 41218045,
        },
        debug => {
            app_name => 'wiki',
            app_version => '7.13-3',
            exec_duration => 289,
            hostname => 'cs-wikifront01e',
            method => 'GET',
            page => 'partner/w/adfox',
            view => 'wiki.api_frontend.views.pages.PageView',
        },
        user => {
            login => 'robot-rpc-partner',
            display => 'robot-rpc-partner robot-rpc-partner',
            email => 'robot-rpc-partner@yandex-team.ru',
            uid => 1120000000025712,
            ...
        },
    }
В случае ошибки метод бросает исключение, которое можно поймать:
    my $page_info;
    eval {
        $page_info = $wiki->rest_get_page('/partner/w/adfox/');
    };
=cut

sub rest_get_page {
    my ($self, $page_address) = @_;

    my $page_content = $self->__call(
        method    => 'GET',
        url_part  => $page_address,
    );

    return $page_content;
}

=head2 get_page

The method return the content of the wiki page. The methond will die in case
of any error.

    say $wiki->get_page('/HomePage');

Also see method get_page_revision().

=cut

sub get_page {
    my ($self, $page_address) = @_;

    my $page_content = $self->__call(
        method    => 'GET',
        url_part  => "$page_address/.raw",
    );

    return $page_content->{data}{body};
}

=head2 get_cached_page

The method return the content of the wiki page. The methond will die in case
of any error.

    say $wiki->get_cached_page('/HomePage');

The method works the same as get_page(). The only difference is that method
get_cached_page() will save recived page in the object and in case the same
page is requested it will be returned without accessing wiki. The save_page()
method also stores the saved content in the object cache.

=cut

sub get_cached_page {
    my ($self, $page_address) = @_;

    if (not defined $self->{_cache}->{$page_address}) {
        my $page_content = $self->get_page($page_address);
        $self->{_cache}->{$page_address} = $page_content;
    }

    return $self->{_cache}->{$page_address};
}

=head2 save_page

The method saves the page content on specified page .The methond will die in
case of any error. The method return noting interesting. The method stores
saved page content in the object so get_cached_page() will return what was
saved.

    $wiki->save_page(
        url => '/tmp/Sandbox',
        content => 'Test page content',

        # Title is optional when the page is edited, but is obligatory when
        # createing new page
        title => 'Page title',
    );

=cut

sub save_page {
    my ($self, @params) = @_;

    if (scalar(@params) == 3) {
        croak "Since version 2.0.0 the interface of Yandex::Wiki has changed."
            . " Please use key-value parameters in the method save_page(). Stopped"
    }

    my %params = @params;

    croak "Expected to recive 'url'. Stopped" if not defined $params{url};
    croak "Expected to recive 'content'. Stopped" if not defined $params{content};

    my $page_content = $self->__call(
        method   => 'POST',
        url_part => $params{url},
        content  => {
            ($params{title} ? (title => $params{title}) : ()),
            body  => $params{content},
        },
    );

    $self->{_cache}->{$params{url}} = $params{content};
    return '';
}

=head2 get_children

Getting wiki page's children

    my @children = $wiki->get_children('/partner/w');

After that @children will contain somthing like:

    (
        "/partner/w/ad",
        "/partner/w/ad-platform",
    );

=cut

sub get_children {
    my ($self, $page_address) = @_;

    my $result = $self->__call(
        method    => 'GET',
        url_part  => "$page_address/.tree?depth=1",
    );

    $result = [map { $_->{page}{url} } @{$result->{data}->{subpages}}];

    return @{$result};
}

sub get_all_children {
    my ($self, $page_address, %params) = @_;

    $params{depth} //= 3;

    my $query_string = join('&', map { join '=', $_, $params{$_}  } keys %params );

    my $result = $self->__call(
        method    => 'GET',
        url_part  => sprintf('%s/.tree?%s', $page_address, $query_string),
    );

    #$result = {
    #  'data' => {
    #    'expand_all_url' => '//wiki.yandex-team.ru/_api/frontend/partner/w/.actions_view?action_n...',
    #    'limit_exceeded' => 0,
    #    'page' => {
    #      'cluster' => 'w',
    #      'title' => 'Вики о партнерском интерфейсе',
    #      'type' => 'P',
    #      'url' => '/partner/w'
    #    },
    #    'subpages' => [
    #      {
    #        'page' => {
    #          'cluster' => 'task-partner2-person',
    #          'title' => ' ',
    #          'type' => 'P',
    #          'url' => '/partner/w/task-partner2-person'
    #          ...

    return $result;
}

=head2 get_revisions

Getting wiki page's revisions

    my @revisions = $wiki->get_revisions('/partner/w');

After that @revisions will contain somthing like:

    (
        {
            author => {
                display => 'Иван Бессарабов',
                email => 'bessarabov@yandex-team.ru',
                first_name => 'Иван',
                is_admin => 0,
                is_dismissed => 0,
                is_external_employee => 0,
                last_name => 'Бессарабов',
                login => 'bessarabov',
                uid => 1120000000005558,
            },
            created_at => '2012-09-25T15:07:37',
            id => 2905808,
        },
        ...
    )

=cut

sub get_revisions {
    my ($self, $page_address) = @_;


    my $result = $self->__call(
        method    => 'GET',
        url_part  => "$page_address/.revisions",
    );

    $result = $result->{data}{data};

    if (ref $result eq "ARRAY") {
        return @{$result};
    } else {
        croak "Expected to receive array from /.revisions. Stopped";
    }
}

=head2 get_page_revision

The method return the content of the wiki page for the specified revision. The
method will die in case of any error.

    say $wiki->get_page_revision('/partner/w', 2905808);

Revision id can be found by method get_revisions()

=cut

sub get_page_revision {
    my ($self, $page_address, $revision_id) = @_;

    my $page_content  = $self->__call(
        method    => 'GET',
        url_part  => "$page_address/.raw?revision=$revision_id",
    );

    return $page_content->{data}{body};
}

sub __call {
    my ($self, %params) = @_;

    croak "Expected to recive 'method'. Stopped" if not defined $params{method};
    croak "Expected to recive 'url_part'. Stopped" if not defined $params{url_part};

    $params{content} = encode_json($params{content}) if ref $params{content};

    my $request = HTTP::Request->new(
        $params{method} => $self->{_url} . $params{url_part},
        [
            'Content-Type'  => 'application/json',
            'Authorization' => 'OAuth ' . $self->{_oauth_token},
        ],
        $params{content},
    );

    my $response = $self->{_lwp}->request($request);

    local $JSON::PP::true = 1;
    local $JSON::PP::false = 0;

    my $content = $response->decoded_content;
    my $data = length($content // '') ? decode_json $content
                                          : $content;

    if ($response->is_success) {
        return $data;
    } else {
        my $error = 'error';
        eval {
            if ($data->{error}->{error_code}) {
                $error = $data->{error}->{error_code};
            }
        };
        croak $error;
    }
}

=head1 SOURCE CODE

The source code for this module is hosted on Yandex.GitHub
L<https://github.yandex-team.ru/bessarabov/Yandex-Wiki>

=head1 BUGS

Please report any bugs or feature requests in GitHub Issues
L<https://github.yandex-team.ru/bessarabov/Yandex-Wiki/issues>

=cut

1;
