#!/usr/bin/perl

use lib qw(./lib);

use Net::INET6Glue::INET_is_INET6;
use Mojolicious::Lite;
use JSON::PP qw();
use JSON::XS;
use Moment;
use File::Slurp;
use HTTP::Tiny;

use CreatorAPIConstants qw(
    $TVM_BLACKBOX_ID
);
use Utils;
use Yandex::Blackbox;
use Yandex::TVM qw(get_x_ya_service_ticket);
use Time::HiRes qw(gettimeofday tv_interval);

my @PORTS = 8501..8599;
my $BETAS_LIMIT = 20;
my $BACKEND_SERVICES_ENDPOINT = 'partner-test-backend-services-stage.Services';

my $GITHUB_CACHE_FILE_NAME = '/local/creator/meta/github.json';

my $TEMPL_BETA_JSON_PATH = '/local/creator/meta/%s.json';
my $TEMPL_BETA_PATH      = '/local/creator/%s';

my $DOCKER_COMPOSE_YML_PATH = 'docker-compose.yml';
my $UBUNTU_VERSION_PRECISE  = 'precise';
my $UBUNTU_VERSION_BIONIC   = 'bionic';

my $ARCADIA_PERL_PATH = 'partner/perl/partner2';
my $ARCADIA_FRONTEND_PATH = 'adv/frontend/services/yharnam';

my $BETA_HOST = $ENV{BETA_HOST};
die 'Env BETA_HOST must be set' if not defined $BETA_HOST;

hook before_dispatch => sub {
    my ($c) = @_;
    $c->stash(t0 => [gettimeofday()]);

    my $login;

    my $parsed_body = {};
    eval {
        $parsed_body = decode_json $c->req->body;
    };

    if ($parsed_body->{Session_id} or $c->cookie('Session_id')) {
        eval {
            my $data = _get_yandex_blackbox()->sessionid(
                sessionid => $parsed_body->{Session_id} // $c->cookie('Session_id'),
                sslsessionid => $parsed_body->{sessionid2} // $c->cookie('sessionid2'),
                userip => $parsed_body->{userip} // $c->req->headers->header('X-Real-IP'),
                host => 'creator.partner.yandex-team.ru',
            );
            $login = $data->{login};
        };
    }

    if ($login) {
        $c->stash(
            login => $login,
            has_login => JSON::PP::true,
        );
    } else {
        $c->stash(
            has_login => JSON::PP::false,
        );
    }
};

hook after_dispatch => sub {
    my ($c, $args) = @_;

    my $elapsed = tv_interval($c->stash('t0'), [gettimeofday()]);

    my $json_coder = JSON::PP
        ->new
        ->canonical
        ;

    my $now = Moment->now();

    my $log_line = $json_coder->encode({
        timestamp    => $now->get_iso_string(),
        elapsed      => $elapsed,
        code         => $c->{tx}->{res}->{code},
        method       => $c->{tx}->{req}->{method},
        url          => $c->{tx}->{req}->{url}->to_string(),
        'user-agent' => $c->{tx}->{req}->{content}->{headers}->header('user-agent'),
        ip           => $c->{tx}->{req}->{content}->{headers}->header('X-Real-IP'),
        has_login    => $c->stash('has_login'),
        ($c->stash('has_login') ? (login => $c->stash('login')) : ()),
    }) . "\n";

    write_file(
        sprintf('/local/creator/log/api.%s.jsonl', $now->get_d()),
        {
            append => 1,
        },
        $log_line,
    );
};

hook before_render => sub {
    my ($c, $args) = @_;

    my $status = $args->{status} // '';

    if ($status eq 404) {
        $args->{json} = {
            success       => JSON::PP::false,
            error_message => 'not found',
            $args->{msg} ? (msg => $args->{msg}) : ()
        };
    }

    if ($status eq 500) {
        $args->{json} = {
            success       => JSON::PP::false,
            error_message => 'internal error',
            $@ ? (msg => $@) : ()
        };
    }
};

sub get_tags_from_github_cache {
    my ($repo) = @_;

    if (not grep {$_ eq $repo} qw(yharnam partner2)) {
        die 'inrorrect repo';
    }

    my $github_info = get_cached_github_info();

    my @tags = map { [%{$_}]->[0] } @{$github_info->{$repo}->{tags}};

    return @tags;
}

sub get_branches_from_github_cache {
    my ($repo) = @_;

    if (not grep {$_ eq $repo} qw(yharnam partner2)) {
        die 'inrorrect repo';
    }

    my $github_info = get_cached_github_info();

    my @branches = map { [%{$_}]->[0] } @{$github_info->{$repo}->{branches}};

    return @branches;
}

sub get_cached_github_info {

    if (-e $GITHUB_CACHE_FILE_NAME) {
        my $github_info = decode_json scalar read_file $GITHUB_CACHE_FILE_NAME;
        return $github_info;
    } else {
        return {};
    }
}

sub to_pretty_json {
    my ($data) = @_;

    my $json_coder = JSON::PP
        ->new
        ->pretty
        ->canonical
        ->indent_length(4)
        ;

    my $pretty_json = $json_coder->encode($data);

    return $pretty_json;
}

sub get_partner2_production_version {
    my (%params) = @_;

    my @files;

    my $url = 'https://partner.yandex-team.ru/partner2_production_version.json';

    my $data = _get_json_data_from_url($url);

    return $data;
}


=head2 get_branch_sha1

    my $sha1 = get_branch_sha1(
        branch => 'users/login/branch_name',
    );

https://a.yandex-team.ru/arcadia/arc/api/public/

=cut

sub get_branch_sha1 {
    my (%opts) = @_;

    my $branch = delete $opts{branch};
    my $beta_log = delete $opts{log};

    my $arc_token = get_secret('arc-token');
    my $body = "{\"Revision\": \"$branch\"}";

    my $cmd = "/usr/local/sbin/grpcurl/grpcurl -H 'authorization: Bearer $arc_token' -d '$body' api.arc-vcs.yandex-team.ru:6734 NVcs.NPublic.CommitService/GetCommit";
    $beta_log->("Get sha for branch: $branch");
    my $output = `$cmd`;
    $beta_log->("Result: \n$output");
    my $data = decode_json($output);
    return $data->{Commit}->{Oid};
}

=head2 get_all_branches_from_github

Возвращает ARRAY с информацией про все ветки из указанного github репозитория.

    my @branches = get_all_branches_from_github(
        owner => 'partner',
        repo => 'partner2',
    );

Пример \@branches:

    [
        {
            INFRASTRUCTUREPI-647_test => '2c2b1957588e6bfafaada7861149df17eb74573e',
        },
        {
            PI-3230_group_by_in_stat => 'ad2817f64f0a459bd9294b76e1f2d8a396f66d91',
        },
        ...
    ]

L<https://developer.github.com/v3/repos/#list-branches>

=cut

sub get_all_branches_from_github {
    my (%params) = @_;

    my @branches = ();

    # TODO remove or adapt to arcadia
    # my $url = sprintf
    #     'https://github.yandex-team.ru/api/v3/repos/%s/%s/branches',
    #     $params{owner},
    #     $params{repo},
    #     ;
    #
    # while (1) {
    #     my ($response, $data ) = _get_json_data_from_url($url);
    #
    #     push @branches, map {{$_->{name} => $_->{commit}->{sha}}} @$data;
    #
    #     if (
    #         defined($response->{headers}->{link})
    #         && $response->{headers}->{link} =~ /<([^>]+)>; rel="next"/
    #     ) {
    #         $url = $1;
    #     } else {
    #         last;
    #     }
    # }

    return @branches;
}

=head2 get_some_tags_from_github

    my @tags = get_some_tags_from_github(
        owner => 'partner',
        repo => 'partner2',
    );

Возвращает ARRAY с информацией про 30 тегов из указанного github репозитория.

Пример \@tags:

    [
        {
            2.18.232 => '5bde3f4e8566fce6fd280345185d0b0d8bd14cd3',
        },
        {
            2.18.231 => '699e71526130111d7c71729d48129450ecc38eaa',
        },
        ...
    ]

L<https://developer.github.com/v3/repos/#list-tags>

=cut

sub get_some_tags_from_github {
    my (%params) = @_;

    my $url = sprintf
            'https://github.yandex-team.ru/api/v3/repos/%s/%s/tags',
            $params{owner},
            $params{repo};

    my @tags = ();

    # TODO remove or adapt to arcadia
    # my $count = 0;
    # while (1) {
    #     my ($response, $data ) = _get_json_data_from_url($url);
    #
    #     push @tags, map {{$_->{name} => $_->{commit}->{sha}}} grep {$_->{name} =~ /^\d+\.\d+\.\d+$/}  @$data;
    #
    #     if (
    #         defined($response->{headers}->{link})
    #         && $response->{headers}->{link} =~ /<([^>]+)>; rel="next"/
    #     ) {
    #         $url = $1;
    #         last if ++$count > 50;
    #         last if @tags > 50;
    #     } else {
    #         last;
    #     }
    # }

    return @tags;
}

=head2 get_files_list_from_github

Возвращает ARRAY со списокм всех файлов в корне указанного github репозитория.

    my @files = get_files_list_from_github (
        owner => 'partner',
        repo => 'migrationsql',
    );

Пример \@files:

    [
        'PI-5732_last_version.sql',
        ...
        'misc.sql',
        'release_103',
    ]

Возвращаются только файлы (папки не возвращаются). В ответе файлы
отсортированы по имени.

L<http://stackoverflow.com/questions/25022016/get-all-file-names-from-a-github-repo-through-the-github-api>

=cut

sub get_files_list_from_github {
    my (%params) = @_;

    my @files = ();

    # TODO remove or adapt to arcadia
    # my $url = sprintf
    #     'https://github.yandex-team.ru/api/v3/repos/%s/%s/commits/master',
    #     $params{owner},
    #     $params{repo};
    #
    # my $data = _get_json_data_from_url($url);
    #
    # $data = _get_json_data_from_url( $data->{commit}->{tree}->{url} );
    #
    # foreach my $element (@{$data->{tree}}) {
    #     if ($element->{type} eq 'blob') {
    #         push @files, $element->{path};
    #     }
    # }

    return sort @files;
}

=head2 get_branch_migrations

Возвращает список ARRAYREF. Первый элемент - это ARRAYREF всех sql-файлов миграции для указанной ветки.
Второй элемент - это ARRAYREF всех скриптов миграций для указанной ветки.
Файлы собирает по директориям ./migrations/before_release и ./migrations/after_release

Для забора используется ручка L<https://developer.github.com/v3/git/trees/>

    my ($sql, $scripts) = get_branch_migrations(
        branch => 'PI-1234_feature_branch',
    );

Пример $sql:

    [
        'after_release/release_103.sql',
        'before_release/PI-5732_last_version.sql',
        ...
    ]

Пример $scripts:

    [
        'after_release/PI-7640_change_public_id.pl',
    ]

В ответе файлы отсортированы по имени.

=cut

sub get_branch_migrations {
    my (%params) = @_;

    my @sql = ();
    my @scripts = ();

    # TODO remove or adapt to arcadia
    # my $url = "https://github.yandex-team.ru/api/v3/repos/partner/partner2/git/trees/$params{branch}";
    # my $data = _get_json_data_from_url($url);
    #
    # my $migration_sha = [ grep { $_->{path} eq 'migrations' } @{$data->{tree}} ]->[0]->{sha};
    #
    # $url = "https://github.yandex-team.ru/api/v3/repos/partner/partner2/git/trees/$migration_sha";
    # $data = _get_json_data_from_url($url);
    #
    # my %stage2sha = map { $_->{path} => $_->{sha} } @{$data->{tree}};
    #
    # foreach my $stage (keys %stage2sha) {
    #     my $sha = $stage2sha{$stage};
    #
    #     $url = "https://github.yandex-team.ru/api/v3/repos/partner/partner2/git/trees/$sha";
    #     $data = _get_json_data_from_url($url);
    #
    #     my @files = grep { $_->{type} eq 'blob' } @{$data->{tree}};
    #
    #     foreach my $file (@files) {
    #         my $file_name = $stage . '/' . $file->{path};
    #         if ($file->{path} =~ /\.sql\z/) {
    #             push @sql, $file_name;
    #         } elsif ($file->{mode} eq '100755') {
    #             push @scripts, $file_name;
    #         }
    #     }
    # }

    return [sort { by_stage($a, $b) } @sql], [sort { by_stage($a, $b) } @scripts];
}

sub change_docker_compose_yml {
    my ($content, $https_port, $adfox_host, $ubuntu_version) = @_;

    my $http_port = $https_port - 400;

    my $rosetta_server_port = $http_port + 10_000;
    my $nodejs_port = $http_port + 20_000;

    my $blackbox_mock_port = $http_port + 14_000;
    my $bs_mock_port = $http_port + 13_000;
    my $db_port = $http_port + 100;
    my $backend_http_port = $http_port + 11_000;
    my $backend_https_port = $http_port + 9_000;

    $content =~ s/"443:443"/"$https_port:443"/;
    $content =~ s/"1080:80"/"$blackbox_mock_port:80"/;
    $content =~ s/"1082:80"/"$bs_mock_port:80"/;
    $content =~ s/"3306:3306"/"$db_port:3306"/;
    $content =~ s/"8066:8066"/"$backend_http_port:8066"/;
    $content =~ s/"8466:8466"/"$backend_https_port:8466"/;
    $content =~ s/"18066:18066"/"$rosetta_server_port:18066"/;
    $content =~ s/"8000:8000"/"$nodejs_port:8000"/;

    $ubuntu_version //= $UBUNTU_VERSION_BIONIC;
    $content =~ s/UBUNTU_VERSION/$ubuntu_version/;

    my $hostname = `hostname`;
    chomp($hostname);
    $content =~ s/(hostname:).*?-(backend)/$1 ${hostname}-${https_port}-$2/;

    my $env =
        "\n"
        . "  environment:\n"
        . "    - PARTNER2_HOST=$ENV{BETA_HOST}\n"
        . "    - PARTNER2_HTTPS_PORT=$https_port\n"
        ;

    $content =~ s/(backend:\s+?build: \.)\n/$1$env/;

    $adfox_host //= '';
    my $frontendEnv =
        "\n"
        . "    - ADFOX_HOST=$adfox_host";

    $content =~ s/(frontend_node:(\s|.)*?environment:)/$1$frontendEnv/;

    return $content;
}

sub change_frontend_dockerfile {
    my ($content, $sandbox_api_token, $arc_token) = @_;

    $content =~ s/ARG SANDBOX_API_TOKEN/ARG SANDBOX_API_TOKEN=$sandbox_api_token/;
    $content =~ s/ARG ARC_TOKEN/ARG ARC_TOKEN=$arc_token/;

    return $content;
}

sub switch_to_database {
    my (%opts) = @_;

    my $db = delete $opts{db};
    my $dir = delete $opts{dir};

    if ($db eq 'docker') {
        # nothing should be done
    } elsif ($db eq 'dev') {
        run_cmd("cp $dir/configs/dev/DatabaseConfig.json $dir/configs/docker/DatabaseConfig.json");
        _rm_db_from_docker_compose_yml($dir);
        _add_database_config_file_to_docker_compose_yml($dir);
    } elsif ($db eq 'ts') {
        run_cmd("cp $dir/configs/test/DatabaseConfig.json $dir/configs/docker/DatabaseConfig.json");
        _change_db_config("$dir/configs/docker/DatabaseConfig.json", password => get_secret('partner2-mysql-password'));
        _rm_db_from_docker_compose_yml($dir);
        _add_database_config_file_to_docker_compose_yml($dir);
    } else {
        die "Unknown database '$db'";
    }
}

sub switch_to_database_api_3 {
    my (%opts) = @_;

    my $db = delete $opts{db};
    my $dir = delete $opts{dir};

    if ($db->{type} eq 'preset') {
        if ($db->{preset_id} eq 'dev') {
            my $file_name = "$dir/configs/docker/DatabaseConfig.json";
            run_cmd("cp $dir/configs/dev/DatabaseConfig.json $file_name");
            _change_db_config(
                $file_name,
                host     => 'c-mdblecqg23267fere3rs.rw.db.yandex.net',
                port     => 3306,
            );

            _rm_db_from_docker_compose_yml($dir);
            _add_database_config_file_to_docker_compose_yml($dir);
        } elsif ($db->{preset_id} eq 'ts') {
            run_cmd("cp $dir/configs/test/DatabaseConfig.json $dir/configs/docker/DatabaseConfig.json");
            _change_db_config(
                "$dir/configs/docker/DatabaseConfig.json",
                host     => 'c-mdbc9o3v21cvbi9bpda6.rw.db.yandex.net',
                password => get_secret('partner2-mysql-password'),
                port     => 3306,
            );
            _rm_db_from_docker_compose_yml($dir);
            _add_database_config_file_to_docker_compose_yml($dir);
        } else {
            die "Unknown db preset_id";
        }
    } elsif ($db->{type} eq 'docker') {

        change_file_content(
            file_name => "$dir/$DOCKER_COMPOSE_YML_PATH",
            changer => sub {
                my ($content, $docker_image) = @_;

                $content =~ s(registry\.partner\.yandex-team\.ru:443/partner2-db-[0-9-]+)($docker_image);

                return $content;
            },
            opts => [$db->{docker_image}],
        );

        _add_database_config_file_to_docker_compose_yml($dir, 0);
    } else {
        die "Unknown db type";
    }
}

sub _change_db_config {
    my ($file_name, %opts) = @_;

    my $config = decode_json scalar read_file $file_name;

    $config->{partnerdb2} = {%{$config->{partnerdb2}}, %opts};

    write_file(
        $file_name,
        {
            binmode => ':utf8',
        },
        to_pretty_json($config),
    );
}

sub switch_blackbox {
    my ($content, $url) = @_;

    my $data = JSON::PP->new()->decode($content);
    $data->{api_blackbox}->{URL} = $url;

    return to_pretty_json $data;
}

sub switch_yacotools {
    my ($content, $url) = @_;

    my $data = JSON::PP->new()->decode($content);
    $data->{api_yacotools}->{url} = $url;

    return to_pretty_json $data;
}

sub switch_bs {
    my ($content, $url) = @_;

    my $data = JSON::PP->new()->decode($content);
    $data->{api_bk}->{url} = $url;

    return to_pretty_json $data;
}

sub switch_clickhouse_host {
    my ($content) = @_;

    my $data = JSON::PP->new()->decode($content);

    my $host = service_discovery_get_active_hostname($BACKEND_SERVICES_ENDPOINT, $data->{clickhouse_db}{balancer_port});

    $data->{clickhouse_db}{host} = $host;

    return to_pretty_json $data;
}

sub _rm_db_from_docker_compose_yml {
    my ($dir) = @_;

    my $file_name = "$dir/$DOCKER_COMPOSE_YML_PATH";

    if (!-f $file_name) {
        die "No file '$file_name'";
    }

    my $content = read_file $file_name;

    $content =~ s/ +- db\n//;
    $content =~ s/db:.*:3306"\n\n//ms;  # " just sintax color correction

    write_file(
        $file_name,
        {
            binmode => ':utf8',
        },
        $content,
    );

    return 1;
}

sub _add_database_config_file_to_docker_compose_yml {
    my ($dir, $remove_db_from_links) = @_;

    $remove_db_from_links //= 1;

    my $file_name = "$dir/$DOCKER_COMPOSE_YML_PATH";

    if (!-f $file_name) {
        die "No file '$file_name'";
    }

    my $content = read_file $file_name;

    if ($remove_db_from_links) {
        $content =~ s/ +- db\n//;
    }

    $content =~ s{(backend.*volumes:\n)}{$1    - ./configs/docker/DatabaseConfig.json:/home/partner/beta.8066/lib/DatabaseConfig.json\n}ms;

    write_file(
        $file_name,
        {
            binmode => ':utf8',
        },
        $content,
    );

    return 1;
}

sub _get_beta_path {
    my ($port) = @_;
    return sprintf $TEMPL_BETA_PATH, $port;
}

sub _get_beta_json_path {
    my ($port) = @_;
    return sprintf $TEMPL_BETA_JSON_PATH, $port;
}

sub _is_beta_free {
    my ($port) = @_;

    my $is_free = ( -f _get_beta_json_path($port) || -d _get_beta_path($port) )
        ? 0
        : 1;

    return $is_free;
}

sub _is_valid_port {
    my ($port) = @_;

    die "port is not defined" if not defined $port;

    my $is_valid = grep {$_ eq $port} @PORTS;

    die "port is not valid" if not $is_valid;

    return 1;
}

sub _get_yandex_blackbox {
    $ENV{TVMTOOL_LOCAL_AUTHTOKEN} = `cat /var/lib/tvmtool/local.auth`;
    chomp $ENV{TVMTOOL_LOCAL_AUTHTOKEN};

    my $yb = Yandex::Blackbox->new(
        url => 'http://blackbox.yandex-team.ru/blackbox',
        x_ya_service_ticket => get_x_ya_service_ticket($TVM_BLACKBOX_ID),
    );
    return $yb;
}

sub _get_json_data_from_url {
    my ($url) = @_;

    my $response = HTTP::Tiny->new()->get($url);

    my $data;
    if ($response->{status} eq '200') {
        $data = eval{ decode_json( $response->{content} ) };
        if ($@){
            warn $@;
            Carp::cluck(sprintf 'Malformed JSON string from url "%s"', $url);
        }
    } else {
        Carp::cluck(sprintf 'Got non 200 status from url "%s"', $url );
    }

    return ($response, $data);
}

sub _get_beta_info {
    my ($port) = @_;

    my $github_info = get_cached_github_info();

    $github_info->{partner2}->{branches} = { map { %{$_}} @{$github_info->{partner2}->{branches}} };
    $github_info->{partner2}->{tags} = { map { %{$_}} @{$github_info->{partner2}->{tags}} };
    $github_info->{yharnam}->{branches} = { map { %{$_}} @{$github_info->{yharnam}->{branches}} };
    $github_info->{yharnam}->{tags} = { map { %{$_}} @{$github_info->{yharnam}->{tags}} };

    my $beta = {
        port => $port,
        status => _is_beta_free($port) ? 'free' : 'occupied',
    };

    my $file_name = _get_beta_json_path($port);
    if (-f $file_name) {
        my $content = read_file $file_name;
        $beta = decode_json $content;

        foreach my $side (qw(backend frontend)) {
            my $repo;
            if ($side eq 'backend') {
                $repo = 'partner2';
            } elsif ($side eq 'frontend') {
                $repo = 'yharnam';
            } else {
                die 'this should never happen';
            }

            if (
                exists $github_info->{$repo}->{branches}->{$beta->{$side}}
                && $github_info->{$repo}->{branches}->{$beta->{$side}} eq $beta->{"${side}_sha1"}
                ) {
                $beta->{"${side}_status"} = 'actual';
            } elsif (
                exists $github_info->{$repo}->{branches}->{$beta->{$side}}
                && $github_info->{$repo}->{branches}->{$beta->{$side}} ne $beta->{"${side}_sha1"}
                ) {
                $beta->{"${side}_status"} = 'outdated';
            } elsif (
                exists $github_info->{$repo}->{tags}->{$beta->{$side}}
                && $github_info->{$repo}->{tags}->{$beta->{$side}} eq $beta->{"${side}_sha1"}
                ) {
                $beta->{"${side}_status"} = 'actual';
            } elsif (
                exists $github_info->{$repo}->{tags}->{$beta->{$side}}
                && $github_info->{$repo}->{tags}->{$beta->{$side}} ne $beta->{"${side}_sha1"}
                ) {
                $beta->{"${side}_status"} = 'outdated';
            } else {
                $beta->{"${side}_status"} = 'deleted';
            }

        }
    }

    return $beta;
}

# 0 - превышен лимит количества бет
# 1 - превышения нет
sub _check_betas_limit {
    my ($port) = @_;

    # Нет лимита при создании канонической и скриншутерных бет
    return 1 if $port >= 8597;

    my $count = 0;
    foreach (@PORTS) {
        $count++ unless _is_beta_free($_);
    }

    return($count > $BETAS_LIMIT ? 0 : 1);
}


########################################
#
# API 3
#
########################################

## Ручки про работу с бетами


get '/api/3/betas' => sub {
    my $c = shift;

    my $betas;

    foreach my $port (@PORTS) {
        push @{$betas}, _get_beta_info($port);
    }

    $c->res->headers->content_type('application/json;charset=UTF-8');

    return $c->render(
        text => to_pretty_json $betas,
    );
};

get '/api/3/betas/:port' => sub {
    my $c = shift;

    my $port = $c->stash('port');

    eval {
        _is_valid_port($port);
    };

    if (not $@) {
        my $beta = _get_beta_info($port);

        $c->res->headers->content_type('application/json;charset=UTF-8');

        return $c->render(
            text => to_pretty_json $beta,
        );
    } else {
        return $c->render(
            json => {
                status => JSON::PP::false,
                error_message => 'Incorrect port',
            },
            status => 400,
        );
    }

};

sub get_fail_answer {
    my ($description, $status) = @_;

    return (
        json => {
            status => 'fail',
            description => $description,
        },
        status => $status,
    );
}

put '/api/3/betas/:port' => sub {
    my $c = shift;

    my $now = Moment->now();

    my $port = $c->stash('port');

    my $is_valid_port = eval { _is_valid_port($port) };
    return $c->render(get_fail_answer('Invalid port', 400)) if not $is_valid_port;

    return $c->render(get_fail_answer('Port is already in use', 400)) unless _is_beta_free($port);

    return $c->render(get_fail_answer('Count of betas limit reached', 400)) unless _check_betas_limit($port);

    my $parsed_body   = eval { decode_json $c->req->body };

    return $c->render(get_fail_answer('Invalid json in beta configuration', 400)) if $@;

    $parsed_body->{ubuntu_version} //= $UBUNTU_VERSION_BIONIC;
    return $c->render(get_fail_answer('Unsupported ubuntu_version', 400)) unless ($parsed_body->{ubuntu_version} eq $UBUNTU_VERSION_PRECISE || $parsed_body->{ubuntu_version} eq $UBUNTU_VERSION_BIONIC);

    my (
        $is_valid_user_input,
        $user_input_error_description,
    ) = is_valid_user_input_for_beta_creation_version_3($parsed_body);
    return $c->render(get_fail_answer($user_input_error_description, 400)) if not $is_valid_user_input;

    my $is_error = !( $is_valid_port && $parsed_body && $is_valid_user_input );
    return $c->render(get_fail_answer('Something wrong in input', 400)) if $is_error;

    my $dir = _get_beta_path($port);
    sleep 1 + int rand 5;
    return $c->render(get_fail_answer('Port is already in use', 400)) unless _is_beta_free($port);
    mkdir $dir;

    my $log_file = "/local/creator/meta/$port.log";

    my $beta_log = sub {
        my ($line) = @_;

        warn scalar(localtime(time)) . " Beta $port log: $line\n";
        write_file(
            $log_file,
            {
                append => 1,
                binmode => ':utf8',
                err_mode => 'carp',
            },
            $line, "\n"
        );
    };

    if ( eval {
        warn scalar(localtime(time)) . " Beta $port start process\n";

        my $created_by = $c->stash('login');

        my $backend_branch = $parsed_body->{backend};
        my $backend_sha1 = get_branch_sha1(branch => $backend_branch, log => $beta_log);
        $beta_log->(sprintf('For backend branch "%s" found commit "%s"', $backend_branch, $backend_sha1));
        my $frontend_branch = $parsed_body->{frontend};
        my $frontend_sha1 = get_branch_sha1(branch => $frontend_branch, log => $beta_log);
        $beta_log->(sprintf('For frontend branch "%s" found commit "%s"', $frontend_branch, $frontend_sha1));

        $beta_log->('Downloading perl backend');
        my $arc_token = get_secret('arc-token');
        $ENV{'ARC_TOKEN'} = $arc_token;
        run_cmd("cd $dir; mkdir arcadia");
        $dir .= "/arcadia";
        run_cmd("cd $dir; arc init --bare");
        run_cmd("cd $dir; arc --version");
        run_cmd("cd $dir; arc export $backend_branch '$ARCADIA_PERL_PATH' --to .");
        $beta_log->("Perl backend is downloaded into $dir/$ARCADIA_PERL_PATH");
        run_cmd("cd $dir; arc export $frontend_branch '$ARCADIA_FRONTEND_PATH' --to .");
        $beta_log->("Frontend is downloaded into $dir/$ARCADIA_FRONTEND_PATH");
        $dir .= "/$ARCADIA_PERL_PATH";

        $beta_log->('Prepare meta-file');
        write_file(
            _get_beta_json_path($port),
            {
                binmode => ':utf8',
            },
            to_pretty_json({
                backend        => $backend_branch,
                backend_sha1   => $backend_sha1,
                blackbox       => $parsed_body->{blackbox},
                bs             => $parsed_body->{bs},
                comment        => $parsed_body->{comment} // '',
                created        => $now->get_iso_string(),
                created_by     => $created_by // 'robot-partner',
                db             => $parsed_body->{db},
                frontend       => $frontend_branch,
                frontend_sha1  => $frontend_sha1,
                java           => $parsed_body->{java},
                java_sha1      => get_branch_sha1(branch => $parsed_body->{java}, log => $beta_log),
                port           => $port + 0,
                status         => 'building',
                ttl            => $parsed_body->{ttl},
                ubuntu_version => $parsed_body->{ubuntu_version} // 'bionic',
                url            => sprintf("https://%s:%d", $BETA_HOST, $port),
                yacotools      => $parsed_body->{yacotools},
            })
        );

        $beta_log->('Prepare docker-compose');
        change_file_content(
            file_name => "$dir/$DOCKER_COMPOSE_YML_PATH",
            changer   => \&change_docker_compose_yml,
            opts      => [ $port, $parsed_body->{adfox_host}->{url}, $parsed_body->{ubuntu_version} ],
        );

        $beta_log->('Prepare DB');
        switch_to_database_api_3(
            dir => $dir,
            db  => $parsed_body->{db},
        );

        $beta_log->('Prepare fronend docker-file (1)');
        change_file_content(
            file_name => $dir . '/configs/docker/frontend_node/Dockerfile',
            changer   => \&switch_frontend,
            opts      => [ $frontend_sha1 ],
        );

        $beta_log->('Prepare fronend docker-file (2)');
        change_file_content(
            file_name => $dir . '/configs/docker/frontend_node/Dockerfile',
            changer   => \&change_frontend_dockerfile,
            opts      => [ get_secret('sandbox-api-token'), get_secret('arc-token') ],
        );

        $beta_log->('Prepare blackbox');
        change_file_content(
            file_name => $dir . '/configs/docker/Application.json',
            changer   => \&switch_blackbox,
            opts      => [ $parsed_body->{blackbox}->{url} ],
        );

        $beta_log->('Prepare yacotools');
        change_file_content(
            file_name => $dir . '/configs/docker/Application.json',
            changer   => \&switch_yacotools,
            opts      => [ $parsed_body->{yacotools}->{url} ],
        );

        $beta_log->('Prepare BK');
        change_file_content(
            file_name => $dir . '/configs/docker/Application.json',
            changer   => \&switch_bs,
            opts      => [ $parsed_body->{bs}->{url} ],
        );

        $beta_log->('Prepare CH');
        change_file_content(
            file_name => $dir . '/configs/docker/Application.json',
            changer   => \&switch_clickhouse_host,
            opts      => [],
        );

        $beta_log->('Prepare after-file');
        create_after_file(
            file_name => $dir . '/after',
        );

        $beta_log->('Login to registry');
        docker_registry_login();

        $beta_log->('Make docker_restart');
        my $build_java_cmd = sprintf('JAVA_BRANCH=%s make build_java', $parsed_body->{java} // '');
        run_cmd("cd $dir; sh -c '$build_java_cmd && BETA_PORT=$port make docker_restart; ./after --exit_status=\$?' 2>>$log_file >>$log_file &");

        return 1;
    } ) {
        return $c->render(
            json => {
                status => 'ok',
            },
        );
    } else {
        my $err = $@ // 'Unknown error';
        warn scalar(localtime(time)) . " Beta $port failed: $err\n";
        return $c->render(get_fail_answer($err, 500));
    }

};

del '/api/3/betas/:port' => sub {
    my $c = shift;

    my $port = $c->stash('port');
    _is_valid_port($port);

    my $dir = _get_beta_path($port);
    my $docker_dir = $dir . "/arcadia/$ARCADIA_PERL_PATH";

    if (-d $dir) {
        my $beta_containers_count = `docker ps -a -f "name=$port" -q | wc -l`;
        if (0 < $beta_containers_count) {
            run_cmd("cd $docker_dir; docker-compose -p $port -f $DOCKER_COMPOSE_YML_PATH stop && COMPOSE_HTTP_TIMEOUT=300 docker-compose -p $port -f $DOCKER_COMPOSE_YML_PATH rm --force");
        }
    }

    run_cmd("rm -rf $dir");
    run_cmd("rm -rf " . _get_beta_json_path($port) );
    run_cmd("rm -rf /local/creator/meta/$port.log");

    return $c->render(
        json => {
            status => 'ok',
        },
    );
};

put '/api/3/betas/:port/action/:action' => sub {
    my $c = shift;

    my $port = $c->stash('port');
    my $action = $c->stash('action');

    eval {
        _is_valid_port($port);
    };

    if ($@) {
        return $c->render(
            json => {
                status => JSON::PP::false,
                error_message => 'Incorrect port',
            },
            status => 400,
        );
    }

    my $file_name = _get_beta_json_path($port);

    if (not -e $file_name) {
        return $c->reply->not_found();
    }

    if (not grep { $action eq $_} qw(start stop)) {
        return $c->reply->not_found();
    }

    my $coder = JSON::PP->new();

    # Обработка ошибок завершена, пошел настоящий код
    my $data = $coder->decode(scalar read_file(
        $file_name,
        {
            binmode => ':utf8',
        },
    ));

    my %action2status = (
        stop  => 'stopped',
        start => 'running',
    );

    $data->{status} = $action2status{$action};

    write_file(
        $file_name,
        {
            binmode => ':utf8',
        },
        to_pretty_json($data),
    );

    my %action2cmd = (
        stop => "docker-compose -p $port -f $DOCKER_COMPOSE_YML_PATH stop",
        start => "docker-compose -p $port -f $DOCKER_COMPOSE_YML_PATH start",
    );

    my $dir = _get_beta_path($port) . "arcadia/$ARCADIA_PERL_PATH";

    run_cmd("cd $dir; $action2cmd{$action}");

    return $c->render(
        json => {
            status => 'ok',
        },
    );

};

get '/api/3/betas/log/:port' => sub {
    my $c = shift;

    my $port = $c->stash('port');
    _is_valid_port($port);

    my $file_name = "/local/creator/meta/$port.log";

    if (-e $file_name) {
        my $content = read_file $file_name;
        my $result = get_last_lines_from_content(
            content => $content,
            lines   => $c->param('tail'),
        );
        $c->render(text => $result, format => 'txt');
    } else {
        return $c->reply->not_found();
    }
};

## Ручки про взаимодействие с github

# NOTE! for debug purpose only
get '/api/3/github/debug_cached_github_info' => sub {
    my $c = shift;

    my $github_info = get_cached_github_info();

    return $c->render(
        json => $github_info,
    );
};

get '/api/3/github/cached_tags' => sub {
    my $c = shift;

    my @tags = get_tags_from_github_cache( $c->param('repo') );

    return $c->render(
        json => \@tags,
    );
};

get '/api/3/github/cached_branches' => sub {
    my $c = shift;

    my @tags = get_branches_from_github_cache( $c->param('repo') );

    return $c->render(
        json => \@tags,
    );
};

get '/api/3/github/cached_migrationsql_files' => sub {
    my $c = shift;
    my $branch = $c->param('branch') // 'master';

    my $github_info = get_cached_github_info();

    my $files = $github_info->{migrations}->{$branch} // [];

    return $c->render(
        json => $files,
    );
};

get '/api/3/github/cached_sql_migrations' => sub {
    my $c = shift;
    my $branch = $c->param('branch');

    if (defined $branch) {
        my $github_info = get_cached_github_info();

        if (exists $github_info->{migrations}->{$branch}) {
            return $c->render(
                json => $github_info->{migrations}->{$branch},
            );
        } else {
            return $c->render(
                json => {
                    status => 'fail',
                },
                status => 404,
            );
        }
    } else {
        return $c->render(
            json => {
                status => 'fail',
            },
            status => 400,
        );
    }
};

get '/api/3/github/cached_script_migrations' => sub {
    my $c = shift;
    my $branch = $c->param('branch');

    if (defined $branch) {
        my $github_info = get_cached_github_info();

        if (exists $github_info->{migrations_scripts}->{$branch}) {
            return $c->render(
                json => $github_info->{migrations_scripts}->{$branch},
            );
        } else {
            return $c->render(
                json => {
                    status => 'fail',
                },
                status => 404,
            );
        }
    } else {
        return $c->render(
            json => {
                status => 'fail',
            },
            status => 400,
        );
    }
};

get '/api/3/github/cached_latest_stable_tag' => sub {
    my $c = shift;
    my $branch = $c->param('branch');

    if (defined $branch) {
        my $github_info = get_cached_github_info();
        my $tags = $github_info->{latest_stable_tags};

        if (exists $tags->{$branch} && $tags->{$branch} ne 'too_old') {
            return $c->render(
                json => {
                    status => 'ok',
                    tag => $github_info->{latest_stable_tags}->{$branch},
                },
            );
        } else {
            return $c->render(
                msg => 'latest stable tag not found',
                status => 404,
            );
        }
    } else {
        return $c->render(
            status => 400,
            msg => 'cache file not found',
        );
    }
};

post '/api/3/github/update' => sub {
    my $c = shift;

    my $branch = 'master';
    my $repository = 'partner2';

    eval {
        my $parsed_body = decode_json $c->req->body;
        if ($parsed_body->{ref} =~ /refs\/heads\/(.*)/) {
            $branch = $1;
        }

        $repository = $parsed_body->{repository}->{name};
    };

    my @partner2_branches = get_all_branches_from_github(
        owner => 'partner',
        repo  => 'partner2',
    );

    my @partner2_tags = get_some_tags_from_github(
        owner => 'partner',
        repo  => 'partner2',
    );

    my @yharnam_branches = get_all_branches_from_github(
        owner => 'partner',
        repo  => 'yharnam',
    );

    my @yharnam_tags = get_some_tags_from_github(
        owner => 'partner',
        repo  => 'yharnam',
    );

    my $github_info = get_cached_github_info();
    my $cached_latest_stable_tags = $github_info->{latest_stable_tags} // {};

    if ($repository eq 'partner2') {
        # Просто берем ключ из первого элемента в списке тагов [ {"2.18.2613": "5270ee087 ..."}, ... ]
        my $latest_tag = @partner2_tags ? ( keys %{ $partner2_tags[0] } )[0] : undef;

        my $partner2_perl_version = get_partner2_production_version()->{'s'};

        # NOTE! это просто заглущка, а не рельно последний таг в ветке
        # TODO: получать все версии из Deploy для prod стейджа и искать их по комитам ветки (долго и сложно)
        $cached_latest_stable_tags->{$branch} = $partner2_perl_version && ( $partner2_perl_version lt $latest_tag)
            ? $partner2_perl_version
            : $latest_tag;
    }

    write_file(
        $GITHUB_CACHE_FILE_NAME,
        {
            binmode => ':utf8',
        },
        to_pretty_json({
            partner2 => {
                tags     => \@partner2_tags,
                branches => \@partner2_branches,
            },
            yharnam  => {
                tags     => \@yharnam_tags,
                branches => \@yharnam_branches,
            },
            latest_stable_tags => $cached_latest_stable_tags,
        })
    );

    return $c->render(
        json => {
            status => 'ok',
        },
    );
};

## Другие ручки

get '/api/3/alive' => sub {
    my $c = shift;

    return $c->render(
        json => {
            status => 'ok',
        },
    );
};

post '/api/3/login' => sub {
    my $c = shift;

    return $c->render(
        json => {
            login => $c->stash('login'),
        },
    );
};

get '/api/3/login' => sub {
    my $c = shift;

    return $c->render(
        json => {
            login => $c->stash('login'),
        },
    );
};

get '/api/3/cached_docker_db_images' => sub {
    my $c = shift;

    my $file_name = '/local/creator/meta/docker_db_images.json';
    my @images;

    if (-e $file_name) {
        @images = grep { $_ !~ /-null-/ } @{decode_json scalar read_file $file_name};
    }

    $c->res->headers->content_type('application/json;charset=UTF-8');

    return $c->render(
        text => to_pretty_json \@images,
    );
};

get '/api/3/swagger.json' => sub {
    my $c = shift;

    $c->res->headers->content_type('application/json;charset=UTF-8');

    return $c->render(
        text => to_pretty_json {
            swagger  => "2.0",
            schemes  => [
                (($BETA_HOST eq 'docker') ? 'http' : 'https')
            ],
            basePath => '/api/3/',
            tags     => [
                {
                    name        => "beta",
                    description => "Ручки про работу с бетами",
                },
                {
                    name        => "github",
                    description => "Ручки про взаимодействие с github",
                },
                {
                    name        => "smth",
                    description => "Другие ручки",
                },
            ],
            paths => {
                'betas' => {
                    get => {
                        summary     => "Возвращает JSON массив с информацией обо всех бетах",
                        description => "
Возвращает JSON с массивом объектов к которых содержится информация про все беты.

Поля backend_status и frontend_status могут принимать значения: actual,
outdated, deleted. Для того чтобы выяснить значение этих полей используются
данные из кеша, который обновляется с помощью ручки POST github/update.
                        ",
                        tags        => [ 'beta' ],
                        responses   => {
                            200 => {
                                schema => {
                                    type  => "array",
                                    items => {
                                        '$ref' => "#/definitions/BetaResponse"
                                    }
                                }
                            },
                        },
                    },
                },
                'betas/{betaPort}' => {
                    get => {
                        summary     => "Возвращает JSON объект с информацией о бете",
                        description => "
Возвращает ту же информацию что и ручка GET betas, но только про одну указанную бету.
                        ",
                        tags        => [ 'beta' ],
                        responses   => {
                            200 => {
                            },
                            400 => {
                            },
                        },
                        parameters  => [
                            { '$ref' => "#/definitions/betaPort" },
                        ],
                    },
                    put => {
                        summary     => "Создает новую бету на указанном порту",
                        tags        => [ 'beta' ],
                        description => "
С помощью этой ручки происходит запуск процесса сборки беты. После того
как эта ручка ответила 200 процесс сборки продолжает идти. С помощью ручки
GET betas/log/PORT можно следить за процессом сборки.
                        ",
                        parameters  => [
                            { '$ref' => "#/definitions/betaPort" },
                            {
                                in       => 'body',
                                name     => 'body',
                                required => JSON::PP::true,
                                schema   => {
                                    '$ref' => "#/definitions/BetaRequest"
                                }
                            }
                        ],
                        responses => {
                            200 => {
                                description => "Сборка беты успешно запущена.",
                                schema      => {
                                    '$ref' => "#/definitions/OkResponse"
                                }
                            },
                            400 => {
                                description => "Сборка беты не запущена, так как указаны неправильные данные.",
                                schema      => {
                                    '$ref' => "#/definitions/FailResponse"
                                }
                            },
                        },
                    },
                    delete => {
                        summary     => "Удаляет бету",
                        description => "
Ручка удаляет бету. Ручка возвращает 200 в том случае если бету успешно
удалена. Ручка так же возврщает 200 если на указанном порту уже не было
никакой беты.
                        ",
                        tags        => [ 'beta' ],
                        responses   => {
                            200 => {
                                description => "Бета успешно удалена.",
                                schema      => {
                                    '$ref' => "#/definitions/OkResponse"
                                }
                            },
                        },
                        parameters  => [
                            { '$ref' => "#/definitions/betaPort" },
                        ],
                    },
                },
                'betas/log/{betaPort}' => {
                    get => {
                        summary     => "Возвращает plain text в котором содержится лог сборки беты",
                        description => "
С помощью ручки PUT betas/PORT начинается процесс сборки беты. Этот
процесс идет весьма продолжительное время. С помощью этой ручки можно
получить текст с логом того как идет сборка. По мере сборке беты этот
лог становится все больше.
                        ",
                        tags        => [ 'beta' ],
                        parameters  => [
                            { '$ref' => "#/definitions/betaPort" },
                            {
                                name        => 'tail',
                                in          => 'query',
                                description => 'Количество строк',
                                required    => JSON::PP::false,
                                type        => 'integer',
                            },
                        ],
                    },
                },

                'betas/{betaPort}/action/{action}' => {
                    put => {
                        summary     => "Останавливает или запускает бету",
                        description => "
С помощью этой ручки можно остановить или запустить бету.
Алгоритм работы такой — сначала бета создается. Ее статус сначала building,
потом становится build_fail или running. Из running с помощью stop
бету можно перевести в stopped, а с помощью start можно вернуть ее обратно
в running.

Во време действия stop выполняется полная остановка всех микросервисов
которые были созданы для этой беты.
",
                        tags        => [ 'beta' ],
                        parameters  => [
                            { '$ref' => "#/definitions/betaPort" },
                            {
                                name        => "action",
                                in          => "path",
                                description => "Действие",
                                required    => JSON::PP::true,
                                type        => "string",
                                items       => {
                                    type => "string",
                                    enum => [
                                        "stop",
                                        "start",
                                    ],
                                },
                            },
                        ],
                        responses   => {
                            200 => {
                                description => "Действие успешно выполнено",
                            },
                            400 => {
                                description => "Указан неверный порт",
                            },
                            404 => {
                                description => "На указанном порту нет беты",
                            },
                        },
                    },
                },


                'github/cached_tags'     => {
                    get => {
                        summary     => 'Возвращает теги указанного github репозитория',
                        tags        => [ 'github' ],
                        description => "
Ручке нужно обязательно передать параметр repo.
Значение этого параметра может быть либо partner2, либо yharnam.
Ручка вернет json с массивом тегов, которые есть в указанном репозитории.
Ручка возвращает не более 30 тегов.
Ручка использует данные из кеша, который обновляется с помощью ручки POST github/update.
",
                        parameters  => [
                            {
                                name        => "repo",
                                in          => "query",
                                description => "имя репозитория",
                                required    => JSON::PP::true,
                                type        => "string",
                                items       => {
                                    type => "string",
                                    enum => [
                                        "partner2",
                                        "yharnam",
                                    ],
                                },
                            },
                        ],
                    }
                },
                'github/cached_branches' => {
                    get => {
                        summary     => 'Возвращает ветки указанного github репозитория',
                        tags        => [ 'github' ],
                        description => "
Ручке нужно обязательно передать параметр repo.
Значение этого параметра может быть либо partner2, либо yharnam.
Ручка вернет json с массивом веток, которые есть в указанном репозитории.
Ручка возвращает все ветки. Ручка использует данные из кеша, который обновляется с помощью ручки POST github/update.
",
                        parameters  => [
                            {
                                name        => "repo",
                                in          => "query",
                                description => "имя репозитория",
                                required    => JSON::PP::true,
                                type        => "string",
                                items       => {
                                    type => "string",
                                    enum => [
                                        "partner2",
                                        "yharnam",
                                    ],
                                },
                            },
                        ],
                    }
                },

                'github/cached_sql_migrations'    => {
                    get => {
                        summary     => 'Возвращает список файлов sql миграций для ветки',
                        tags        => [ 'github' ],
                        description => "
Ручке нужно обязательно передать параметр branch.
Значение этого параметра — имя ветки в репозитории partner2.
Ручка вернет json с массивом файлов с sql миграциями, которые есть в указанной ветке.
Ручка использует данные из кеша, который обновляется с помощью ручки POST github/update.
",
                        parameters  => [
                            {
                                name        => "branch",
                                in          => "query",
                                description => "имя ветки",
                                required    => JSON::PP::true,
                                type        => "string",
                            },
                        ],

                        responses   => {
                            200 => {
                                description => "Действие успешно выполнено",
                            },
                            400 => {
                                description => "Ветка не указана",
                            },
                            404 => {
                                description => "Нет информации о sql миграциях для указанной ветки",
                            },
                        },
                    }
                },

                'github/cached_script_migrations' => {
                    get => {
                        summary     => 'Возвращает список файлов скриптов миграций для ветки',
                        tags        => [ 'github' ],
                        description => "
Ручке нужно обязательно передать параметр branch.
Значение этого параметра — имя ветки в репозитории partner2.
Ручка вернет json с массивом файлов со скриптами миграций, которые есть в указанной ветке.
Ручка использует данные из кеша, который обновляется с помощью ручки POST github/update.
",
                        parameters  => [
                            {
                                name        => "branch",
                                in          => "query",
                                description => "имя ветки",
                                required    => JSON::PP::true,
                                type        => "string",
                            },
                        ],

                        responses   => {
                            200 => {
                                description => "Действие успешно выполнено",
                            },
                            400 => {
                                description => "Ветка не указана",
                            },
                            404 => {
                                description => "Нет информации о скриптах миграций для указанной ветки",
                            },
                        },
                    }
                },


                'github/cached_migrationsql_files' => {
                    get => {
                        summary     => 'Возвращает список файлов миграции в репозитории partner2',
                        deprecated  => JSON::PP::true,
                        tags        => [ 'github' ],
                        description => "

WARNING. Ручка deprecated. Вместо этой ручки нужно использовать 'GET github/cached_sql_migrations'. Ручка работает, но в следующей
мажорной версии будут убрана.

Ручке в параметре branch нужно передать имя бранча для которого нужно вернуть существующие миграции. Если параметр не передан, то берутся миграции из ветки master
Ручка вернет json с массивом файлов, которые находятся в директориях migrations/before_release и migrations/after_release репозитория partner2 в соответствующей ветке.
Ручка использует данные из кеша, который обновляется с помощью ручки POST github/update.
",
                    }
                },

                'github/cached_latest_stable_tag'  => {
                    get => {
                        summary     => 'Возвращает самый свежий стабильный тэг в истории коммитов ветки',
                        tags        => [ 'github' ],
                        description => "
Ручке нужно обязательно передать параметр branch.
Значение этого параметра — имя ветки в репозитории partner2.
Ручка вернет json с последней стабильным тэгом в истории коммитов указанной ветки.
Ручка использует данные из кеша, который обновляется с помощью ручки POST github/update.
",
                        parameters  => [
                            {
                                name        => "branch",
                                in          => "query",
                                description => "имя ветки",
                                required    => JSON::PP::true,
                                type        => "string",
                            },
                        ],

                        responses   => {
                            200 => {
                                description => "Действие успешно выполнено",
                            },
                            400 => {
                                description => "Ветка не указана",
                            },
                            404 => {
                                description => "Нет информации о стабильных тегах в указанной ветке. Скорее всего, она сильно отстала от текущей стабильной версии",
                            },
                        },
                    }
                },

                'github/update'           => {
                    post => {
                        summary     => 'Обновляет внутренний кеш с информаций из github',
                        tags        => [ 'github' ],
                        description => '
Ручка обновляет внутренний кеш с информаций из GitHub.

Вот эти GitHub репозитории должны быть настроены таким образом, чтобы на
каждый push происходила отпрака вебхуков на всех сервера на которых работает
creator:

 * https://github.yandex-team.ru/partner/partner2
 * https://github.yandex-team.ru/partner/yharnam

Данные из кеша, который обновялется этой ручкой используются в ручках
GET github/cached_branches, GET github/cached_tags, GET github/cached_migrationsql_files
',
                        responses   => {
                            200 => {
                                description => "Кеш успешно обновлен.",
                                schema      => {
                                    '$ref' => "#/definitions/OkResponse"
                                }
                            },
                        },
                    },
                },

                'cached_docker_db_images' => {
                    get => {
                        summary     => 'Список докерных образов с базой',
                        tags        => [ 'smth' ],
                        description => "
Ручка возвращает список докер образов с базой данных, которые есть на
машине с креатором.

Элементы в списке отсортированы. Первый элемент — это самый старый образ,
а последний элемент — это самый свежий образ.

Возвращается список докерных образов из кеша. Кеш обновляется раз в час
(и одновременно с этим обновленим происходит привоз новых докерных баз
и удаление старых).
",
                    }
                },
                'alive' => {
                    get => {
                        summary     => 'Ручка для проверки работы API',
                        tags        => [ 'smth' ],
                        description => '
Возвращает статус 200 если API работает нормально. Если есть какие-то
проблемы с работой API, то возвращет другой status code.
',
                        responses   => {
                            200 => {
                                description => "API работает.",
                                schema      => {
                                    '$ref' => "#/definitions/OkResponse"
                                }
                            },
                        },
                    },
                },
                login   => {
                    get  => {
                        summary       => 'Проверяет данные пользователи с помощью blackbox и возвращает логин если пользователь залогинен',
                        "description" => "
Ручке читает значения кук Session_id, sessionid2.
В ответ эта ручка отдаст логин пользователя если указанным данным соответствет какой-нибудь пользователь.
Ручка использует метод sessionid сервиса blackbox. Дока про этот метод — https://doc.yandex-team.ru/blackbox/reference/MethodSessionID.xml
",
                        tags          => [ 'smth' ],
                        responses     => {
                            200 => {
                                description => "Указанным данным соответствует логин.",
                                schema      => {
                                    '$ref' => "#/definitions/LoginResponse"
                                }
                            },
                        },
                    },
                    post => {
                        summary       => 'Проверяет данные пользователи с помощью blackbox и возвращает логин если пользователь залогинен',
                        "description" => "
Ручке нужно передать значения кук Session_id, sessionid2, установленых на домен .yadex-team.ru и ip адрес пользователя.
В ответ эта ручка отдаст логин пользователя если указанным данным соответствет какой-нибудь пользователь.
Ручка использует метод sessionid сервиса blackbox. Дока про этот метод — https://doc.yandex-team.ru/blackbox/reference/MethodSessionID.xml
",
                        tags          => [ 'smth' ],
                        parameters    => [
                            {
                                in       => 'body',
                                name     => 'body',
                                required => JSON::PP::true,
                                schema   => {
                                    '$ref' => "#/definitions/LoginRequest"
                                }
                            }
                        ],
                        responses     => {
                            200 => {
                                description => "Указанным данным соответствует логин.",
                                schema      => {
                                    '$ref' => "#/definitions/LoginResponse"
                                }
                            },
                        },
                    },
                },
            },
            definitions => {
                OkResponse    => {
                    type       => "object",
                    required   => [
                        "status",
                    ],
                    properties => {
                        "status" => {
                            type    => "string",
                            enum    => [
                                "ok",
                            ],
                            example => "ok"
                        },
                    },
                },
                FailResponse  => {
                    type       => "object",
                    required   => [
                        "status",
                    ],
                    properties => {
                        "status" => {
                            type    => "string",
                            enum    => [
                                "fail",
                            ],
                            example => "fail"
                        },
                    },
                },
                betaPort      => {
                    name        => "betaPort",
                    in          => "path",
                    description => "Номер беты",
                    required    => JSON::PP::true,
                    type        => "integer",
                },
                BetaRequest   => {
                    type       => "object",
                    required   => [
                        "backend",
                        "frontend",
                        "db",
                        "blackbox",
                        "yacotools",
                        "bs",
                    ],
                    properties => {
                        "backend"   => {
                            type        => "string",
                            example     => "master",
                            description => 'Имя git ветки, тег или комит из которого нужно собирать perl server side беты',
                        },
                        "frontend"  => {
                            type        => "string",
                            example     => "master",
                            description => 'Имя git ветки, тег или комит из которого нужно собирать nodejs часть беты',
                        },
                        "db"        => {
                            type        => "object",
                            required    => [
                                "type",
                            ],
                            properties  => {
                                type                  => {
                                    enum        => [
                                        "preset",
                                        "docker",
                                    ],
                                    description => 'Тип базы данных которую нужно использовать. Если указать docker, '
                                        . 'то нужно будет дополнить указать еще некоторе настройки — и тогда для '
                                        . 'беты будет создана отдельная база данных. Если указать тип preset, '
                                        . 'то в дополнительном параметре preset_id нужно указать имя пресета, '
                                        . 'который нужно использовать.',

                                },
                                docker_image          => {
                                    type        => "string",
                                    example     => "registry.yandex.net/partners/partner2-db-general:2.18.2547-2021-09-14",
                                    description => 'Имя докер образа с базой данных которую нужно использовать. '
                                        . 'Этот параметр можно указать только если указан type = docker.'
                                        . 'Список доступных докер образов можно узнать с помощью ручки GET cached_docker_db_images.'
                                        ,
                                },
                                migrationsql_files    => {
                                    type        => "array",
                                    example     => [ "before_release/release_176.sql" ],
                                    description => 'Массив sql файлов из репозитория partner2. Указанные '
                                        . 'файлы будут налиты в докерную базу. Файлы будут налиты в том порядке '
                                        . 'как они перечислены в этом массиве. Получить список файлов, которые '
                                        . 'есть в репозитории partner2 можно с помощью ручки GET github/cached_sql_migrations '
                                        . 'Этот параметр можно указать только если указан type = docker. '
                                        . 'Если указан type = docker, то этот параметр нужно обязательно указывать (но это может быть пустой массив). '
                                        ,
                                },
                                migrationscript_files => {
                                    type        => "array",
                                    example     => [ "before_release/INFRASTRUCTUREPI-1754_test.pl" ],
                                    description => 'Массив исполняемых файлов из репозитория partner2. Указанные '
                                        . 'файлы будут выполнены при сборке беты. Файлы будут выполнены в том порядке '
                                        . 'как они перечислены в этом массиве. Получить список исполняемых файлов, которые '
                                        . 'есть в репозитории partner2 можно с помощью ручки GET github/cached_script_migrations '
                                        . 'Этот параметр можно указать только если указан type = docker. '
                                        . 'Если указан type = docker, то этот параметр нужно обязательно указывать (но это может быть пустой массив). '
                                        ,
                                },
                                preset_id             => {
                                    enum        => [
                                        "dev",
                                        "ts",
                                    ],
                                    description => 'Сохранные набор подключений – dev это подключение к стандратной '
                                        . 'базе, которая используется при разработке. Вариант ts — это подключение к '
                                        . 'базе данных, которая используется на https://partner2-test.yandex.ru/. '
                                        . 'Этот параметр можно указать только если указан type = preset.',
                                },
                            },
                            example     => {
                                type                  => 'docker',
                                docker_image          => "registry.yandex.net/partners/partner2-db-general:2.18.2547-2021-09-14",
                                migrationsql_files    => [ "before_release/release_176.sql" ],
                                migrationscript_files => [ "before_release/INFRASTRUCTUREPI-1754_test.pl" ],
                            },
                            description => 'База данных с которой будет работать бета',
                        },
                        "blackbox"  => {
                            type        => 'object',
                            required    => [
                                "type",
                            ],
                            properties  => {
                                "type" => {
                                    type    => "string",
                                    enum    => [
                                        "url",
                                    ],
                                    example => "url"
                                },
                                url    => {
                                    type    => "string",
                                    example => "http://example.com:80"
                                }
                            },
                            example     => {
                                type => 'url',
                                url  => "http://blackbox-mimino.yandex.net",
                            },
                            description => 'Адрес blackbox с которым будет работать бета',
                        },
                        "yacotools" => {
                            type        => 'object',
                            required    => [
                                "type",
                            ],
                            properties  => {
                                "type" => {
                                    type    => "string",
                                    enum    => [
                                        "url",
                                    ],
                                    example => "url"
                                },
                                url    => {
                                    type    => "string",
                                    example => "http://example.com:80"
                                }
                            },
                            example     => {
                                type => 'url',
                                url  => "http://yacotools01et.yandex.ru",
                            },
                            description => 'Адрес yacotools которым будет работать бета',
                        },
                        "bs"        => {
                            type        => 'object',
                            required    => [
                                "type",
                            ],
                            properties  => {
                                "type" => {
                                    type    => "string",
                                    enum    => [
                                        "url",
                                    ],
                                    example => "url"
                                },
                                url    => {
                                    type    => "string",
                                    example => "http://example.com:80"
                                }
                            },
                            example     => {
                                type => 'url',
                                url  => "http://bssoap-1.bssoap.pi-integration-qloud.bssoap.yabs-qabs-osc.stable.qloud-d.yandex.net/",
                            },
                            description => 'Адрес БК которым будет работать бета',
                        },
                        "comment"   => {
                            type        => "string",
                            example     => "Задача PI-1234",
                            description => 'Опциональный комментарий к бете.',
                        },
                        "ttl"       => {
                            type        => "number",
                            example     => 86400,
                            description =>
                                'Опциональный параметр.
Указывает через сколько секунд после создания бета будет удалена.
Если параметр не указан, то автоудаления беты не будет.
Если указано, то число должно быть в диапазоне от 0 (включительно) до 31536000 (включительно).
',
                        },
                    }
                },
                BetaResponse  => {
                    type       => "object",
                    required   => [ qw(
                        backend
                        backend_sha1
                        backend_status
                        blackbox
                        bs
                        comment
                        created
                        created_by
                        db
                        frontend
                        frontend_sha1
                        frontend_status
                        port
                        status
                        url
                        yacotools
                    ) ],
                    properties => {
                        status            => {
                            type        => "string",
                            enum        => [
                                "free",
                                "unknown",
                                "occupied",
                                "building",
                                "running",
                                "stopped",
                                "build_fail",
                            ],
                            example     => "running",
                            description => 'Статус беты',
                        },
                        port              => {
                            type        => "integer",
                            example     => 8591,
                            description => 'Порт на котором работает https бета. И это еще и главный ID беты',
                        },
                        comment           => {
                            type        => "string",
                            maxLength   => 1000,
                            example     => "Бета для тестирования задачи PI-123",
                            description => 'Комментарий к собраной бете',
                        },
                        "created"         => {
                            type        => "string",
                            example     => "2016-05-21T16:48:25Z",
                            description => 'Дата и время создания беты в формате ISO 8601',
                        },
                        "url"             => {
                            type        => "string",
                            example     => "https://partner2-creator.yandex.ru:8591",
                            description => 'Адрес на котором работает https сервер с бетой',
                        },
                        "created_by"      => {
                            type        => "string",
                            example     => "bessarabov",
                            description => 'Доменный логин пользователя кто создал бету',
                        },
                        "blackbox"        => {
                            type        => 'object',
                            required    => [
                                "type",
                            ],
                            properties  => {
                                "type" => {
                                    type    => "string",
                                    enum    => [
                                        "url",
                                    ],
                                    example => "url"
                                },
                                url    => {
                                    type    => "string",
                                    example => "http://example.com:80"
                                }
                            },
                            example     => {
                                type => 'url',
                                url  => "http://blackbox-mimino.yandex.net",
                            },
                            description => 'Адрес blackbox с которым работает бета',
                        },
                        "yacotools"       => {
                            type        => 'object',
                            required    => [
                                "type",
                            ],
                            properties  => {
                                "type" => {
                                    type    => "string",
                                    enum    => [
                                        "url",
                                    ],
                                    example => "url"
                                },
                                url    => {
                                    type    => "string",
                                    example => "http://example.com:80"
                                }
                            },
                            example     => {
                                type => 'url',
                                url  => "http://yacotools01et.yandex.ru",
                            },
                            description => 'Адрес yacotools с которым работает бета',
                        },
                        "bs"              => {
                            type        => 'object',
                            required    => [
                                "type",
                            ],
                            properties  => {
                                "type" => {
                                    type    => "string",
                                    enum    => [
                                        "url",
                                    ],
                                    example => "url"
                                },
                                url    => {
                                    type    => "string",
                                    example => "http://example.com:80"
                                }
                            },
                            example     => {
                                type => 'url',
                                url  => "http://bssoap-1.bssoap.pi-integration-qloud.bssoap.yabs-qabs-osc.stable.qloud-d.yandex.net/",
                            },
                            description => 'Адрес БК которым работает бета',
                        },
                        db                => {
                            type        => "object",
                            required    => [
                                "type",
                            ],
                            properties  => {
                                type               => {
                                    enum        => [
                                        "preset",
                                        "docker",
                                    ],
                                    description => 'Тип используемой базы данных.',
                                },
                                docker_image       => {
                                    type        => "string",
                                    example     => "registry.yandex.net/partners/partner2-db-general:2.18.2547-2021-09-14",
                                    description => 'Имя докерной базы данных.'
                                        . 'Этот параметр будет указан только если указан type = docker.'
                                        ,
                                },
                                migrationsql_files => {
                                    type        => "array",
                                    example     => [ "misc.sql" ],
                                    description => 'Массив sql файлов из репозитория migrationsql которые налиты на докерную базу '
                                        . 'Этот параметр будет указан только если указан type = docker.',
                                },
                                preset_id          => {
                                    enum        => [
                                        "dev",
                                        "ts",
                                    ],
                                    description => 'Сохранные набор подключений – dev это подключение к стандратной '
                                        . 'базе, которая используется при разработке. Вариант ts — это подключение к '
                                        . 'базе данных, которая используется на https://partner2-test.yandex.ru/. '
                                        . 'Этот параметр будет указан только если указан type = preset.',
                                },
                            },
                            example     => {
                                type               => 'docker',
                                docker_image       => "registry.yandex.net/partners/partner2-db-general:2.18.2547-2021-09-14",
                                migrationsql_files => [ "misc.sql" ],
                            },
                            description => 'База данных с которой работает бета.',
                        },
                        "backend"         => {
                            type        => "string",
                            example     => "master",
                            description => 'Имя git ветки, тега или комит из которого собран perl server side беты',
                        },
                        "backend_sha1"    => {
                            type        => "string",
                            example     => "c3a3a95d255690599084caf7d79e0e23d2cd2e40",
                            description => 'SHA1 git комита из которого собран perl server side беты',
                        },
                        "backend_status"  => {
                            type        => "string",
                            enum        => [
                                "actual",
                                "outdated",
                                "deleted",
                            ],
                            example     => "actual",
                            description => 'Статус git ветки или тега из которого собран perl server side беты',
                        },
                        "frontend"        => {
                            type        => "string",
                            example     => "master",
                            description => 'Имя git ветки, тега или комит из которого собрана nodejs часть беты',
                        },
                        "frontend_sha1"   => {
                            type        => "string",
                            example     => "3fe59e214e44f3f13f18ab0e3a21f99754fc3cbe",
                            description => 'SHA1 git комита из которого собрана nodejs часть беты',
                        },
                        "frontend_status" => {
                            type        => "string",
                            enum        => [
                                "actual",
                                "outdated",
                                "deleted",
                            ],
                            example     => "actual",
                            description => 'Статус git ветки или тега из которого собрана nodejs часть беты',
                        },
                    },
                },
                LoginRequest  => {
                    type       => "object",
                    required   => [
                        "Session_id",
                        "sessionid2",
                        "userip",
                    ],
                    properties => {
                        "Session_id" => {
                            type      => "string",
                            maxLength => 1000,
                            example   => "3:14..."
                        },
                        "sessionid2" => {
                            type      => "string",
                            maxLength => 1000,
                            example   => "3:14..."
                        },
                        "userip"     => {
                            type      => "string",
                            maxLength => 1000,
                            example   => "127.0.0.1"
                        },
                    },
                },
                LoginResponse => {
                    type       => "object",
                    required   => [
                        "login",
                    ],
                    properties => {
                        "login" => {
                            type      => "string",
                            maxLength => 1000,
                            example   => "bessarabov"
                        },
                    },
                },
            }
        }
    );
};

app->start;
