package Utils;

use strict;
use warnings FATAL => 'all';
use feature 'say';
use utf8;
use open qw(:std :utf8);

use Carp;
use Data::Dumper;
use Data::Rmap;
use File::Slurp;
use HTTP::Tiny;
use JSON::PP;
use JSV::Validator;
use lib::abs;
use Moment;

JSV::Validator->load_environments("draft4");

our @ISA = qw(Exporter);
our @EXPORT_OK = qw(
    by_stage
    change_file_content
    create_after_file
    docker_registry_login
    find_today_docker_image
    get_beta_json_schema_as_perl_structure
    get_current_docker_db_images
    get_images_to_delete
    get_last_lines_from_content
    get_registry_auth_data
    get_secret
    is_valid_user_input_for_beta_creation
    is_valid_user_input_for_beta_creation_version_3
    run_cmd
    service_discovery_get_active_hostname
    switch_frontend
);
our @EXPORT = @EXPORT_OK;

my $DOCKER_REGISTRY = 'registry.yandex.net';

my @CLUSTERS = qw(vla sas man);
my $CLIENT_NAME = 'robot-partner';

my $BETA_JSON_SCHEMA = {
    type => 'object',
    properties => {
        backend => {
            type => 'string',
        },
        frontend => {
            type => 'string'
        },
        db => {
            enum => [
                'dev',
                'ts',
            ],
        },
        blackbox => {
            type => 'string',
        },
        yacotools => {
            type => 'string',
        },
        bs => {
            type => 'string',
        },
        comment => {
            type => 'string',
            maxLength => 1000,
        },
        Session_id => {
            type => 'string',
            maxLength => 1000,
        },
        sessionid2 =>  {
            type => 'string',
            maxLength => 1000,
        },
        userip =>  {
            type => 'string',
            maxLength => 1000,
        },
    },
    required => [ 'backend', 'frontend', 'db', 'blackbox' ],
    additionalProperties => JSON::PP::false,
};

my $PI_SECRETS_PATH = '/etc/pi-secrets.json';

sub get_beta_json_schema_as_perl_structure {

    my $json = encode_json $BETA_JSON_SCHEMA;
    my $data = decode_json $json;

    rmap_all { if (ref($_) eq 'JSON::PP::Boolean') { $_ = !!$_ } } $data;

    return $data;
}

sub is_valid_user_input_for_beta_creation {
    my ($perl_data_from_user) = @_;

    my $jv = JSV::Validator->new(
        environment => "draft4",
    );

    my $is_valid = $jv->validate(get_beta_json_schema_as_perl_structure(), $perl_data_from_user);

    return $is_valid;
}

sub change_file_content {
    my (%opts) = @_;

    my $file_name = delete $opts{file_name};
    my $changer = delete $opts{changer};
    my $opts = delete $opts{opts};

    die 'no file_name' if not defined $file_name;
    die 'no changes' if not defined $changer;
    die 'changes must be coderef' if ref $changer ne 'CODE';

    if ($opts) {
        die 'opts must be array' if ref $opts ne 'ARRAY';
    }

    if (!-f $file_name) {
        die "No file '$file_name'";
    }

    my $content = read_file(
        $file_name,
        {
            binmode => ':utf8',
        },
    );

    my $changed_content = $changer->(
        $content,
        (defined $opts ? @{$opts} : ()),
    );

    write_file(
        $file_name,
        {
            binmode => ':utf8',
        },
        $changed_content,
    );

    return 1;
}

sub switch_frontend {
    my ($content, $frontend_ref) = @_;

    $content =~ s/ARG FRONTEND_NODE_REF/ARG FRONTEND_NODE_REF=$frontend_ref/;

    return $content;
}

sub is_valid_user_input_for_beta_creation_version_3 {
    my ($input) = @_;

    if (exists $input->{ttl}) {
        my $ttl = $input->{ttl};

        if (not defined $ttl) {
            return (0, '"ttl" must be a number');
        }

        if (!($ttl =~ /^-?[0-9]+\z/)) {
            return (0, '"ttl" must be a number');
        }

        if ($ttl < 0 || $ttl > 31536000) {
            return (0, '"ttl" must be in range [0, 31536000]');
        }
    }

    return 1;
}

sub create_after_file {
    my (%opts) = @_;

    my $file_name = delete $opts{file_name};
    die 'no file_name' if not defined $file_name;

    my $content = read_file lib::abs::path('../bin/after');

    write_file(
        $file_name,
        {
            binmode => ':utf8',
        },
        $content,
    );

    `chmod a+x $file_name;`;

    return 1;
}

sub get_last_lines_from_content {
    my (%params) = @_;

    my $content = delete $params{content};
    my $lines = delete $params{lines};

    if ($lines && $lines =~ /^[0-9]+\z/) {

        my @content_lines = split /\n/, $content;

        if ($lines > @content_lines) {
            return $content;
        } else {
            my @last_n = @content_lines[-$lines..-1];
            return join("\n", @last_n) . "\n";
        }

    } else {
        return $content;
    }
}

sub get_registry_auth_data {
    my ($registry) = @_;

    croak "Must specify registry" if not defined $registry;

    my $file_name = '/etc/registry_auth.json';

    if (-e $file_name) {

        my $data = decode_json scalar read_file $file_name;

        if ($data->{$registry}) {
            croak "username is unknown" if not defined $data->{$registry}->{username};
            croak "password is unknown" if not defined $data->{$registry}->{password};
            croak "email is unknown" if not defined $data->{$registry}->{email};

            return $data->{$registry}->{username}, $data->{$registry}->{password}, $data->{$registry}->{email};
        } else {
            croak sprintf 'No info about registry "%s" in config file %s', $registry, $file_name;
        }


    } else {
        croak "File $file_name not found";
    }
}

=head2 by_stage

    my @sorted = sort by_stage @files;

После этого в @sorted будет:

    before_release/PI-123_feature.sql
    after_release/PI-123_feature.sql

Основная идея этой сабы - чтобы before_release был перед after_release.

=cut

sub by_stage {
    my ($aa, $bb) = @_;

    my ($a_prefix) = $aa =~ m(^(.*?)/);
    my ($b_prefix) = $bb =~ m(^(.*?)/);

    if ($a_prefix eq 'before_release' and $a_prefix ne $b_prefix) {
        return -1;
    } elsif ($a_prefix eq 'after_release' and $a_prefix ne $b_prefix) {
        return 1;
    } else {
        return $aa cmp $bb;
    }
}


sub get_secret {
    my ($secret_name) = @_;

    croak "File $PI_SECRETS_PATH not found" unless -e $PI_SECRETS_PATH;

    my $secrets = decode_json scalar read_file $PI_SECRETS_PATH;

    return exists($secrets->{$secret_name}) ? $secrets->{$secret_name} : croak "Secret '$secret_name' not found";
}

sub service_discovery_get_active_hostname {
    my ($id, $chek_port) = @_;

    my $ua = HTTP::Tiny->new();

    for my $c (@CLUSTERS) {
        my $result = $ua->get(
            "http://sd.yandex.net:8080/resolve_endpoints/json",
            {
                content => encode_json(
                    {
                        cluster_name    => $c,
                        endpoint_set_id =>  $id,
                        client_name     => $CLIENT_NAME,
                    }
                )
            }
        );

        next unless $result && $result->{success} && $result->{content};

        my $data = eval {decode_json($result->{content})};
        next if $@;

        for my $ep (@{$data->{endpoint_set}{endpoints} // []}) {
            if (my $fqdn = $ep->{fqdn}) {
                my $responce = $ua->get("http://$fqdn:$chek_port/");

                if ($responce && $responce->{success} && $responce->{content} && $responce->{content} eq "Ok.\n") {
                    return $fqdn;
                }
            }
        }
    }

    croak "Can't get host for clickhouse";
}

sub run_cmd {
    my ($cmd) = @_;

    my $exit_status = system($cmd);

    unless ($exit_status == 0) {
        die "Command '$cmd' exited with non zero exit status";
    }
}

sub docker_registry_login {
    my ($registry) = @_;

    $registry //= $DOCKER_REGISTRY;

    my ($username, $password, $email) = get_registry_auth_data($registry);
    warn "docker login --username=$username --password=XXX  $registry &> /dev/null";
    run_cmd("docker login --username=$username --password=$password $registry &> /dev/null");
}

sub find_today_docker_image {
    my ($registry) = @_;

    $registry //= $DOCKER_REGISTRY;

    my $dt = Moment->now()->get_d();

    my ($username, $password) = get_registry_auth_data($registry);

    my $cmd = sprintf q[
curl -s --insecure --user %s:%s 'https://%s/v2/partners/partner2-db-general/tags/list' \
    | jq -r .tags[] \
    | grep -Pe '^2.18.\d+-%s$'
], $username, $password, $registry, $dt;

    my $tag = `$cmd`;
    chomp($tag);

    unless ($tag) {
        warn "Can't find docker image for the date=$dt";

        return undef;
    }

    return "$registry/partners/partner2-db-general:$tag";
}

=head2 get_current_docker_db_images

    my @images = get_current_docker_db_images();

После этого в @images будет список вида:

    (
        'registry.yandex.net/partners/partner2-db-general:2.18.2547-2021-09-13',
        'registry.yandex.net/partners/partner2-db-general:2.18.2547-2021-09-14',
    )

Элементы в списке отсортированы. Первый элемент - это самый старый образ,
а последний элемент - это самый свежий образ.

=cut

sub get_current_docker_db_images {
    my $resp = `docker images --format '{{.Repository}}:{{.Tag}}' | grep partner2-db-general | perl -nale 'if (\$_ =~ /(.+?):latest\$/) {print \$1} else {print \$_}' | sort -V`;
    chomp($resp);

    my @images = split /\n/, $resp;

    return @images;
}

=head2 get_images_to_delete

    my @images_to_delete = get_images_to_delete(
        current_images => [
            'image1',
            'image2',
        ],
        max_left => $max_left,
    );

Возвращает массив с образами, которые нужно удалить.

=cut

sub get_images_to_delete {
    my (%p) = @_;

    my @current_images = @{delete $p{current_images}};
    my $max_left = delete $p{max_left};

    my %images_with_tags;
    my @images_without_tags;

    foreach my $image (@current_images) {
        if ($image =~ /(2\.18\.[0-9]+)/) {
            push @{$images_with_tags{$1}}, $image;
        } else {
            push @images_without_tags, $image;
        }
    }

    my @images_to_be_saved;

    foreach my $tag (keys %images_with_tags) {
        # по каждому тегу нужно сохранить образ с последней датой
        push @images_to_be_saved, [sort @{$images_with_tags{$tag}}]->[-1];
    }

    @images_to_be_saved = (
        sort({ $a cmp $b} @images_without_tags),
        sort({
            my ($version_a, $date_a) = $a =~ /2\.18\.(\d+)-([\d-]+)\z/;
            my ($version_b, $date_b) = $b =~ /2\.18\.(\d+)-([\d-]+)\z/;

            $version_a <=> $version_b
                || $date_a cmp $date_b
        } @images_to_be_saved),
    );

    # Оставляем не больше $max_left образов
    if ( $max_left > scalar @images_to_be_saved) {
        $max_left = scalar @images_to_be_saved;
    }
    my %images_to_be_saved = map {$_ => 1} @images_to_be_saved[-$max_left..-1];

    return grep {!$images_to_be_saved{$_}} @current_images;
}

1;
