package Yandex::Blackbox;

# File is copyied from https://github.yandex-team.ru/bessarabov/Yandex-Blackbox
# commit 08c4492b49729cc8b9c8b07fce6045ac1a580bfe

use strict;
use warnings FATAL => 'all';
use feature 'say';
use utf8;
use open qw(:std :utf8);

use Carp;
use HTTP::Tiny;
use JSON::PP;
use URI;

=encoding UTF-8

=head1 NAME

Yandex::Blackbox

=head1 SYNOPSYS

    my $yb = Yandex::Blackbox->new(
        url => 'http://blackbox.yandex-team.ru/blackbox',
        x_ya_service_ticket => 'asdf...',
    );

    my $data = $yb->sessionid(
        sessionid => '3:14...',
        sslsessionid => '3:14...',
        userip => '130.193.34.44',
        host => 'creator.partner.yandex-team.ru',
    );

    say $data->{login}; # 'bessarabov'

=cut


=head1 METHODS

=cut

=head2 new

Конструктор. Возврщает объект Yandex::Blackbox. Нужно обязательно передать
параметр url, значение которого - это адрес черного ящика. Адреса черных
ящиков описаны на странице
L<https://doc.yandex-team.ru/blackbox/concepts/blackboxLocation.xml>

    my $blackbox = Yandex::Blackbox->new(
        url => 'http://blackbox.yandex-team.ru/blackbox',
        x_ya_service_ticket => 'asdf...',
    );

=cut

sub new {
    my ($class, %params) = @_;

    my $self = {};
    bless $self, $class;

    $self->{_url}                 = delete $params{url};
    $self->{_x_ya_service_ticket} = delete $params{x_ya_service_ticket};

    croak "No url" if not defined $self->{_url};

    return $self;
}

=head2 sessionid

Метод возвращает структуру данных с ответом ручки черного ящика sessionid.

Методу нужно обязательно передать параметры:

 * sessionid - значение куки Session_id
 * sslsessionid - значение куки sessionid2
 * userip - IP пользователя
 * host - хост куки

Метод возвращает HASHREF с развесистой структорой, описание этой структуры
есть на странице L<https://doc.yandex-team.ru/blackbox/reference/method-sessionid-response-json.xml>.

Метод бросает исключение если указанным данным не соответствует никакой
залогиненный пользователь.

    my $data = $yb->sessionid(
        sessionid => '3:14...',
        sslsessionid => '3:14...',
        userip => '130.193.34.44',
        host => 'creator.partner.yandex-team.ru',
    );

    say $data->{login}; # 'bessarabov'

=cut

sub sessionid {
    my ($self, %params) = @_;

    my $uri = URI->new($self->{_url});
    $uri->query_form(
        sessionid => delete $params{sessionid},
        sslsessionid => delete $params{sslsessionid},
        userip => delete $params{userip},
        host => delete $params{host},
        format => 'json',
    );

    my $response = HTTP::Tiny->new()->request(
        'GET',
        $uri->as_string(),
        {
            headers =>
              {defined($self->{_x_ya_service_ticket}) ? ('X-Ya-Service-Ticket' => $self->{_x_ya_service_ticket}) : (),},
        },
    );

    if ($response->{status} eq '200') {
        my $answer = decode_json $response->{content};

        # https://doc.yandex-team.ru/blackbox/reference/method-sessionid-response-json.xml#status
        if (($answer->{status}->{id} // '') eq '0') {
            return $answer;
        } else {
            croak sprintf 'Got status id %s. Stopped', $answer->{status}->{id} // '';
        }
    } else {
        croak sprintf 'Got http status %s', $response->{status};
    }

    return 1;
}

1;
