use Net::INET6Glue::INET_is_INET6;
use Mojolicious::Lite;
use Mojo::Template;

use Carp;
use JSON::PP;
use HTTP::Tiny;
use Moment;
use Time::HiRes qw(gettimeofday tv_interval);
use File::Slurp;

plugin 'tt_renderer' => {
    template_options => {
        INCLUDE_PATH => '/app/templates'
    },
};

die "Env API_HOST is not defined" unless defined $ENV{API_HOST};

sub get_data_from_url {
    my ($url) = @_;

    my $response = HTTP::Tiny->new()->get($url);

    if ($response->{success}) {
        my $data = decode_json( $response->{content} );

        return $data;
    } else {
        croak "$url return $response->{status} status";
    }

}

my $API_URL_PREFIX;

if ($ENV{API_HOST} eq 'api') {
    $API_URL_PREFIX = 'http://api/api/3/';
} else {
    $API_URL_PREFIX = sprintf 'https://%s/api/3/', $ENV{API_HOST};
}

sub get_betas {
    return get_data_from_url( $API_URL_PREFIX . 'betas' );
}

sub get_branches_tags {
    my (%params) = @_;

    my @elements = @{decode_json(HTTP::Tiny->new()->get(
        $API_URL_PREFIX
        . sprintf(
            'github/cached_%s?repo=%s',
            $params{what},
            $params{repo},
        )
    )->{content})};

    return @elements;
}

app->hook(before_dispatch => sub {
    my $c = shift;

    if ( ($c->req->url() eq '/') or ($c->req->url() eq '/add') ) {

        $c->stash(t0 => [gettimeofday()]);

        my $login = 'unknown';

        eval {
            my $response = HTTP::Tiny->new()->request(
                'POST',
                $API_URL_PREFIX . 'login',
                {
                    content => encode_json({
                        Session_id => $c->cookie('Session_id'),
                        sessionid2 => $c->cookie('sessionid2'),
                        userip => $c->req->headers->header('X-Real-IP'),
                    }),
                }
            );

            $login = decode_json($response->{content})->{login};
        };

        if ($@) {
            warn "Can't find out user login: " . $@;
        }

        $c->stash(
            login => $login,
            userip => $c->req->headers->header('X-Real-IP'),
        );
    }

});

app->hook(after_dispatch => sub {
    my ($c, $args) = @_;

    if ( ($c->req->url() eq '/') or ($c->req->url() eq '/add') ) {
        my $elapsed = tv_interval($c->stash('t0'), [gettimeofday()]);

        my $json_coder = JSON::PP
            ->new
            ->canonical
            ;

        my $now = Moment->now();

        write_file(
            sprintf('/log/frontend.%s.jsonl', $now->get_d()),
            {
                append => 1,
            },
            $json_coder->encode({
                timestamp    => $now->get_iso_string(),
                elapsed      => $elapsed,
                code         => $c->{tx}->{res}->{code},
                method       => $c->{tx}->{req}->{method},
                url          => $c->{tx}->{req}->{url}->to_string(),
                'user-agent' => $c->{tx}->{req}->{content}->{headers}->header('user-agent'),
                ip           => $c->stash('userip'),
                (($c->stash('login') // '') eq 'unknown') ? (
                    has_login => JSON::PP::false,
                ) : (
                    login     => $c->stash('login'),
                    has_login => JSON::PP::true,
                ),
            }) . "\n",
        );
    }

});

get '/' => sub {
    my $c = shift;

    my $betas = get_betas();

    my $now = Moment->now();
    my $now_timestamp = $now->get_timestamp();

    foreach my $beta (@{$betas}) {
        if ($beta->{status} ne 'free') {
            if (ref($beta->{db}) ne '') {
                if ($beta->{db}->{type} eq 'docker') {
                    $beta->{db} = 'docker';
                } elsif ($beta->{db}->{type} eq 'preset') {
                    $beta->{db} = $beta->{db}->{preset_id};
                } else {
                    die 'this shoud never happen';
                }
            }

            $beta->{created_by} //= 'robot-partner';

            $beta->{hours_ago} = '?';
            if ($beta->{created}) {
                my $beta_created = Moment->new(iso_string => $beta->{created});
                my $beta_created_timestamp = $beta_created->get_timestamp();

                $beta->{created} = $beta_created->get_dt();
                $beta->{hours_ago} = int(($now_timestamp - $beta_created_timestamp) / 60 / 60);
                $beta->{hours_left} = int(($beta_created_timestamp + $beta->{ttl} - $now_timestamp) / 60 / 60) if $beta->{ttl};
            }
        }
    }

    $c->stash(
        betas          => $betas,
        api_host       => $ENV{API_HOST},
        api_url_prefix => $API_URL_PREFIX,
    );

    $c->render(
        template => 'index',
    );
};

get '/add' => sub {
    my $c = shift;

    my $betas = get_betas();

    my @backend_tags = get_branches_tags( what => 'tags', repo => 'partner2' );
    my @backend_branches = get_branches_tags( what => 'branches', repo => 'partner2' );

    my @frontend_tags = get_branches_tags( what => 'tags', repo => 'yharnam' );
    my @frontend_branches = get_branches_tags( what => 'branches', repo => 'yharnam' );

    my @free_ports = map {$_->{port}} grep {$_->{status} eq 'free'} @{$betas};

    srand();
    my $proposed_port = $free_ports[rand @free_ports];

    $c->stash(
        login             => $c->stash('login') // 'robot-partner',
        api_host          => $ENV{API_HOST},
        api_url_prefix    => $API_URL_PREFIX, 
        free_ports        => \@free_ports,
        proposed_port     => $proposed_port,
        backend_branches  => [ reverse(@backend_tags), sort {uc($a) cmp uc($b)} (@backend_branches) ],
        frontend_branches => [ reverse(@frontend_tags), sort {uc($a) cmp uc($b)} (@frontend_branches) ],
    );

    $c->render(
        template => 'add',
    );
};

my $static = app->static();
push @{$static->paths}, '/app/static';

app->start;
