use strict;
use warnings FATAL => 'all';

use Action::Retry;
use Mojolicious::Lite;

use lib::abs qw(
    ./lib/
);
use Utils;

hook before_render => sub {
    my ($c, $args) = @_;

    my $status = $args->{status} // '';

    if ($status eq 404) {
        $args->{json} = {
            error_description => 'not found',
        };
    }

    if ($status eq 500) {
        $args->{json} = {
            error_description => 'internal error',
        };
    }
};

post '/intapi/get_next_page_id' => sub {
    my ($c) = @_;

    my $service = $c->param('service');

    return $c->render(_get_400_error("Must specify parameter 'service'")) unless defined $service;
    return $c->render(_get_400_error("'service' length must be > 0")) if length($service) == 0;
    return $c->render(_get_400_error("'service' length must be < 256")) if length($service) > 255;

    my $page_id;

    _execute_code(sub {
        my $dbh = get_dbh('partner');

        my $sth = $dbh->prepare(
            'insert into page_id_generator (`page_id`, `service`, `dt`, `ip`, `user_agent`) values (NULL, ?, now(), ?, ?);'
        );

        $sth->execute(
            $service,
            $c->req->headers->header('x-real-ip') // $c->tx->local_address,
            $c->req->headers->header('User-Agent'),
        ) or die $dbh->errstr;

        $page_id = $sth->{mysql_insertid};
    });

    die "No page_id" unless $page_id;

    $c->render(
        json => {
            page_id => $page_id,
        },
    );
};

get '/die' => sub {
    die;
};

get '/alive' => sub {
    my ($c) = @_;

    my $page_id;

    _execute_code(sub {
        my $dbh = get_dbh('partner');

        my $sth = $dbh->prepare('select max(page_id) from page_id_generator');
        $sth->execute();

        my @row = $sth->fetchrow_array;
        $page_id = $row[0];
    });

    if (defined($page_id)) {
        $c->render(
            json => {
                success => JSON::PP::true,
            },
        );
    } else {
        my $error_message = "Can't get data from mysql table page_id_generator.";
        if ($@) {
            $error_message .= ' ' . $@;
        }

        $c->render(
            json => {
                error_message => $error_message,
            },
            status => 503,
        );
    }
};

sub _get_400_error {
    my ($message) = @_;

    return (
        json => {
            error_message => $message,
        },
        status => 400,
    );
}

sub _execute_code {
    my ($code) = @_;

    Action::Retry->new(
        attempt_code => $code,
        retry_if_code => sub {
            my ($error) = @_;

            if ($error) {
               eval {get_dbh('partner', 'force_reconnect')};
               print STDERR $@ if $@;

               $@ = $error;

               return 1;
            } else {
               return 0;
            }
        },
        strategy => {
            Constant => {
                max_retries_number => 2,
                sleep_time => 1000,
            }
        },
        on_failure_code => sub {
            my ($error) = @_;

            print STDERR $error;
        },
    )->run();
}

app->start;
