package Model::CodeReview;

use utf8;
use strict;
use warnings;

use File::Slurp qw( read_file );
use JSON::PP qw( decode_json );
use HTTP::Tiny;
use Moment;


use Utils::Common qw(
    get_tt
    get_all_people
    get_backend_people
    get_frontend_people
);
use Model::PullRequest qw( get_repos_pull_requests );
use Model::ReleaseTickets;

sub get_repo_code_review_data {
    my ( $repos, $team, %opts ) = @_;
    $team ||= 'all';

    my $people = {};
    {
        no strict 'refs';
        my $method = sprintf 'get_%s_people', $team;
        $people = { map { $_->{login} => $_ } @{ $method->() } };
    }

    my $now = Moment->now();
    my $now_timestamp = $now->get_timestamp();

    my $days_colors = [
        [365, 'fatal' ],
        [7,   'warn'  ],
        [3,   'info'  ],
    ];

    my $data = get_repos_pull_requests($repos);

    my $release_tickets = Model::ReleaseTickets::get_release_tickets_data();

    my $found_logins = {};
    my $filtered_data = [];
    foreach my $el ( @$data ) {
        my %approved_by = map { $_ => 1} @{$el->{approved_by}};
        my %disapproved_by = map { $_ => 1} @{$el->{disapproved_by}}, (split /\s*,\s*/, $el->{changes_requested} // '');

        next if $team && !( $people->{$el->{author}} || $people->{$el->{assignee}} );
        next if defined $opts{login}      && !( $el->{assignee} eq $opts{login} && !$approved_by{ $opts{login} } );
        next if defined($opts{not_today}) && $now->get_d()   eq Moment->new(timestamp => $el->{timestamp_created_at})->get_d();
        next if defined($opts{author})    && $opts{author}   ne $el->{author};
        next if defined($opts{assignee})  && $opts{assignee} ne $el->{assignee};

        map { $found_logins->{$_} = 1 } grep {$_}  map { $el->{$_} } qw( assignee  author );

        push @$filtered_data, $el;
    }

    my @logins = grep { $people->{$_} && !$people->{$_}->{ex} } sort keys %$found_logins;

    my $statuses = {};
    foreach my $el (sort { $a->{timestamp_created_at} <=> $b->{timestamp_created_at} } @$filtered_data ) {

        next if defined($opts{start_id}) && $el->{id} < $opts{start_id};

        my %approved_by = map { $_ => 1} @{$el->{approved_by}};
        my %disapproved_by = map { $_ => 1} @{$el->{disapproved_by}}, (split /\s*,\s*/, $el->{changes_requested} // '');

        my $has_approve_from_assignee =
            exists($el->{assignee})
            && defined($el->{assignee})
            && grep { $_ eq $el->{assignee}} @{$el->{approved_by}};

        my $has_disapproves = @{$el->{disapproved_by}} > 0;

        my $review_state = $el->{codereview_state};
        if ($el->{repo} eq 'partner2') {
            if ($review_state eq 'pending' && $has_approve_from_assignee){
                $review_state = 'success';
            }
        }

        $review_state = $review_state eq 'success' && !$el->{changes_requested} ? 'success' : 'failed';

        my $status = 'author';
        if ( $el->{mergeable} eq '1' && $el->{tests_state} eq 'success' && !$has_disapproves && !$el->{changes_requested} ) {
            $status = $has_approve_from_assignee
                ? 'ready'
                : 'review_not_approved';
        }

        my $days_ago = int(($now_timestamp - $el->{timestamp_created_at})/86400);

        my $days_class = '';
        map { $days_class = $_->[1] } grep { $days_ago <= $_->[0] } @$days_colors;

        my $ticket_data      = $el->{issue} // {
        #    "key"        => "PI-10574",
        #    "resolution" => "fixed",
        #    "status"     => "codeReview",
        #    "summary"    => "Потерялась кнопка \"создать\" в старой верстке"
        };

        my $release_ticket_id = $release_tickets->{release_tickets}->{ $ticket_data->{key} } // '';
        my $release_data      = $release_tickets->{releases}->{ $release_ticket_id } // {};
        ( my $title_release   = $release_data->{summary} // '' ) =~ s|"|&quot;|g;

        my $review_by_login = [];
        foreach my $login (@logins) {
            my $status = $login eq $el->{author}
                ? '-'
                : $approved_by{$login}
                    ? 'OK'
                    : $disapproved_by{$login}
                        ? 'FAIL'
                        : '';
            push $review_by_login, {
                login => $login,
                status  => $status,
                is_ok   => $status eq 'OK'   ? 1 : 0,
                is_fail => $status eq 'FAIL' ? 1 : 0,
            };
        }

        push @{ $statuses->{ $status } }, {
            review_by_login => $review_by_login,
            row_logins       => join(',', grep {$_} $el->{author}, $el->{assignee}),
            days_class       => $days_class,
            created_days_ago => $days_ago,
            author           => $el->{author},
            assignee         => $el->{assignee},
            from_branch      => $el->{from_branch},
            to_branch        => $el->{to_branch},
            (length($el->{to_branch}) > 6 ? (to_branch_short  => substr($el->{to_branch}, 0, 30)) : () ),
            tests_state      => $el->{tests_state} eq 'success' ? 1 : 0,
            mergeable        => $el->{mergeable} eq '1' ? 1 : 0,
            codereview_state => $review_state eq 'success' ? 1 : 0,
            author_class     => ( grep { $status eq $_ } qw( ready review_approved  review_not_approved ) ) ? 'inactive ' : '',
            assignee_class   => ( grep { $status eq $_ } qw( ready review_approved  author) ) ? 'inactive ' : '',
            author_login_class     => ( grep { $status eq $_ } qw( ready review_approved  review_not_approved ) ) ? 'inactive_login ' : '',
            assignee_login_class   => ( grep { $status eq $_ } qw( ready review_approved  author) ) ? 'inactive_login ' : '',
            release_ticket_id => $release_ticket_id,
            title_release => $title_release,
            pr_url => $el->{url},
            ticket_summary => $ticket_data->{summary},
        };
    }

    $data = [
        ['Waiting reviewers', $statuses->{review_approved}     ],
        ['Waiting assignee',  $statuses->{review_not_approved} ],
        ['Waiting author',    $statuses->{author}       ],
        ['Ready',             $statuses->{ready}        ],
    ];

    my $tables = [
      map +{
          title     => $_->[0],
          rows      => $_->[1],
          logins    => \@logins,
          login_str => join(',', @logins),
      }, grep { @$_ && $_->[1] && @{ $_->[1] } } @$data
    ];

    return $tables;
};


1;