package Model::Hosts;

use utf8;
use strict;
use warnings;

use JSON::PP;

use base qw/ Exporter /;
our @EXPORT_OK = qw(
    get_all_hosts
    get_all_host_names
    get_creator_host_names
    get_prod_host_names
    get_all_host_names_with_dc
);

my @CLUSTERS = qw(sas vla iva man);
my $ENDPOINT_SET_ID = {
    production => {
        Backend => 'partner-production-stage.Backend',
    }
};
my $CLIENT_NAME = 'robot-partner';

my $DC_SAS = 'sas';
my $DC_VLA = 'vla';
my $DC_IVA = 'iva';
my $DC_MYT = 'myt';
my $DC_MAN = 'man';

my $HOSTS = [
    {
        hostname => 'dev-partner2.yandex.ru',
        tag      => [ 'dev' ],
        dc       => $DC_SAS,
    },
    {
        hostname => 'dev-partner.yandex.ru',
        tag      => [ 'dev' ],
        dc       => $DC_MAN,
    },
    {
        hostname => 'partner2-test.yandex.ru',
        tag => ['partner2_ts'],
    },
    {
        hostname => 'partner-creator1.sas.yp-c.yandex.net',
        name     => 'creator1-sas.partner.yandex-team.ru',
        tag      => [ 'creator' ],
        dc       => $DC_SAS,
    },
    {
        hostname => 'partner-creator2.vla.yp-c.yandex.net',
        name     => 'creator2-vla.partner.yandex-team.ru',
        tag      => [ 'creator' ],
        dc       => $DC_VLA,
    },
    {
        hostname => 'partnerdb-mysql-test2.yandex.ru',
        tag => ['partner2_ts_db'],
    }
];

for my $h (@$HOSTS) {
    my %t = map {$_ => 1} @{$h->{tag}};
    $h->{htags} = \%t;
}

sub get_all_hosts {
    return $HOSTS;
}

sub get_all_host_names {
    return _get_host_names_by_tag();
}

sub get_creator_host_names {
    return _get_names_by_tag('creator');
}

sub get_all_host_names_with_dc {
    my ($tag) = @_;

    my @hosts = sort map {
        {
            hostname => $_->{hostname},
            dc       => $_->{dc},
        }
    } grep {!$tag || $_->{htags}{$tag}} @{ get_all_hosts() };

    return \@hosts;
}

sub get_prod_host_names {
    my (%p) = @_;

    unless($p{with_type}) {
        return [
            @{_get_host_names_by_tag('prod')},
            @{service_discovery_get_hostnames($ENDPOINT_SET_ID->{production}{Backend})},
        ];
    } else {
        return +{
            (map {$_ => ''} @{_get_host_names_by_tag('prod')}),
            (map {$_ => 'deploy'} @{service_discovery_get_hostnames($ENDPOINT_SET_ID->{production}{Backend})}),
        };
    }
}

sub _get_names_by_tag {
    my ($tag) = @_;
    return _get_field_by_tag($tag, 'name');
}

sub _get_host_names_by_tag {
    my ($tag) = @_;
    return _get_field_by_tag($tag, 'hostname');
}

sub _get_field_by_tag {
    my ($tag, $field_name) = @_;

    my @hosts = sort map {$_->{$field_name}} grep {!$tag || $_->{htags}{$tag}} @{ get_all_hosts() };

    return \@hosts;
}

sub service_discovery_get_hostnames {
    my ($id) = @_;
    my $ua = HTTP::Tiny->new();
    my @result;
    for my $c (@CLUSTERS) {
        my $ruid = "resolve_${id}_${c}_1";
        my $result = $ua->get(
            "http://sd.yandex.net:8080/resolve_endpoints/json",
            {
                content => encode_json(
                    {
                        cluster_name    => $c,
                        endpoint_set_id => $id,
                        client_name     => $CLIENT_NAME,
                        ruid            => $ruid
                    }
                )
            }
        );
        next unless $result && $result->{success} && $result->{content};
        my $data = decode_json($result->{content});
        next unless $data->{ruid} eq $ruid;
        for my $ep (@{$data->{endpoint_set}{endpoints} // []}) {
            if (my $fqdn = $ep->{fqdn}) {
                push @result, $fqdn;
            }
        }
    }

    return \@result;
}

1;
