package Model::ReleaseTickets;

use utf8;
use strict;
use warnings;

use File::Slurp qw( read_file  write_file );
use JSON::PP qw( decode_json );
use HTTP::Tiny;
use Storable;

use Utils::Common qw(
    get_startrek
    to_pretty_json
);


our $RELEASE_DATA_FILE_PATH = '/data/releases.json';

sub get_release_tickets_cached {
    my $content_ref = eval{ read_file(
        $RELEASE_DATA_FILE_PATH,
        { binmode => ':utf8', scalar_ref => 1 },
    ) } // \'{}';
    warn $@ if $@;
    return $content_ref;
};


sub get_release_tickets_data {
    my ($repo ) = @_;

    my $content_ref = get_release_tickets_cached();

    utf8::encode($$content_ref) if utf8::is_utf8($$content_ref);
    my $data = eval{ decode_json( $$content_ref ) } // {
    #    "release_tickets" : {
    #        "PI-10049" => "PI-10567",
    #        "PI-10192" => "PI-10567",
    #    },
    #    "releases" : {
    #        "PI-10567" => {
    #            type    => $release_type,
    #            num     => $release_num,
    #            summary => $ticket->{summary},
    #       }
    #    }
    };
    warn $@ if $@;

    return $data;
}

sub update_release_tickets_cache {

    my $YS = get_startrek();

    my $releases = get_release_tickets($YS);

    my $debug = 0;

    my $release_tickets = {};
    foreach my $release_ticket ( keys %$releases ){
         my $all_tickets = get_all_tickets( $YS, $release_ticket, $debug );
        map { $release_tickets->{$_} = $release_ticket } keys %$all_tickets;
    }

    my $data = {
        releases        => $releases,
        release_tickets => $release_tickets
    };

    write_file(
        $RELEASE_DATA_FILE_PATH,
        { binmode => ':utf8' },
        to_pretty_json($data),
    );

    return $data;
}

sub get_release_tickets {
    my ( $ys ) = @_;

    my $tickets = $ys->search_all(
        query => q[Queue: PI and Status: !Закрыт and Status: !Решен and Status: !Отменено  and  ( Type: Release  or  Summary: "Тестирование на общей бете" ) "Sort By": Created],
    );

    my $releases = {};
    foreach my $ticket ( @$tickets ){
        my $release_num = '';
        my $release_type = '';
        if ( $ticket->{summary} =~ m/Релиз\s+(\d+)/i ) {
            $release_type = 'release';
            $release_num = $1;
        }
        elsif ( $ticket->{summary} =~ m/Хотфикс.*?(\d+)/i ) {
            $release_type = 'hotfix';
            $release_num = $1;
        }

        $releases->{ $ticket->{key} } = {
            type    => $release_type,
            num     => $release_num,
            summary => $ticket->{summary},
        };
    }

    return $releases;
}

sub get_all_tickets {
    my ( $ys, $release_ticket, $debug ) = @_;

    my $tickets_from_description = get_tickets_from_ticket_description(
        ys => $ys,
        key => $release_ticket,
    );

    my $cache_tickets_file_path    = 'tmp_all_tikets.bin';

    my $all_tickets = $debug && -e $cache_tickets_file_path
        ?  Storable::retrieve( $cache_tickets_file_path ) // {}
        :  {};

    unless ( %$all_tickets ) {
        foreach my $ticket (@$tickets_from_description){
            get_all_ticket_subtickets(
                ys => $ys,
                key => $ticket,
                ticket_storage_hash => $all_tickets,
                depth => 1
            );
        }
        Storable::store($all_tickets, $cache_tickets_file_path ) if $debug;
    }
    else {
        map { printf "\tst/%s\n", $_ } sort keys %$all_tickets;
    }

    return $all_tickets;
}

sub get_all_ticket_subtickets {
    my (%params) = @_;

    my ( $ys, $ticket_storage_hash, $key, $depth ) = @params{qw( ys  ticket_storage_hash  key  depth)};
    $depth //= 0;

    return if exists $ticket_storage_hash->{ $key };

    my $ticket = _get_issue($ys, $key);
    return if !$ticket;

    my $assignee =
        $ticket->{assignee}->{id}
        ? $ticket->{assignee}->{id} . '@'
        : 'Not assigned'
        ;

    $ticket_storage_hash->{$ticket->{key}} = 1;

    printf  "\t" x $depth . "st/%s\n", $ticket->{key};

    my $links = $ys->get_issue_links( $key );
    foreach my $element (@{$links}) {
        next if $element->{type}->{id} ne 'subtask';
        next if $element->{type}->{id} eq 'subtask' and $element->{direction} eq 'inward';

        get_all_ticket_subtickets(
            ys => $ys,
            key => $element->{object}->{key},
            ticket_storage_hash => $ticket_storage_hash,
            depth => $depth + 1
        );
    }
}

sub get_tickets_from_ticket_description {
    my (%params) = @_;
    my $ys = delete $params{ys};
    my $key = delete $params{key};

    my $ticket = _get_issue($ys, $key);
    return if !$ticket;

    my %tickets; my @tickets;
    while ($ticket->{description} =~ /([A-Z]{2,}-\d+)/ga) {
        push @tickets, $1 unless $tickets{$1} ++;
    }

    return \@tickets;
}

sub _get_issue {
    my $ys = shift;
    my $key = shift;
    my $ticket;

    eval {
        $ticket = $ys->get_issue( $key );
    };
    if ($@) {
        if ($@ =~ /Got unexpected status 403/) {
            warn "Can't access st/$key. Got 403.";
            return;
        } else {
            die $@;
        }
    }

    return $ticket;
}



1;
