package Partner2::Juggler::API;

=encoding UTF-8

=cut

use strict;
use warnings FATAL => 'all';
use utf8;

use Scalar::Util qw(blessed);
use JSON::XS;
use LWP::UserAgent;
use HTTP::Request;
use Digest::MD5 qw(md5_hex);
use JSV::Validator;
use Carp qw(croak);

use Partner2::Juggler::API::Schema::Aggregator;
use Partner2::Juggler::API::Schema::Notification;

# https://wiki.yandex-team.ru/sm/juggler/

our $CONFIG_PATH = '/etc/partner2-juggler-api/config.json';

our %URLS = (
    checks_url => 'http://juggler-api.search.yandex.net/api/',
    push_url   => 'http://juggler-push.search.yandex.net/',
);

my %CHAR_TO_SECONDS = (
    s => 1,
    m => 60,
    h => 60 * 60,
);

my $CHAR = join('', keys(%CHAR_TO_SECONDS));

my %KWARGS_STATUSES = map {$_ => 1} qw(OK INFO WARN CRIT);

sub new {
    my ($self, %opts) = @_;

    foreach my $option (qw(juggler_token host namespace)) {
        die sprintf('Expected option "%s"', $option) unless defined($opts{$option});
    }

    my $class = blessed($self) // $self;

    my $obj = bless {opts => \%opts}, $class;

    $obj->init();

    return $obj;
}

sub init {
    my ($self) = @_;

    $self->{'__JSON__'} //= JSON::XS->new->utf8->allow_nonref;

    my $config_json = '{}';
    if (-e $CONFIG_PATH) {
        $config_json = read_file($CONFIG_PATH);
    }

    $self->{'__CONFIG__'} = {
        %URLS, %{$self->decode_from_json($config_json)},
        md5_hex_config => md5_hex($config_json),
        %{$self->{'opts'}},
    };

    $self->{'__LWP__'} = LWP::UserAgent->new(timeout => $self->get_option('timeout', 300));
}

sub decode_from_json {
    my ($self, $json) = @_;

    die 'Expected json' unless defined($json);

    my $original_json = $json;

    utf8::encode($json);
    my $result;
    eval {$result = $self->{'__JSON__'}->decode($json);};

    if ($@) {
        my ($error) = ($@ =~ m'(.+) at /');
        $error //= $@;
        die sprintf("Error: %s\nInput:\n%s\n", $error, $original_json);
    }

    return $result;
}

sub encode_to_json {
    my ($self, $data) = @_;

    my $json = $self->{'__JSON__'}->pretty->canonical->encode($data);

    return $json;
}

sub get_option {
    my ($self, $name, $default) = @_;

    die 'Expected name' unless defined($name);

    return $self->{'__CONFIG__'}{$name} // $default
      // die sprintf('Can not find option "%s" in config: %s', $name, $CONFIG_PATH);
}

sub call {
    my ($self, $method, $uri, %opts) = @_;

    my $data = delete($opts{'data'});

    my $uri_obj = URI->new($uri);
    $uri_obj->query_form(%opts);

    my $request = HTTP::Request->new($method => $uri_obj);
    $request->header('authorization' => 'OAuth ' . $self->get_option('juggler_token'));
    $request->header('Content-Type'  => 'application/json');

    if (defined($data)) {
        $request->content($self->encode_to_json($data));
    }

    my $attempts = $self->get_option('attempts', 3);
    my $delay    = $self->get_option('delay',    1);

    my $response         = undef;
    my $response_content = undef;
    my $msg              = 'Response undefined';

    for my $retry (1 .. $attempts) {
        if ($retry - 1) {
            warn sprintf("RETRY #%s. SLEEP %s", $retry, $delay);
            sleep($delay);
        }

        $response = $self->{'__LWP__'}->request($request);

        if ($response->is_success()) {
            $response_content = $response->decoded_content();

            if (defined($response_content)) {
                last;
            }
        } else {
            $msg = $response->status_line;

            $response_content = eval {$response->decoded_content()};
            $msg .= "\nResponse: " . substr($response_content // '', 0, 500) if $response_content;

            my $response_heders = $response->headers->as_string;
            $msg .= "\nHeaders: " . substr($response_heders, 0, 500) if $response_heders;

            warn "Got error response\n";
            warn $msg;

            last
              if $response->code =~ m/4\d\d/;
        }
    }

    die sprintf('Can not get result: %s', $msg) unless defined($response_content);

    return $self->decode_from_json($response_content);
}

sub _check_options {
    my ($self, $options, $opts, $prefix) = @_;
    $prefix //= '';
    my %known_options = map {$_ => 1} @$options;

    my @unknown_options = grep {!$known_options{$_}} keys(%$opts);
    die sprintf('Unknown options: %s%s', $prefix, join(", $prefix", @unknown_options)) if @unknown_options;

    return 1;
}

sub list_checks {
    my ($self, %opts) = @_;

    $self->_check_options([qw(host_name service_name namespace_name)], \%opts);

    die sprintf('Max length for "service_name" equals 128 symbols')
      if defined($opts{'service_name'}) && length($opts{'service_name'}) > 128;

    my $result = $self->call(
        'GET',
        $self->get_option('checks_url') . 'checks/list_checks',
        'do' => 1,
        %opts
    );

    return $result;
}

sub _check_add_or_update_options {
    my ($self, %opts) = @_;

    JSV::Validator->load_environments("draft4");
    my $v = JSV::Validator->new(
      environment => "draft4"
      );

    my $aggregator_schema = aggregator_schema_map($opts{aggregator});
    unless (defined($aggregator_schema)) {
        croak "Unknown aggregator type: $opts{aggregator}";
    }

    my $result = $v->validate($aggregator_schema, \%opts);
    _check_validation_result($result);

    if ($opts{'notifications'}) {
        my $i = 0;
        for my $notification (@{$opts{'notifications'}}) {
            my $notification_schema = notification_schema_map($notification->{template_name});
            unless (defined($notification_schema)) {
                croak sprintf("Unknown notification type: %s", $notification->{template_name} // 'undef');
            }
            my $r = $v->validate($notification_schema, $notification);
            _check_validation_result($r, "/notifications/$i");

            my $template_kwargs = $notification->{template_kwargs};

            if (exists($template_kwargs->{time_start}) && exists($template_kwargs->{time_end})) {
                croak
                  sprintf('"time_start" must be less than "time_end" in "/notifications/%d/template_kwargs"', $i)
                  if $template_kwargs->{'time_start'} gt $template_kwargs->{'time_end'};
            } elsif (exists($template_kwargs->{time_start}) || exists($template_kwargs->{time_end})) {
                croak sprintf(
                    'Options "time_start" and "time_end" must be together in "/notifications/%d/template_kwargs"',
                    $i);
            }

            if (exists($template_kwargs->{day_start}) && exists($template_kwargs->{day_end})) {
                croak
                  sprintf('"day_start" must be less than "day_end" in "/notifications/%d/template_kwargs"', $i)
                  if $template_kwargs->{'day_start'} gt $template_kwargs->{'day_end'};
            } elsif (exists($template_kwargs->{day_start}) || exists($template_kwargs->{day_end})) {
                croak sprintf(
                    'Options "day_start" and "day_end" must be together in "/notifications/%d/template_kwargs"',
                    $i);
            }

            $i += 1;
        }

    }

    if (!exists($opts{'active'})) {
        croak sprintf('Option "active_kwargs" depends on "active"') if defined($opts{'active_kwargs'});
    }

    return 1;
}

sub _check_add_or_update_notify_rule_options {
    my ($self, %opts) = @_;

    JSV::Validator->load_environments("draft4");
    my $v = JSV::Validator->new(
      environment => "draft4"
      );

    my $notify_rule_schema = notify_rule_schema_map($opts{template_name});
    unless (defined($notify_rule_schema)) {
        croak sprintf("Unknown notification type: %s", $opts{template_name} // 'undef');
    }

    my $result = $v->validate($notify_rule_schema, \%opts);
    _check_validation_result($result);

    return 1;
}

sub add_or_update {
    my ($self, %opts) = @_;

    #https://wiki.yandex-team.ru/sm/juggler/objects/
    $self->_check_add_or_update_options(%opts);

    $opts{'namespace'} //= $self->get_option('namespace');
    $opts{'host'}      //= $self->get_option('host');
    $opts{'ttl'} = _ttl_to_seconds($opts{'ttl'});

    my $result = $self->call(
        'POST',
        $self->get_option('checks_url') . 'checks/add_or_update',
        do   => 1,
        data => \%opts
    );

    die 'Request to a juggler finished with a not success status'
      unless $result->{'success'};

    return $result;
}

sub remove_check {
    my ($self, %opts) = @_;

    $self->_check_options([qw(host_name service_name namespace_name)], \%opts);

    die 'Option "host_name" is required' unless defined($opts{'host_name'});

    if (!defined($opts{'service_name'}) || length($opts{'service_name'}) > 128) {
        die sprintf('Length of a "service_name" cannot be more than 128 symbols');
    }

    die 'Option "namespace_name" is required' unless defined($opts{'namespace_name'});

    my $result = $self->call(
        'GET',
        $self->get_option('checks_url') . 'checks/remove_check',
        'do' => 1,
        %opts
    );

    return $result;
}

sub remove_rule {
    my ($self, $rule_id) = @_;

    my $result = $self->call(
        'GET',
        $self->get_option('checks_url') . 'notify_rules/remove_notify_rule',
        'rule_id' => $rule_id,
        'do' => 1,
    );

    return $result;
}

sub get_equally_notify_rules {
    my ($self, %opts) = @_;

    my $result = $self->call(
        'POST',
        $self->get_option('checks_url') . 'notify_rules/get_notify_rules',
        'do' => 1,
        data => {
            filters => [{
                namespace => $opts{namespace}
            }],
        }
    );

    die 'Request to a juggler finished with a not success status'
        unless $result->{'rules'};
    # в response нет атрибута success как в других запросах

    my $opts_tag = $self->_get_tag_from_selector($opts{selector});
    my @result = grep {$self->_get_tag_from_selector($_->{selector}) eq $opts_tag} @{$result->{rules}};

    return @result;
}

sub add_or_update_notify_rule {
    my ($self, %opts) = @_;

    $self->_check_add_or_update_notify_rule_options(%opts);

    my @equally_notify_rules = $self->get_equally_notify_rules(%opts);
    my $last_id = shift @equally_notify_rules;
    for my $rule (@equally_notify_rules) {
        $self->remove_rule($rule->{rule_id});
    }
    if ($last_id) {
        $opts{rule_id} = $last_id->{rule_id};
        delete $opts{creation_time};
        delete $opts{hints};
    }

    my $result = $self->call(
        'POST',
        $self->get_option('checks_url') . 'notify_rules/add_or_update_notify_rule',
        'do' => 1,
        data => \%opts
    );

    die 'Request to a juggler finished with a not success status'
      unless $result->{'success'};

    return $result;
}

sub send {
    my ($self, %opts) = @_;

    die 'Option "events" is required' if !defined($opts{'events'}) || @{$opts{'events'}} == 0;

    $opts{'source'} //= $self->get_option('namespace');
    my $host = $self->get_option('host');

    my $count = 0;
    foreach my $event (@{$opts{'events'}}) {
        $self->_check_options([qw(host service status description)], $event);

        $event->{'host'} //= $host;
        $event->{'description'} //= 'event';

        if (!defined($event->{'service'}) || length($event->{'service'}) > 128) {
            die sprintf('Length of a "/events/[%d]/service" cannot be more than 128 symbols', $count);
        }

        die
          sprintf('Option "/events/[%d]/status" must be from set: %s', $count, join(', ', sort keys(%KWARGS_STATUSES)))
          if !defined($event->{'status'}) || !$KWARGS_STATUSES{$event->{'status'}};

        $count++;
    }

    my $result = $self->call('POST', $self->get_option('push_url') . 'events', data => \%opts);

    die 'Request to a juggler finished with a not success status'
      unless $result->{'success'};

    if (@{$opts{'events'}} != $result->{'accepted_events'}) {
        my @errors = ();
        for (my $i = 0; $i < @{$result->{'events'}}; $i++) {
            my $res = $result->{'events'}[$i];

            if ($res->{'code'} != 200) {
                push(@errors, $opts{'events'}->[$i]);
            }
        }

        my $errors_str = $self->encode_to_json(\@errors);
        utf8::decode($errors_str);
        die sprintf('Invalid events: %s', $errors_str);
    }

    return $result;
}

sub read_file {
    my ($path) = @_;

    open(my $fh, '<:utf8', $path) or die $!;

    my $content = join('', <$fh>);

    close($fh);

    return $content;
}

sub _get_url {
    my ($self, $method) = @_;

    return $URLS{$method};
}

sub _ttl_to_seconds {
    my ($ttl) = @_;

    die 'Option "ttl" is required' unless defined($ttl);

    my ($number, $char) = $ttl =~ /^([1-9][0-9]*)([$CHAR]*)\z/;

    die 'Expected format like "5s"' unless defined($number);

    if ($char) {
        $number *= $CHAR_TO_SECONDS{$char};
    }

    return $number;
}

sub _check_validation_result {
    my ($r, $prefix) = @_;

    $prefix //= "";

    if ($r->get_error()) {
        my @errors  = ($r->get_error());
        if ($errors[0]{pointer}) {
            croak "$prefix$errors[0]{pointer}: $errors[0]{message}";
        } else {
            croak $errors[0]{message};
        }
    }
}

sub _get_tag_from_selector {
    my ($self, $selector) = @_;
    my $tag = $selector;
    $tag =~ s/^.*tag\s*=\s*(\w+).*/$1/;
    return $tag;
}

1;
