package Partner2::Juggler::API::Schema::Aggregator;

use strict;
use warnings FATAL => 'all';
use utf8;

use Exporter;
use JSV::Validator;
use Partner2::Juggler::API::Schema::Util;

our @ISA       = qw(Exporter);
our @EXPORT_OK = qw(
    aggregator_schema_map
  );
our @EXPORT = @EXPORT_OK;


my %NODATA_MODE_TYPES = map {$_ => 1} qw(
  force_crit
  force_ok
  skip
  );

my %DOWNTIMES_MODE_TYPES = map {$_ => 1} qw(
  force_ok
  ignore
  skip
  );

my %UNREACH_MODE_TYPES = map {$_ => 1} qw(
  force_ok
  skip
  );

my %MODE_TYPES = map {$_ => 1} qw(
  normal
  percent
  );

my %CHILDREN_TYPES = map {$_ => 1} qw(
  HOST
  CGROUP
  CPROJECT
  SKYLI
  SKYLH
  NANNY
  QLOUD
  GENCFG
  EVENTS
  BETWEEN_DC
  INSIDE_DC
  INSIDE_SWITCH
  NOCRT
  YP_PODSET
  YP_ENDPOINT
  );

my %ACTIVE = map {$_ => 1} qw(
  icmpping
  tcp_chat
  ssh
  http
  https
  nanny_deploy_status
  graphite
  x509_cert
  https_cert
  smtp
  imap
  dns
  netmon
  );

sub check_schema {
    return { type => 'string', pattern => '^[^:]*:[^:]+$' };
}

sub unreach_service_schema {
    return {
        type => 'array',
        minItems => 1,
        items => {
            type => 'object',
            properties => {
                check => check_schema(),
                hold => { type => 'integer', minimum => 0 },
            },
            additionalProperties => JSON::false,
        },
    };
}

sub desc_schema {
    return { type => 'string' };
}

sub problem_limit_schema {
    return { type => 'string', pattern => '^\d+%?$' };
}

sub limits_schema {
    return {
        type => 'array',
        minItems => 1,
        items => {
            type => 'object',
            properties => {
                day_start_property(),
                day_end_property(),
                time_start_property(),
                time_end_property(),
                warn => problem_limit_schema(),
                crit => problem_limit_schema(),
            },
        },
    };
}

sub common_aggregator_kwargs_properties {
    return (
        nodata_mode => { type => 'string', enum => [keys(%NODATA_MODE_TYPES)] },
        hold_crit => { type => 'integer', minimum => 0 },
        downtimes_mode => { type => 'string', enum => [keys(%DOWNTIMES_MODE_TYPES)] },
        unreach_checks => { type => 'array', minItems => 1, items => check_schema() },
        unreach_mode => { type => 'string', enum => [keys(%UNREACH_MODE_TYPES)] },
        unreach_service => unreach_service_schema(),
        ok_desc => desc_schema(),
        warn_desc => desc_schema(),
        crit_desc => desc_schema(),
        nodata_desc => desc_schema(),
    );
}

sub logic_or_aggregator_kwargs_schema {
    return {
        type => 'object',
        properties => {
            common_aggregator_kwargs_properties(),
        },
        additionalProperties => JSON::false,
    };
}

sub logic_and_aggregator_kwargs_schema {
    return {
        type => 'object',
        properties => {
            common_aggregator_kwargs_properties(),
        },
        additionalProperties => JSON::false,
    };
}

sub more_than_limit_is_crit_aggregator_kwargs_schema {
    return {
        type => 'object',
        properties => {
            common_aggregator_kwargs_properties(),
            limit => { type => 'integer', minimum => 0 },
            percent => { type => 'integer', minimum => 0, maximum => 100 },
        },
        additionalProperties => JSON::false,
    };
}

sub more_than_limit_is_problem_aggregator_kwargs_schema {
    return {
        type => 'object',
        properties => {
            common_aggregator_kwargs_properties(),
            mode => { type => 'string', enum => [keys(%MODE_TYPES)] },
            crit_limit => { type => 'integer', minimum => 0 },
            warn_limit => { type => 'integer', minimum => 0 },
            show_ok => { type => 'boolean' },
        },
        additionalProperties => JSON::false,
    };
}

sub timed_more_than_limit_is_problem_aggregator_kwargs_schema {
    return {
        type => 'object',
        properties => {
            common_aggregator_kwargs_properties(),
            limits => limits_schema(),
        },
        additionalProperties => JSON::false,
    };
}

sub common_aggregator_properties {
    return (
        host => {type => 'string', minLength => 1, maxLength => 192},
        service => {type => 'string', minLength => 1, maxLength => 128},
        active => {
            type => 'string',
            enum => [keys(%ACTIVE)]
        },
        active_kwargs => {},
        refresh_time => { type => 'integer', minimum => 5, maximum => 300 },
        ttl => { type => 'string', pattern => '^[1-9][0-9]*[smh]*$' },
        description => { type => 'string', maxLength => 1024 },
        pronounce => { type => 'string' },
        tags => { type => 'array', items => { type => 'string', maxLength => 128 } },
        meta => {},
        flaps => {},
        namespace => { type => 'string' },
        notifications => {
            type => 'array',
            items => {
                type => 'object',
                properties => {
                    template_name => { type => 'string' },
                    template_kwargs => { type => 'object' },
                },
                required => [qw(template_name template_kwargs)],
            },
        },
    );
}

sub children_property() {
    return (
        children => {
            type => 'array',
            items => {
                type => 'object',
                properties => {
                    host => { type => 'string'},
                    service => {type => 'string', minLength => 1, maxLength => 128},
                    type => {
                        type => 'string',
                        enum => [keys(%CHILDREN_TYPES)],
                    },
                    instance => { type => 'string'},
                },
                required => [qw(host service)],
                additionalProperties => JSON::false,
            },
        },
    );
}

sub aggregator_properties_map {
    my ($aggregator_name) = @_;

    $aggregator_name //= 'NONE';

    my $map = {
        'NONE' =>  {
            common_aggregator_properties(),
        },
        logic_or =>  {
            aggregator => literal_string('logic_or'),
            aggregator_kwargs => logic_or_aggregator_kwargs_schema(),
            common_aggregator_properties(),
            children_property(),
        },
        logic_and => {
            aggregator => literal_string('logic_and'),
            aggregator_kwargs => logic_and_aggregator_kwargs_schema(),
            common_aggregator_properties(),
            children_property(),
        },
        more_than_limit_is_crit => {
            aggregator => literal_string('more_than_limit_is_crit'),
            aggregator_kwargs => more_than_limit_is_crit_aggregator_kwargs_schema(),
            common_aggregator_properties(),
            children_property(),
        },
        more_than_limit_is_problem => {
            aggregator => literal_string('more_than_limit_is_problem'),
            aggregator_kwargs => more_than_limit_is_problem_aggregator_kwargs_schema(),
            common_aggregator_properties(),
            children_property(),
        },
        timed_more_than_limit_is_problem => {
            aggregator => literal_string('timed_more_than_limit_is_problem'),
            aggregator_kwargs => timed_more_than_limit_is_problem_aggregator_kwargs_schema(),
            common_aggregator_properties(),
            children_property(),
        },
    };

    return %{$map->{$aggregator_name} // {}};
}

sub aggregator_schema_map {
    my ($aggregator_name) = @_;

    my @properties = aggregator_properties_map($aggregator_name);

    return undef unless @properties;

    return {
        type => 'object',
        properties => {
            @properties
        },
        required => [qw(service ttl)],
        additionalProperties => JSON::false,
    };
};

1;
