package Partner2::Juggler::API::Schema::Notification;

use strict;
use warnings FATAL => 'all';
use utf8;

use JSV::Validator;
use Exporter;
use JSV::Validator;
use Partner2::Juggler::API::Schema::Util;

our @ISA       = qw(Exporter);
our @EXPORT_OK = qw(
    notification_schema_map
    notify_rule_schema_map
  );
our @EXPORT = @EXPORT_OK;


my %KWARGS_STATUSES = map {$_ => 1} qw(OK INFO WARN CRIT);

my %KWARGS_METHODS = map {$_ => 1} qw(
  phone
  telegram
  email
  sms
  );

sub escalation_repeat_schema {
    return { type => 'integer', minimum => 0,  maximum => 100 };
}

sub repeat_schema {
    return { type => 'integer', minimum => 0,  maximum => 864000 };
}

sub logins_schema {
    return { type => 'array', minItems => 1, items => { type => 'string', minLength => 1} };
}

sub common_notification_kwargs_properties {
    return (
        time_start_property(),
        time_end_property(),
        day_start_property(),
        day_end_property(),
        delay => { type => 'integer', minimum => 0 },
        igonre_gaps => { type => 'boolean' },
    );
}

sub phone_escalation_kwargs_schema {
    return {
        type => 'object',
        properties => {
            common_notification_kwargs_properties(),
            call_tries => { type => 'integer', minimum => 1,  maximum => 5 },
            on_success_next_call_delay => { type => 'boolean' },
            restart_after => { type => 'integer', minimum => 1800,  maximum => 604800 },
            repeat => escalation_repeat_schema(),
            logins => logins_schema(),
        },
        required => [qw(logins)],
        additionalProperties => JSON::false,
    };
}

sub on_change_kwargs_schema {
    return {
        type => 'object',
        properties => {
            common_notification_kwargs_properties(),
            repeat => repeat_schema(),
            login => logins_schema(),
            golem_responsible => { type => 'boolean' },
            calendar_id => {},
            status => { type => 'array', items => {type => 'string', enum => [keys(%KWARGS_STATUSES)]} },
            method => { type => 'array', minItems => 1, items => { type => 'string', enum => [keys(%KWARGS_METHODS)] } },
            ignore_golem => { type => 'boolean' },
            ignore_weekends => { type => 'boolean' },
            description_globs => { type => 'array', minItems => 1, items => { type => 'string', minLength => 1} },
            min_interval => { type => 'integer', minimum => 0 },
        },
        required => [qw(status method)],
        additionalProperties => JSON::false,
    };
}

sub on_status_change_kwargs_schema {
    return on_change_kwargs_schema();
}

sub on_desc_change_kwargs_schema {
    return on_change_kwargs_schema();
}

sub notification_properties_map {
    my ($template_name) = @_;

    $template_name //= 'UNKNOWN';

    my $map = {
        on_status_change => {
            template_name => literal_string('on_status_change'),
            template_kwargs => on_status_change_kwargs_schema(),
        },
        on_desc_change => {
            template_name => literal_string('on_desc_change'),
            template_kwargs => on_desc_change_kwargs_schema(),
        },
        phone_escalation => {
            template_name => literal_string('phone_escalation'),
            template_kwargs => phone_escalation_kwargs_schema(),
        },
    };

    return %{$map->{$template_name} // {}};
}

sub notification_schema_map {
    my ($template_name) = @_;

    my @properties = notification_properties_map($template_name);

    return undef unless @properties;

    return {
        type => 'object',
        properties => {
            notification_properties_map($template_name),
        },
    };
}

sub notify_rule_schema_map {
    my ($template_name) = @_;

    my @properties = notification_properties_map($template_name);

    return undef unless @properties;

    return {
        type => 'object',
        properties => {
            notification_properties_map($template_name),
            rule_id => { type => 'string' },
            selector => { type => 'string' },
            namespace => { type => 'string' },
            description => { type => 'string' },
            match_raw_events => { type => 'boolean' },
            check_kwargs => { type => 'object' },
        },
        additionalProperties => JSON::false,
    };
}

1;
