#!/usr/bin/perl

use strict;
use warnings FATAL => 'all';
use utf8;

use Test::More tests => 18;
use Test::Differences qw(eq_or_diff);

use Partner2::Juggler::API;

{
    no warnings 'once';
    $Partner2::Juggler::API::CONFIG_PATH = './t/config.json';
}

my $FIXTURES = [
    {
        name   => 'without params',
        mock   => sub { },
        params => [],
        error  => 'The instance properties has not required properties (missing: service, ttl)',
    },
    {
        name => 'unknown param',
        mock => mock_call(opts => {do => 1},),
        params => [not_exists => 1],
        error  => 'The instance properties has not required properties (missing: service, ttl)',
    },
    {
        name => 'with service and ttl - success',
        mock => mock_call(
            opts => {
                do   => 1,
                data => {
                    host      => 'test.host',
                    namespace => 'test.namespace',
                    service   => 'service',
                    ttl       => 5
                }
            },
            result => {success => 1}
        ),
        params => [service => 'service', ttl => '5s'],
        result => {success => 1},
    },
    {
        name => 'with service and ttl - error',
        mock => mock_call(
            opts => {
                do   => 1,
                data => {
                    host      => 'test.host',
                    namespace => 'test.namespace',
                    service   => 'service',
                    ttl       => 5
                }
            },
            result => {success => 0}
        ),
        params => [service => 'service', ttl => '5s'],
        error  => 'Request to a juggler finished with a not success status',
    },
    {
        name => 'with namespace',
        mock => mock_call(
            opts => {
                do   => 1,
                data => {
                    host      => 'test.host',
                    namespace => 'namespace',
                    service   => 'service',
                    ttl       => 300
                }
            },
            result => {success => 1}
        ),
        params => [service => 'service', ttl => '5m', namespace => 'namespace'],
        result => {success => 1},
    },
    {
        name => 'with host',
        mock => mock_call(
            opts => {
                do   => 1,
                data => {
                    host      => 'host',
                    namespace => 'test.namespace',
                    service   => 'service',
                    ttl       => 300
                }
            },
            result => {success => 1}
        ),
        params => [service => 'service', ttl => '5m', host => 'host'],
        result => {success => 1},
    },
];

main();

sub main {
    my $pja = Partner2::Juggler::API->new(
        juggler_token => 'juggler-token',
        host          => 'test.host',
        namespace     => 'test.namespace'
    );

    foreach my $test (@$FIXTURES) {
        $test->{'mock'}->();

        my $result;

        eval {$result = $pja->add_or_update(@{$test->{'params'}})};

        if ($@ || $test->{'error'}) {
            $test->{'error'} //= 'ERROR';
            like($@, qr/\Q$test->{'error'}\E/, 'add_or_update ' . $test->{'name'});

            next;
        }

        eq_or_diff($result, $test->{'result'}, 'add_or_update ' . $test->{'name'});
    }
}

sub mock_call {
    my (%opts) = @_;

    return sub {
        no warnings 'redefine';
        no strict 'refs';

        *{'Partner2::Juggler::API::call'} = sub {
            my ($self, $method, $uri, %params) = @_;

            is($method, 'POST', 'call: Get expected method');

            is($uri, $Partner2::Juggler::API::URLS{'checks_url'} . 'checks/add_or_update', 'call: Get expected uri');

            eq_or_diff(\%params, $opts{'opts'} // {}, 'call: Get expected opts');

            return $opts{'result'};
        };
    };
}
