#!/usr/bin/perl

use strict;
use warnings FATAL => 'all';
use utf8;

use Test::More tests => 56;
use Test::Differences qw(eq_or_diff);

use Partner2::Juggler::API;

{
    no warnings 'once';
    $Partner2::Juggler::API::CONFIG_PATH = './t/config.json';
}

my $FIXTURES = [
    {
        name => 'Get with params',
        mock => mock_lwp(
            method           => 'GET',
            uri              => 'https://test.juggler.yandex.ru/checks?do=1',
            response_status  => 200,
            response_content => '[]'
        ),
        params => ['GET', 'https://test.juggler.yandex.ru/checks', do => 1],
        result => [],
    },
    {
        name => 'Get with params (400)',
        mock => mock_lwp(
            method           => 'GET',
            uri              => 'https://test.juggler.yandex.ru/checks?do=1',
            response_status  => 400,
            response_content => '{"error":"ERROR"}'
        ),
        params => ['GET', 'https://test.juggler.yandex.ru/checks', do => 1],
        result => {error => 'ERROR'},
    },
    {
        name => 'Get with params (500)',
        mock => mock_lwp(
            method           => 'GET',
            uri              => 'https://test.juggler.yandex.ru/checks?do=1',
            response_status  => 500,
            response_content => '<html>'
        ),
        params => ['GET', 'https://test.juggler.yandex.ru/checks', do => 1],
        error =>
'Error: malformed JSON string, neither array, object, number, string or atom, at character offset 0 (before "<html>")',
    },
    {
        name => 'Get with params (500 without content)',
        mock => mock_lwp(
            method           => 'GET',
            uri              => 'https://test.juggler.yandex.ru/checks?do=1',
            response_status  => 500,
            response_content => undef,
        ),
        params => ['GET', 'https://test.juggler.yandex.ru/checks', do => 1],
        error  => 'Can not get result: 500 Internal Server Error',
    },
    {
        name => 'Post with params',
        mock => mock_lwp(
            method           => 'POST',
            uri              => 'https://test.juggler.yandex.ru/push?do=1',
            response_status  => 200,
            response_content => '[]',
        ),
        params => ['POST', 'https://test.juggler.yandex.ru/push', do => 1],
        result => [],
    },
    {
        name => 'Post with params and content',
        mock => mock_lwp(
            method  => 'POST',
            uri     => 'https://test.juggler.yandex.ru/push?do=1',
            content => '{
   "param" : true
}
',
            response_status  => 200,
            response_content => '[]',
        ),
        params => ['POST', 'https://test.juggler.yandex.ru/push', do => 1, data => {param => \1}],
        result => [],
    },
];

main();

sub main {
    my $pja = Partner2::Juggler::API->new(
        juggler_token => 'juggler-token',
        host          => 'test.host',
        namespace     => 'test.namespace'
    );

    foreach my $test (@$FIXTURES) {
        $test->{'mock'}->();

        my $result;

        eval {$result = $pja->call(@{$test->{'params'}})};

        if ($@ || $test->{'error'}) {
            $test->{'error'} //= 'ERROR';
            like($@, qr/\Q$test->{'error'}\E/, $test->{'name'});

            next;
        }

        eq_or_diff($result, $test->{'result'}, $test->{'name'});
    }
}

sub mock_lwp {
    my (%opts) = @_;

    return sub {
        no warnings 'redefine';
        no strict 'refs';

        *{'LWP::UserAgent::request'} = sub {
            my ($self, $request) = @_;

            is($request->method(), $opts{'method'}, 'request: Get expected request method',);

            is(
                $request->header('authorization'),
                'OAuth juggler-token',
                'request: Get expected request header "authorization"'
              );
            is($request->header('Content-Type'),
                'application/json', 'request: Get expected request header "Content-Type"');

            is($request->uri->as_string, $opts{'uri'}, 'request: Get expected request uri',);

            eq_or_diff($request->content(), $opts{'content'} // '', 'request: Get expected request content',);

            my $r = HTTP::Response->new($opts{'response_status'});
            $r->request($request);
            $r->content($opts{'response_content'});

            return $r;
        };
    };
}
