#!/usr/bin/perl

use strict;
use warnings FATAL => 'all';
use utf8;

use Test::More tests => 18;
use Test::Differences qw(eq_or_diff);

use Partner2::Juggler::API;

{
    no warnings 'once';
    $Partner2::Juggler::API::CONFIG_PATH = './t/config.json';
}

my $FIXTURES = [
    {
        name => 'without params',
        mock => mock_call(
            opts   => {do => 1},
            result => {},
        ),
        params => [],
        result => {},
    },
    {
        name => 'with unknown param',
        mock => mock_call(opts => {do => 1},),
        params => [not_exists => 1],
        error  => 'Unknown options: not_exists',
    },
    {
        name => 'with uncorrected service_name',
        mock => mock_call(opts => {do => 1, service_name => 'a' x 129},),
        params => [service_name => 'a' x 129],
        error  => 'Max length for "service_name" equals 128 symbols',
    },
    {
        name => 'with service_name',
        mock => mock_call(
            opts   => {do => 1, service_name => 'service_name'},
            result => {},
        ),
        params => [service_name => 'service_name'],
        result => {},
    },
    {
        name => 'with host_name',
        mock => mock_call(
            opts   => {do => 1, host_name => 'host_name'},
            result => {},
        ),
        params => [host_name => 'host_name'],
        result => {},
    },
    {
        name => 'with namespace_name',
        mock => mock_call(
            opts   => {do => 1, namespace_name => 'namespace_name'},
            result => {},
        ),
        params => [namespace_name => 'namespace_name'],
        result => {},
    },
];

main();

sub main {
    my $pja = Partner2::Juggler::API->new(
        juggler_token => 'juggler-token',
        host          => 'test.host',
        namespace     => 'test.namespace'
    );

    foreach my $test (@$FIXTURES) {
        $test->{'mock'}->();

        my $result;

        eval {$result = $pja->list_checks(@{$test->{'params'}})};

        if ($@ || $test->{'error'}) {
            $test->{'error'} //= 'ERROR';
            like($@, qr/\Q$test->{'error'}\E/, 'list_checks ' . $test->{'name'});

            next;
        }

        eq_or_diff($result, $test->{'result'}, 'list_checks ' . $test->{'name'});
    }
}

sub mock_call {
    my (%opts) = @_;

    return sub {
        no warnings 'redefine';
        no strict 'refs';

        *{'Partner2::Juggler::API::call'} = sub {
            my ($self, $method, $uri, %params) = @_;

            is($method, 'GET', 'call: Get expected method');

            is($uri, $Partner2::Juggler::API::URLS{'checks_url'} . 'checks/list_checks', 'call: Get expected uri');

            eq_or_diff(\%params, $opts{'opts'} // {}, 'call: Get expected opts');

            return $opts{'result'};
        };
    };
}
