#!/usr/bin/perl

use strict;
use warnings FATAL => 'all';
use utf8;

use Test::More tests => 22;
use Test::Differences qw(eq_or_diff);

use Partner2::Juggler::API;

{
    no warnings 'once';
    $Partner2::Juggler::API::CONFIG_PATH = './t/config.json';
}

my $FIXTURES = [
    {
        name   => 'without params',
        mock   => sub { },
        params => [],
        error  => 'Option "events" is required',
    },
    {
        name   => 'with events',
        mock   => sub { },
        params => [events => []],
        error  => 'Option "events" is required'
    },
    {
        name   => 'with empty events',
        mock   => sub { },
        params => [events => [{}]],
        error  => 'Length of a "/events/[0]/service" cannot be more than 128 symbols',
    },
    {
        name   => 'with uncorrected events.[0].service',
        mock   => sub { },
        params => [events => [{service => 'a' x 129}]],
        error  => 'Length of a "/events/[0]/service" cannot be more than 128 symbols',
    },
    {
        name   => 'with events.[0].service',
        mock   => sub { },
        params => [events => [{service => 'service'}]],
        error  => 'Option "/events/[0]/status" must be from set: CRIT, INFO, OK, WARN'
    },
    {
        name   => 'with events.[0].service and uncorrected events.[0].status',
        mock   => sub { },
        params => [events => [{service => 'service', status => 'not_exists'}]],
        error  => 'Option "/events/[0]/status" must be from set: CRIT, INFO, OK, WARN'
    },
    {
        name => 'with events.[0].service and events.[0].status',
        mock => mock_call(
            opts => {
                data => {
                    events => [
                        {
                            description => 'event',
                            host        => 'test.host',
                            service     => 'service',
                            status      => 'OK'
                        }
                    ],
                    source => 'test.namespace'
                }
            },
            result => {
                accepted_events => 1,
                events          => [{code => 200}],
                success         => 1
            }
        ),
        params => [events => [{service => 'service', status => 'OK'}]],
        result => {
            accepted_events => 1,
            events          => [{code => 200}],
            success         => 1
        }
    },
    {
        name => 'with events.[0].service and events.[0].status - error',
        mock => mock_call(
            opts => {
                data => {
                    events => [
                        {
                            description => 'event',
                            host        => 'test.host',
                            service     => 'service',
                            status      => 'OK'
                        }
                    ],
                    source => 'test.namespace'
                }
            },
            result => {
                accepted_events => 1,
                success         => 0
            }
        ),
        params => [events => [{service => 'service', status => 'OK'}]],
        error  => 'Request to a juggler finished with a not success status'
    },
    {
        name => 'with events.[0].service and events.[0].status - error for events',
        mock => mock_call(
            opts => {
                data => {
                    events => [
                        {
                            description => 'event',
                            host        => 'test.host',
                            service     => 'service',
                            status      => 'OK'
                        }
                    ],
                    source => 'test.namespace'
                }
            },
            result => {
                accepted_events => 0,
                events          => [{code => 400}],
                success         => 1
            }
        ),
        params => [events => [{service => 'service', status => 'OK'}]],
        error  => 'Invalid events: [
   {
      "description" : "event",
      "host" : "test.host",
      "service" : "service",
      "status" : "OK"
   }
]
'
    },
    {
        name => 'with source and events.[0].host and events.[0].description',
        mock => mock_call(
            opts => {
                data => {
                    events => [
                        {
                            description => 'description',
                            host        => 'host',
                            service     => 'service',
                            status      => 'OK'
                        }
                    ],
                    source => 'source'
                }
            },
            result => {
                accepted_events => 1,
                events          => [{code => 200}],
                success         => 1
            }
        ),
        params => [
            source => 'source',
            events => [{service => 'service', host => 'host', status => 'OK', description => 'description'}]
        ],
        result => {
            accepted_events => 1,
            events          => [{code => 200}],
            success         => 1
        }
    },
];

main();

sub main {
    my $pja = Partner2::Juggler::API->new(
        juggler_token => 'juggler-token',
        host          => 'test.host',
        namespace     => 'test.namespace'
    );

    foreach my $test (@$FIXTURES) {
        $test->{'mock'}->();

        my $result;

        eval {$result = $pja->send(@{$test->{'params'}})};

        if ($@ || $test->{'error'}) {
            $test->{'error'} //= 'ERROR';
            like($@, qr/\Q$test->{'error'}\E/, 'send ' . $test->{'name'});

            next;
        }

        eq_or_diff($result, $test->{'result'}, 'send ' . $test->{'name'});
    }
}

sub mock_call {
    my (%opts) = @_;

    return sub {
        no warnings 'redefine';
        no strict 'refs';

        *{'Partner2::Juggler::API::call'} = sub {
            my ($self, $method, $uri, %params) = @_;

            is($method, 'POST', 'call: Get expected method');

            is($uri, $Partner2::Juggler::API::URLS{'push_url'} . 'events', 'call: Get expected uri');

            eq_or_diff(\%params, $opts{'opts'} // {}, 'call: Get expected opts');

            return $opts{'result'};
        };
    };
}
