package Check;

use strict;
use warnings FATAL => 'all';
use utf8;
use open qw(:std :utf8);

use Carp;
use JSON::PP;

use lib::abs qw(
  ../lib
  );

use Constants qw ($DEPLOY_HOSTS $ENV_PROD %ENV_TAGS);
use Util;

sub new {
    my ($self, %opts) = @_;

    my $class = ref($self) || $self;

    return bless \%opts, $class;
}

sub create {
    my ($self, $pja) = @_;

    my $checks = $self->get_juggler_checks();
    push(@$checks, @{$self->get_juggler_aggregator_checks($checks)});

    my @hosts = @{decode_json(get_content('https://partner.yandex-team.ru/api/0/all_hosts_with_dc.json'))};

    my %host_dc_hash;

    foreach my $host (@hosts) {
        $host_dc_hash{$host->{hostname}} = $host->{dc};
    }

    foreach my $check (@$checks) {
        delete($check->{'options'});
        $self->_add_source_tag($check);
        $self->_add_environment_tag_if_not_exists($check);
        $self->_add_wiki_if_not_exist($check);
        $pja->add_or_update(%$check);
    }
}

sub send_events {
    my ($self) = @_;

    my $pja = get_partner2_juggler_api();

    my $checks = $self->get_juggler_checks();

    foreach my $check (@$checks) {
        $self->check($check, $pja);
    }
}

sub get_juggler_aggregator_checks {[]}

sub get_dashboard_tag {
    my ($self) = @_;
    # Тег автоматически добавляется при добавлении любих нотификаций на проверку
    return (
        'show-on-dashboard',
    );
}

sub get_notification_tags {
    my ($self) = @_;
    return (
        $self->get_dashboard_tag(),
        'standard-notification',
    );
}

sub get_extended_notification_tags {
    my ($self) = @_;
    return (
        $self->get_dashboard_tag(),
        'extended-notification',
    );
}

sub get_support_notification_tags {
    my ($self) = @_;
    return (
        $self->get_dashboard_tag(),
        'support-notification',
    );
}

sub get_support_tg_notification_tags {
    my ($self) = @_;
    return (
        $self->get_dashboard_tag(),
        'support-tg-notification',
    );
}

sub get_shorttime_notification_tags {
    my ($self) = @_;
    return (
        $self->get_dashboard_tag(),
        'shorttime-notification',
    );
}

sub get_fulltime_notification_tags {
    my ($self) = @_;
    return (
        $self->get_dashboard_tag(),
        'fulltime-notification',
    );
}

sub get_fast_notification_tags {
    my ($self) = @_;
    return (
        $self->get_dashboard_tag(),
        'fast-notification',
    );
}

sub get_host_by_stage {
    my ($self, $env) = @_;

    return (
        sprintf('partner-%s', $env)
    );
}

sub get_host_by_deploy_stage {
    my ($self, $deploy_stage, %opts) = @_;

    my $host = $deploy_stage->{'stage'};

    my $units;
    if (defined($opts{'unit'})) {
        croak "Unit \"$opts{'unit'}\" not found" unless exists($deploy_stage->{'units'}{$opts{'unit'}});

        $units = [$opts{'unit'}];
    } else {
        $units = [keys(%{$deploy_stage->{'units'}})];
    }

    croak 'Expected "unit"' if @$units > 1;

    $host .= ".$units->[0]";

    my $dcs;
    if (defined($opts{'dc'})) {
        croak "DC \"$opts{'dc'}\" not found $host" unless in_array($opts{'dc'}, $deploy_stage->{'units'}{$units->[0]}{'dcs'});

        $dcs = [$opts{'dc'}];
    } else {
        $dcs = [keys(%{$deploy_stage->{'units'}{$units->[0]}{'dcs'}})];
    }

    croak 'Expected "dc"' if @$dcs > 1;

    $host .= "_cluster_$dcs->[0]";

    return $host;
}

sub get_geo_tag_by_dc() {
    my ($self, $dc) = @_;
    return (
        "a_geo_$dc"
    );
}

sub get_source_tag {
    my ($self) = @_;
    return (
        'partner2-juggler-checks-dev',
    );
}

sub add_wiki_meta_url {
    my ($self, $check, $url) = @_;

    $check->{meta} //={};
    $check->{meta}->{urls} //=[];

    # добавим дефолтное значение
    push @{$check->{meta}->{urls}}, $self->get_wiki_meta_url($url);
}

sub get_wiki_meta_url {
    my ($self, $url) = @_;

    return {
        title => 'Что делать, если горит проверка',
        type  => 'wiki',
        url   => $url,
    };
}

sub get_deploy_stages {
    my ($self, %opts) = @_;

    my $deploy_stages = {};

    foreach my $env (keys(%$DEPLOY_HOSTS)) {
        foreach my $stage_with_unit (keys(%{$DEPLOY_HOSTS->{$env}})) {
            my ($stage, $unit) = $stage_with_unit =~ /^(.+?)\.(.+)$/;

            if ($opts{'stages'} && !in_array($stage, $opts{'stages'})) {
                next;
            }

            if ($opts{'units'} && !in_array($unit, $opts{'units'})) {
                next;
            }

            if ($opts{'has_endpoints'} && !exists($DEPLOY_HOSTS->{$env}{$stage_with_unit}{endpoints})) {
                next;
            }

            $deploy_stages->{$stage}{'stage'} = $stage;
            $deploy_stages->{$stage}{'env'} = $env;

            $deploy_stages->{$stage}{'units'}{$unit} = {
                dcs       => $DEPLOY_HOSTS->{$env}{$stage_with_unit}{'dc'},
                endpoints => $DEPLOY_HOSTS->{$env}{$stage_with_unit}{'endpoints'},
            };
        }
    }
    return $deploy_stages;
}

sub _add_source_tag {
    my ($self, $check) = @_;

    my $source_tag = $self->get_source_tag();
    $check->{tags} //= [];
    push @{$check->{tags}}, $source_tag unless grep { $_ eq $source_tag } @{$check->{tags}};
}

sub _add_environment_tag_if_not_exists {
    my ($self, $check) = @_;

    $check->{tags} //= [];
    push @{$check->{tags}}, $ENV_PROD unless grep { $ENV_TAGS{$_} } @{$check->{tags}};
}

sub _add_geo_tag_by_host {
    my ($self, $check, $host_dc_hash) = @_;
    my $dc = $host_dc_hash->{$check->{host}};

    unless ($dc) {
        return;
    }

    my $geo_tag = $self->get_geo_tag_by_dc($dc);
    $check->{tags} //= [];
    push @{$check->{tags}}, $geo_tag;
}

sub _add_wiki_if_not_exist {
    my ($self, $check) = @_;

    $check->{meta} //={};
    $check->{meta}->{urls} //=[];

    # Проверка может уже добавили wiki
    foreach my $url (@{$check->{meta}->{urls}}) {
        if ($url->{type} eq 'wiki') {
            return;
        }
    }

    # добавим дефолтное значение
    push @{$check->{meta}->{urls}}, $self->get_wiki_meta_url(
        'https://wiki.yandex-team.ru/partner/w/devops/monitoring/juggler-checks/notready/'
    );
}

sub get_warden_alert_create_tag {
    my ($self) = @_;
    return (
        'warden_alert_create_spi',
    );
}

1;
