package Check::BKStatLag;

use strict;
use warnings FATAL => 'all';
use utf8;
use open qw(:std :utf8);

use base qw(Check);

use lib::abs qw(
  ../../lib
);

use Constants qw($DEFAULT_HOST);
use Util;

my $TAGS            = [qw(bk_stat_lag)];
my $BS_GRAPHITE_URL = 'https://bs-mg.yandex-team.ru';
my @YT_CLUSTER_LIST = qw(hahn arnold);
my @METRIC_LIST     = (
    {
        n1 => 'DSPStatChevent',
        n2 => 'bs-chevent-log',
    },
    {
        n1 => 'DSPStatDspChecked',
        n2 => 'bs-dsp-checked-log',
    },
    {
        n1 => 'DSPStatSsp',
        n2 => 'bs-ssp-log',
    },
    {
        n1 => 'DSPStatNewsEvent',
        n2 => 'bs-newsevent-log',
    },
);

sub get_juggler_checks {
    my @checks;
    foreach my $yt_cluster (sort @YT_CLUSTER_LIST) {
        foreach my $metric (sort {$a->{n1} cmp $b->{n1}} @METRIC_LIST) {
            my $t = _get_graphite_path_and_service( $yt_cluster, $metric );
            push @checks,
              {
                service       => $t->{service},
                host          => $DEFAULT_HOST,
                refresh_time  => 300,
                ttl           => '900s',
                active        => 'graphite',
                active_kwargs => {
                    base_url    => $BS_GRAPHITE_URL,
                    metric      => $t->{metric},
                    # получаем данные за 1 минуту, т.к. читаем из графита 1min
                    # нам интересно текущее значение
                    time_window => '-1min',
                    timeout     => 60,
                    WARN        => 'metric > 2700',
                    CRIT        => 'metric > 7200',
                },
                tags => $TAGS,
              };
        }
    }
    return \@checks;
}

sub get_juggler_aggregator_checks {
    my ($self, $checks) = @_;

    my $check = {
        aggregator => 'logic_or',
        children =>
            [ map { {'host' => $DEFAULT_HOST, 'service' => $_->{'service'}} } @$checks ],
        refresh_time  => 300,
        service       => 'bk_stat_lag_agregator',
        ttl           => '900s',
        tags          => [$self->get_support_tg_notification_tags(), @$TAGS],
        pronounce     => 'Статистика БК начала отставать более чем на час',
    };

    $self->add_wiki_meta_url($check,
        'https://wiki.yandex-team.ru/partner/w/devops/monitoring/juggler-checks/bk-stat-lag/');

    return [$check];
}

sub _get_graphite_path_and_service {
    my ( $yt_cluster, $metric ) = @_;
    return {
        metric => sprintf(
            'one_min.%s.%s.%s.delay',
            $yt_cluster, $metric->{n1}, $metric->{n2}
        ),
        service => sprintf( 'bk_stat_lag_%s_%s', $yt_cluster, $metric->{n1} ),
    };
}

1;
