package Check::DBDockerImage;

use strict;
use warnings FATAL => 'all';
use utf8;
use open qw(:std :utf8);

use base qw(Check);

use Carp;
use Moment;
use JSON::PP;

use lib::abs qw(
  ../../lib
  );

use Constants qw ($ENV_TEST $TEST_VIRTUAL_HOST);
use Util;

my $SERVICE        = 'db.docker_image';
my $MIN_IMAGE_SIZE = 1_000_000;

sub get_juggler_checks {
    my ($self) = @_;

    my $check = {service => $SERVICE, host => $TEST_VIRTUAL_HOST, ttl => '25h', tags => [$ENV_TEST],};

    $self->add_wiki_meta_url($check, 'https://wiki.yandex-team.ru/partner/w/devops/monitoring/juggler-checks/db-docker-image/');

    return [$check];
}

sub check {
    my ($self, $check, $pja) = @_;

    my $secrets  = get_secrets();
    my $user     = $secrets->{'registry-partner'}->{'user'};
    my $password = $secrets->{'registry-partner'}->{'password'};

    my $url = sprintf('https://%s:%s@registry.partner.yandex-team.ru/v1', $user, $password);
    my $data = get_json_content("$url/search?q=partner2-db-general");
    #  {
    #     "num_results" : 51,
    #     "query" : "partner2-db",
    #     "results" : [
    #       {
    #         "name" : "library/partner2-db-general-2.18.566-2017-09-20",
    #         "description" : ""
    #       },
    #       ...
    #     ]
    $data = {results => []} unless $data->{results};

    my @list_names = sort map {$_->{name}} @{$data->{'results'}};
    my $latest_image = $list_names[-1] // '';
    my ($latest_image_date) = ($latest_image =~ m|library/partner2-db-general-.*?(\d{4}-\d{2}-\d{2})|a);

    my $now            = Moment->now();
    my $today_date     = $now->get_d();
    my $yesterday_date = $now->minus(day => 1)->get_d();

    my $ok = $latest_image_date;
    my $description = $ok ? '' : 'Latest_image_date not found';
    $ok &&= ($latest_image_date eq $today_date) || ($latest_image_date eq $yesterday_date);
    $description = ($description ne '' || $ok) ? '' : 'Latest_image_date not today and not yesterday. ';
    my $image_size = get_image_size($url, $latest_image);
    $ok &&= $image_size > $MIN_IMAGE_SIZE;
    $description =
      ($description ne '' || $ok)
      ? ''
      : "Image size is small. MIN_IMAGE_SIZE=$MIN_IMAGE_SIZE, IMAGE_SIZE = $image_size";

    send_event($pja, $check, $ok, $description);
}

sub get_image_size {
    my ($url, $image) = @_;

    my $result = eval {
        my $tags = get_json_content("$url/repositories/$image/tags");
        my $latest = $tags->{latest} // '';

        my $list = get_json_content("$url/images/$latest/files");

        my $sum = 0;
        for my $file (@$list) {
            $sum += $file->[3];
        }
        $sum;
    } // 0;

    warn "SIZE: '$image' is $result\n";

    return $result;
}

1;
