package Check::DeployHTTP;

use strict;
use warnings FATAL => 'all';
use utf8;
use open qw(:std :utf8);

use base qw(Check);

use Carp;

use lib::abs qw(
  ../../lib
);

use Constants qw ($ENV_PROD $ENV_PREPROD %ENV_TAGS);
use Util;

sub get_juggler_checks {
    my ($self) = @_;

    my @checks;

    my $deploy_stages = $self->get_deploy_stages(has_endpoints => 1);

    foreach my $stage (sort keys %{$deploy_stages}) {
        my $deploy_stage = $deploy_stages->{$stage};
        foreach my $unit (sort keys %{$deploy_stage->{units}}) {
            foreach my $dc (sort @{$deploy_stage->{units}->{$unit}->{dcs}}) {
                my $host = "$stage.$unit\@cluster=$dc";
                my $safe_host = get_safe_host($host);
                foreach my $endpoint (sort @{$deploy_stage->{units}->{$unit}->{endpoints}} ) {
                    my $safe_endpoint = get_safe_endpoint($endpoint);
                    my $service = "http_check.$safe_endpoint";
                    push @checks,
                        {
                            active        => 'http',
                            active_kwargs => {
                                path => $endpoint,
                                port => 80,
                            },
                            aggregator    => 'logic_or',
                            children      => [
                                {
                                    host    => $host,
                                    service => $service,
                                    type    => 'YP_PODSET',
                                },
                            ],
                            host          => $safe_host,
                            options       => {
                                stage    => $deploy_stage->{'env'},
                            },
                            refresh_time  => 60,
                            service       => $service,
                            ttl           => '180s',
                            tags          => [
                                $deploy_stage->{'env'},
                                $self->get_geo_tag_by_dc($dc)
                            ],
                        };
                }
            }
        }
    }

    return \@checks;
}

sub get_juggler_aggregator_checks {
    my ( $self, $checks ) = @_;

    my @checks;

    foreach my $stage (sort keys %ENV_TAGS) {
        my @children = map {
            {
                host    => $_->{host},
                service => $_->{service},
            }
          }
          grep { $_->{options}->{stage} eq $stage } @$checks;

        unless(@children) {
            next;
        }

        my $check = {
            aggregator   => 'logic_or',
            children     => \@children,
            refresh_time => 60,
            tags         => [
                (
                    $stage eq $ENV_PREPROD || $stage eq $ENV_PROD
                        ? ($self->get_notification_tags(),)
                        : ()
                ),
                $stage,
            ],
            service      => 'partner2_deploy.http_check_aggregator',
            host         => $self->get_host_by_stage($stage),
            ttl          => '180s',
        };

        $self->add_wiki_meta_url($check, "https://wiki.yandex-team.ru/partner/w/devops/monitoring/juggler-checks/http-https/");

        push(@checks, $check);
    }

    return \@checks;
}


sub update_check() {
    my $pja = get_partner2_juggler_api();
    Check::DeployHTTP->new(logins => $pja->get_option("logins"))->create($pja);
}

1;
