package Check::DeployPing;

use strict;
use warnings FATAL => 'all';
use utf8;
use open qw(:std :utf8);

use base qw(Check);

use Carp;

use lib::abs qw(
  ../../lib
  );

use Constants qw ( $DEPLOY_HOSTS $ENV_DEV $ENV_PROD $ENV_TEST $ENV_PREPROD %ENV_TAGS);
use Util;

my $HOSTS = {
    $ENV_PROD => {
        'partner-production-backend-services-stage.Services' => {
            'dc' => [ qw(sas vla) ],
        },
    },
    $ENV_TEST => {
        'partner-test-backend-services-stage.Services' => {
            'dc' => [ qw(sas vla) ],
        },
    },
    $ENV_DEV  => {
        'partner_func-tests_env.teamcity' => {
            'dc' => [ qw(vla) ],
        },
        'creator-frontend.App'            => {
            'dc' => [ qw(iva sas) ],
        },
        'krush.krush'                     => {
            'dc' => [ qw(iva sas) ],
        },
        'krush.neokrush'                  => {
            'dc' => [ qw(iva sas) ],
        },
        'partner-oneshot-db.Database'     => {
            'dc' => [ qw(myt) ],
        }
    },
};


sub get_juggler_checks {
    my ($self) = @_;

    my @checks = ($self->_add_checks($HOSTS), $self->_add_checks($DEPLOY_HOSTS));
    return \@checks;
}

sub _add_checks {
    my ($self, $hosts) = @_;

    my @checks;

    foreach my $stage (sort keys %{$hosts} ) {
        foreach my $unit (sort keys %{$hosts->{$stage}}) {
            foreach my $dc (sort @{$hosts->{$stage}->{$unit}->{dc}}) {
                my $host = "$unit\@cluster=$dc";
                my $safe_host = get_safe_host($host);
                push @checks,
                    {
                        active     => 'icmpping',
                        aggregator => 'logic_or',
                        children   => [
                            {
                                host => $host,
                                service => 'ping',
                                type => 'YP_PODSET',
                            }
                        ],
                        host => $safe_host,
                        tags => [
                            $stage,
                            $self->get_geo_tag_by_dc($dc)
                        ],
                        options => {
                            stage    => $stage,
                        },
                        refresh_time => 90,
                        service      => 'partner2_deploy.ping',
                        ttl          => '300s',
                    };
            }
        }
    }

    return @checks
}

sub get_juggler_aggregator_checks {
    my ($self, $checks) = @_;

    my @checks;

    foreach my $stage (sort keys(%ENV_TAGS)) {
        my @children = map {
            {
                host    => $_->{host},
                service => $_->{service},
            }
        } grep {$_->{options}->{stage} eq $stage} @$checks;

        unless (@children) {
            next;
        }

        my $check = {
            aggregator => 'logic_or',
            children   => \@children,
            tags => [
                (
                    $stage eq $ENV_PROD || $stage eq $ENV_PREPROD
                        ? ($self->get_notification_tags(),)
                        : ()
                ),
                $stage,
            ],
            refresh_time => 90,
            service      => 'partner2_deploy.ping_aggregator',
            host         => $self->get_host_by_stage($stage),
            ttl          => '300s',
        };

        $self->add_wiki_meta_url($check, "https://wiki.yandex-team.ru/partner/w/devops/monitoring/juggler-checks/ping/");

        push(@checks, $check);
    }

    return \@checks;
}

1;
