package Check::HTTPSExpire;

use strict;
use warnings FATAL => 'all';
use utf8;
use open qw(:std :utf8);

use base qw(Check);

use Carp;
use Net::SSL::ExpireDate;
use Moment;

use lib::abs qw(
  ../../lib
  );

use Constants qw ($ENV_PROD $ENV_TEST $ENV_DEV $META_QYP_OR_HW_HOSTS $FALSE);
use Util;

my %HOSTS = (
    $ENV_PROD => [
        qw(
          partner2.yandex.ru
          partner2.yandex.com
          )
    ],
    $ENV_TEST => [
        qw(
          partner2-test.yandex.ru
          partner2-test.yandex.com
          )
    ]
);

sub get_juggler_checks {
    my ($self) = @_;

    my @checks;
    foreach my $stage (sort keys %HOSTS) {
        foreach my $host (sort @{$HOSTS{$stage}}) {
            push(
                @checks,
                {
                    service => 'https_expire',
                    host    => $host,
                    ttl     => '25h',
                    tags    => [$stage],
                }
            );
        }
    }

    return \@checks;
}

sub get_juggler_aggregator_checks {
    my ($self, $checks) = @_;

    my %all_checks = map {$_->{'host'} => $_->{'service'}} @$checks;

    my @checks = ();

    my $prod_check = {
        service    => 'https_expire.aggregator',
        ttl        => '1h',
        tags       => [$self->get_notification_tags()],
        aggregator => 'logic_or',
        children   => [map {{host => $_, service => $all_checks{$_}}} @{$HOSTS{$ENV_PROD}}],
        tags       => [$ENV_PROD],
    };

    $self->add_wiki_meta_url($prod_check,
        "https://wiki.yandex-team.ru/partner/w/devops/monitoring/juggler-checks/http-https/");

    push(@checks, $prod_check);

    my @active_checks = ();
    foreach my $meta (sort {$a->{'host'} cmp $b->{'host'}} grep {$_->{'env'} eq $ENV_DEV} @$META_QYP_OR_HW_HOSTS) {
        push(
            @active_checks,
            {
                active        => 'https_cert',
                active_kwargs => {
                    crit_expire => 14,
                    port => 443,
                    timeout => 15,
                    validate_hostname => $FALSE,
                    warn_expire => 30,
                },
                flaps => {
                    stable_time => 600,
                    critical_time => 3000,
                    boost_time => 0,
                },
                aggregator    => 'logic_or',
                host          => $meta->{'host'},
                options       => {
                    stage    => $meta->{'env'},
                },
                refresh_time  => 60,
                service       => 'https_expire',
                ttl           => '180s',
                tags          => [
                    $meta->{'env'},
                    $self->get_geo_tag_by_dc($meta->{'dc'})
                ],
            }
        );
    }

    push(@checks, @active_checks);

    my $dev_check = {
        service    => 'https_expire.aggregator',
        host       => 'partner-dev',
        ttl        => '1h',
        tags       => [$self->get_dashboard_tag()],
        aggregator => 'logic_or',
        children   => [
            map {{host => $_->{'host'}, service => $_->{'service'}}}
                @active_checks
        ],
        tags       => [$ENV_DEV],
    };

    $self->add_wiki_meta_url($dev_check,
        "https://wiki.yandex-team.ru/partner/w/devops/monitoring/juggler-checks/http-https/");

    push(@checks, $dev_check);

    return \@checks;
}

sub check {
    my ($self, $check, $pja) = @_;

    my $https_left_days = get_https_left_days($check->{'host'});

    my $success = $https_left_days > 14;
    send_event($pja, $check, $success);
}

sub get_https_left_days {
    my ($domain) = @_;

    my $ed = Net::SSL::ExpireDate->new(https => $domain);

    if ($ed->expire_date()) {
        my $expire_date = datetime2moment($ed->expire_date());
        my $now         = Moment->now();

        my $delta_seconds = $expire_date->get_timestamp() - $now->get_timestamp();

        my $left_days = int($delta_seconds / 86_400);

        return $left_days;
    } else {
        return 0;
    }
}

sub datetime2moment {
    my ($dt) = @_;

    my $moment = Moment->new(dt => $dt->ymd() . " " . $dt->hms());

    return $moment;
}

1;
