package TankerMerge;

use strict;
use warnings FATAL => 'all';
use utf8;
use open qw(:std :utf8);

use lib::abs qw(../lib);
use base qw(Check);

use Moment;
use Carp;
use Constants qw($ENV_PROD);
use Util;

sub get_juggler_checks {
    my ($self) = @_;

    my @checks;

    for (my $i = 0; $i < 7; $i++) {
        my $check = {
            service => $self->get_check_name() . '.' . ($i + 1),
            host    => 'partner-prod',
            ttl     => '4h',
            tags    => [ $ENV_PROD ],
        };

        $self->add_wiki_meta_url($check,
            'https://wiki.yandex-team.ru/partner/w/devops/monitoring/juggler-checks/check-tanker-sync-merge/');

        push(@checks, $check);
    }

    return \@checks;
}

sub get_juggler_aggregator_checks {
    my ($self, $checks) = @_;

    my $check = {
        aggregator => 'logic_and',
        children   =>
            [ map {{ 'host' => $_->{'host'}, 'service' => $_->{'service'} }} @$checks ],
        service    => $self->get_check_name(),
        ttl        => '4h',
        tags       => [ $self->get_dashboard_tag(), $ENV_PROD ],
    };

    $self->add_wiki_meta_url($check,
        'https://wiki.yandex-team.ru/partner/w/devops/monitoring/juggler-checks/check-tanker-sync-merge/');

    return [ $check ];
}

sub check {
    my ($self, $check, $pja) = @_;

    my $arcanum_token = get_secrets()->{'arcanum-token'};
    my $req_opts = {
        headers => {
            "Authorization" => "OAuth $arcanum_token",
        }
    };

    my $sandbox_task_name = $self->get_sandbox_task_name();
    my $sandbox_tasks = get_json_content(
        "https://sandbox.yandex-team.ru/api/v1.0/task?limit=7&type=$sandbox_task_name")->{'items'};

    my $check_number = substr $check->{service}, -1;
    my $i = $check_number - 1;
    my $sandbox_task = $sandbox_tasks->[$i];

    unless ($sandbox_task) {
        return;
    }

    my $pull_request_id = $sandbox_task->{'output_parameters'}{'pr_id'};

    if ($sandbox_task->{status} ne 'SUCCESS') {
        my $description = "Task status: $sandbox_task->{status} . Task Url: $sandbox_task->{url}";
        send_event($pja, $check, 0, $description);
    }
    elsif (defined($pull_request_id)) {
        my $data_with_pr = get_json_content(
            sprintf("https://a.yandex-team.ru/api/v1/pull-requests/%s?fields=status,created_at", $pull_request_id),
            undef, undef, undef, $req_opts)->{data};
        my $pr_status = $data_with_pr->{'status'};
        my $raw_time = $data_with_pr->{'created_at'};

        # Время приходит в формате YYYY-MM-DDThh:mm:ssssssZ
        # Поэтому отдельно берем число и время
        my $time = substr($raw_time, 0, 10) . " " . substr($raw_time, 11, 8);

        my $minutes = (Moment->now()->get_timestamp() - Moment->new(dt => $time)->get_timestamp()) / 60;

        my $success = $pr_status eq 'merged' || $minutes < 60;

        send_event($pja, $check, $success, $sandbox_task->{'output_parameters'}{'pr_url'});
    }
    else {
        send_event($pja, $check, 1, 'No pr_id');
    }
}

1;
