package Util;

use strict;
use warnings FATAL => 'all';
use utf8;
use open qw(:std :utf8);

use Net::INET6Glue::INET_is_INET6;
use LWP::UserAgent ();
use IO::Socket::SSL qw(SSL_VERIFY_NONE);

use Carp;
use JSON::PP;
use Partner2::Juggler::API;
use File::Slurp;
use URI::Escape qw(uri_escape);
use Constants qw($DEFAULT_HOST $PARTNER2_ENDPOINTS);
use Data::Dumper;

our @ISA = qw(Exporter);
our @EXPORT_OK = qw(
    get_content
    get_graphite_paths
    get_json_content
    get_partner2_juggler_api
    get_response
    get_safe_host
    get_secrets
    get_test_data
    get_value_from_graphite
    get_safe_host
    get_safe_endpoint
    in_array
    ldump
    send_event
    service_discovery_get_hostnames
);
our @EXPORT = @EXPORT_OK;

my $NAMESPACE = 'partner';
my $RETRY_COUNT = 3;

sub get_partner2_juggler_api {
    return Partner2::Juggler::API->new(
        juggler_token => get_secrets()->{'juggler_token'},
        namespace     => $NAMESPACE,
        host          => $DEFAULT_HOST,
    );
}

sub get_json_content {
    my $response_text = get_content(@_);
    my $json_data = eval {decode_json($response_text)};
    return $json_data // {};
}

sub get_content {
    my ($url, $timeout, $retries, $sleep, $req_opts, $tvm_opts) = @_;

    my $is_utf_encode = delete $req_opts->{utf_encode};

    my $response = get_response($url, $timeout, $retries, $sleep, $req_opts, $tvm_opts);

    if ($response->is_success) {
        my $response_text = $response->decoded_content;
        utf8::encode($response_text) if $is_utf_encode;
        return $response_text;
    }
    else {
        croak $response->status_line;
    }
}

sub get_response {
    my ($url, $timeout, $retries, $sleep, $req_opts, $tvm_opts) = @_;

    my $headers = $req_opts->{headers} // [];
    $headers = [ %$headers ] if ref($headers) eq 'HASH';

    if ($tvm_opts) {
        my $tvm_ticket = _get_service_ticket(@$tvm_opts{qw(tvm_dst_alias tvm_src)});
        push @$headers, 'X-Ya-Service-Ticket', $tvm_ticket;
    }

    my $ssl_opts = delete $req_opts->{ssl_opts};
    $ssl_opts //= {
        verify_hostname => 0,
        SSL_verify_mode => SSL_VERIFY_NONE,
    };

    my $ua = LWP::UserAgent->new(
        ssl_opts => $ssl_opts,
        timeout  => $timeout // 10,
    );

    $retries = $RETRY_COUNT unless $retries;

    my $request;
    my $response;
    my $success = 0;
    while (!$success && (0 < $retries)) {

        my $request = HTTP::Request->new($req_opts->{method} // 'GET', $url, $headers, $req_opts->{content} // undef,);
        $response = $ua->request($request);

        ldump(
            {
                request  => $request->as_string(),
                response => eval {substr($response->decoded_content, 0, 1000)} || '-',
            }
        ) if ($ENV{DEBUG} // '') gt "1";

        if ($response->is_success) {
            last;
        }
        else {
            sleep($sleep) if $sleep;
        }
        $retries--;
    }

    return $response;
}

sub get_graphite_paths {
    my ($safe_host, $property, $sub_exclude) = @_;

    my $url = sprintf("http://ppcgraphite.yandex.ru/metrics/find/?query=one_min.%s.*", $safe_host);

    my $content = get_content($url);

    my $data = decode_json($content);

    my @paths;
    foreach my $el (@$data) {
        if ($el->{'id'} =~ /${property}_(.*)/ && $el->{'id'} !~ /var_lib_docker_/ && $el->{'id'} !~ /mnt(?:_.+)?$/) {
            next if ($sub_exclude && $sub_exclude->($el->{'id'}));
            push(@paths, $1);
        }
    }

    return @paths;
}

sub get_value_from_graphite {
    my ($target, %opts) = @_;

    my $url = sprintf("http://ppcgraphite.yandex.ru/render/?target=%s&from=%s&format=json",
        uri_escape($target), $opts{'from'} // '-1hour');

    if (exists($opts{'until'})) {
        $url .= "&until=$opts{'until'}";
    }

    my $content = get_content($url);

    my $value;

    eval {
        my $data = decode_json($content);
        my @values = map {$_->[0]} @{$data->[0]{'datapoints'}};

        foreach my $v (reverse @values) {
            if (defined($v)) {
                $value = $v;
                last;
            }
        }

    };

    if ($opts{'debug'}) {
        my $time = `TZ='Europe/Moscow' date`;
        chomp($time);
        printf(
            "TIME: %s\tURL: %s\tCONTENT: %s\tVALUE: %s\n",
            $time, $url,
            $content // 'No content',
            $value // 'No value'
        );
    }

    return $value;
}

sub get_secrets {
    my $secrets = decode_json scalar read_file '/etc/partner2-juggler-checks-dev.json';

    return $secrets;
}

sub get_test_data {
    my ($file, $data) = @_;

    my $path = [ caller(0) ]->[1];
    $path =~ s/\.t$//;

    my $path_to_data = "$path/$file";
    # Usage: > ./prove *.t :: --self_update 2>&1 | less
    if (defined($data) && ($ENV{'SELF_UPDATE'} || grep {'--self_update'} @ARGV)) {
        mkdir($path) unless -d $path;

        my $json_pp = JSON::PP->new->utf8->pretty->allow_nonref();
        write_file($path_to_data, $json_pp->encode($data));
    }

    return decode_json(scalar read_file($path_to_data));
}

sub get_safe_host {
    return $_[0] =~ tr/@=/_/r;
}

sub get_safe_endpoint {
    my ($endpoint) = @_;
    my $safe_endpoint = $endpoint =~ tr/\//_/r;
    $safe_endpoint =~ s/^_//;
    $safe_endpoint =~ s/_$//;
    return $safe_endpoint;
}

sub send_event {
    my ($pja, $check, $success, $description) = @_;

    my $event_data = {
        host    => $check->{'host'},
        service => $check->{'service'},
        status  => $success ? 'OK' : 'CRIT',
        defined($description) ? (description => $description) : ()
    };
    ldump($event_data) if $ENV{DEBUG};

    $pja->send(events => [ $event_data ],);
}

sub in_array {!!grep {$_[0] eq $_} @{$_[1]}}

sub ldump {
    local $Data::Dumper::Indent = 1;
    local $Data::Dumper::Varname = '';
    local $Data::Dumper::Sortkeys = 1;

    my $dump = Data::Dumper->Dump([ @_ ], [ join(':', (caller(0))[1, 2]) ]);
    $dump =~ s/\\x\{([a-f0-9]{2,})\}/chr(hex($1))/ge;
    print $dump;
}

sub _get_service_ticket {
    my ($tvm_dst_alias, $tvm_src) = @_;
    my $local_auth = read_file('/var/lib/tvmtool/local.auth');
    my $answer = get_json_content(
        sprintf('http://localhost:18080/tvm/tickets?dsts=%s&src=%s', $tvm_dst_alias // '', $tvm_src // 0),
        1, 2, 0, { headers => { 'Authorization' => $local_auth } });
    my $tvm_ticket = $answer->{$tvm_dst_alias}->{ticket} // '';
    return $tvm_ticket;
}

sub service_discovery_get_hostnames {
    my ($cluster, $id) = @_;

    my $ua = HTTP::Tiny->new();
    my @result;

    my $ruid = "resolve_${id}_${cluster}_1";
    my $result = $ua->get(
        "http://sd.yandex.net:8080/resolve_endpoints/json",
        {
            content => encode_json(
                {
                    cluster_name    => $cluster,
                    endpoint_set_id => $id,
                    client_name     => 'robot-partner',
                    ruid            => $ruid
                }
            )
        }
    );
    if ($result && $result->{success} && $result->{content}) {
        my $data = decode_json($result->{content});
        if ($data->{ruid} eq $ruid) {
            for my $ep (@{$data->{endpoint_set}{endpoints} // []}) {
                if (my $fqdn = $ep->{fqdn}) {
                    push @result, $fqdn;
                }
            }
        }
    }

    return @result;
}

sub trim($) {
    my ($str) = @_;

    $str =~ s/^\s*|\s*$//g;

    return $str;
}

1;
